"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.LogRetention = void 0;
const path = require("path");
const iam = require("@aws-cdk/aws-iam");
const s3_assets = require("@aws-cdk/aws-s3-assets");
const cdk = require("@aws-cdk/core");
const log_group_1 = require("./log-group");
/**
 * Creates a custom resource to control the retention policy of a CloudWatch Logs log group.
 *
 * The log group is created if it doesn't already exist. The policy
 * is removed when `retentionDays` is `undefined` or equal to `Infinity`.
 * Log group can be created in the region that is different from stack region by
 * specifying `logGroupRegion`
 *
 * @stability stable
 */
class LogRetention extends cdk.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        // Custom resource provider
        const provider = this.ensureSingletonLogRetentionFunction(props);
        // Need to use a CfnResource here to prevent lerna dependency cycles
        // @aws-cdk/aws-cloudformation -> @aws-cdk/aws-lambda -> @aws-cdk/aws-cloudformation
        const retryOptions = props.logRetentionRetryOptions;
        const resource = new cdk.CfnResource(this, 'Resource', {
            type: 'Custom::LogRetention',
            properties: {
                ServiceToken: provider.functionArn,
                LogGroupName: props.logGroupName,
                LogGroupRegion: props.logGroupRegion,
                SdkRetry: retryOptions ? {
                    maxRetries: retryOptions.maxRetries,
                    base: (_a = retryOptions.base) === null || _a === void 0 ? void 0 : _a.toMilliseconds(),
                } : undefined,
                RetentionInDays: props.retention === log_group_1.RetentionDays.INFINITE ? undefined : props.retention,
            },
        });
        const logGroupName = resource.getAtt('LogGroupName').toString();
        // Append ':*' at the end of the ARN to match with how CloudFormation does this for LogGroup ARNs
        // See https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-logs-loggroup.html#aws-resource-logs-loggroup-return-values
        this.logGroupArn = cdk.Stack.of(this).formatArn({
            region: props.logGroupRegion,
            service: 'logs',
            resource: 'log-group',
            resourceName: `${logGroupName}:*`,
            sep: ':',
        });
    }
    /**
     * Helper method to ensure that only one instance of LogRetentionFunction resources are in the stack mimicking the
     * behaviour of @aws-cdk/aws-lambda's SingletonFunction to prevent circular dependencies
     */
    ensureSingletonLogRetentionFunction(props) {
        const functionLogicalId = 'LogRetentionaae0aa3c5b4d4f87b02d85b201efdd8a';
        const existing = cdk.Stack.of(this).node.tryFindChild(functionLogicalId);
        if (existing) {
            return existing;
        }
        return new LogRetentionFunction(cdk.Stack.of(this), functionLogicalId, props);
    }
}
exports.LogRetention = LogRetention;
/**
 * Private provider Lambda function to support the log retention custom resource.
 */
class LogRetentionFunction extends cdk.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        // Code
        const asset = new s3_assets.Asset(this, 'Code', {
            path: path.join(__dirname, 'log-retention-provider'),
        });
        // Role
        const role = props.role || new iam.Role(this, 'ServiceRole', {
            assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
            managedPolicies: [iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSLambdaBasicExecutionRole')],
        });
        // Duplicate statements will be deduplicated by `PolicyDocument`
        role.addToPrincipalPolicy(new iam.PolicyStatement({
            actions: ['logs:PutRetentionPolicy', 'logs:DeleteRetentionPolicy'],
            // We need '*' here because we will also put a retention policy on
            // the log group of the provider function. Referencing its name
            // creates a CF circular dependency.
            resources: ['*'],
        }));
        // Lambda function
        const resource = new cdk.CfnResource(this, 'Resource', {
            type: 'AWS::Lambda::Function',
            properties: {
                Handler: 'index.handler',
                Runtime: 'nodejs12.x',
                Code: {
                    S3Bucket: asset.s3BucketName,
                    S3Key: asset.s3ObjectKey,
                },
                Role: role.roleArn,
            },
        });
        this.functionArn = resource.getAtt('Arn');
        // Function dependencies
        role.node.children.forEach((child) => {
            if (cdk.CfnResource.isCfnResource(child)) {
                resource.addDependsOn(child);
            }
            if (cdk.Construct.isConstruct(child) && child.node.defaultChild && cdk.CfnResource.isCfnResource(child.node.defaultChild)) {
                resource.addDependsOn(child.node.defaultChild);
            }
        });
    }
}
//# sourceMappingURL=data:application/json;base64,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