"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Workload = void 0;
const cdk = require("@aws-cdk/core");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const scheduler_1 = require("./scheduler");
const iam_roles_1 = require("./iam-roles");
const network_1 = require("./network");
const storage_1 = require("./storage");
class Workload extends cdk.Construct {
    constructor(scope, id, props = {}) {
        var _a, _b, _c, _d;
        super(scope, id);
        const stack = cdk.Stack.of(this);
        const s3InstallBucket = (_a = props.s3InstallBucket) !== null && _a !== void 0 ? _a : 'solutions-reference';
        const s3InstallFolder = (_b = props.s3InstallFolder) !== null && _b !== void 0 ? _b : 'scale-out-computing-on-aws/v2.5.0';
        // network construct
        const network = new network_1.Network(stack, 'SocaNetwork');
        // assets construct
        // const assets = new Assets(stack, 'S3Assets');
        // iam roles
        new iam_roles_1.IamRoles(stack, 'IamRoles', {
            s3InstallBucketName: s3InstallBucket,
            network,
        });
        // compute node security group
        const computeNodeSecurityGroup = new aws_ec2_1.SecurityGroup(this, 'ComputeNodeSecurityGroup', {
            vpc: network.vpc,
        });
        /**
         * Allow all traffic internally
         */
        computeNodeSecurityGroup.connections.allowInternally(aws_ec2_1.Port.allTraffic());
        const schedulerSecurityGroup = new aws_ec2_1.SecurityGroup(this, 'SchedulerSecurityGroup', {
            vpc: network.vpc,
        });
        /**
         * SchedulerInboundRule
         * Allow all traffic from computeNodeSecurityGroup to schedulerSecurityGroup
         */
        schedulerSecurityGroup.connections.allowFrom(computeNodeSecurityGroup, aws_ec2_1.Port.allTraffic());
        /**
         * SchedulerInboundRuleAllowClientIP
         * SchedulerInboundRuleAllowClientIPHTTPS
         * SchedulerInboundRuleAllowClientIPHTTP
         * 1. Allow SSH traffic from client IP to master host
         * 2. Allow HTTP traffic from client IP to ELB
         * 3. Allow HTTPS traffic from client IP to ELB
         */
        if (props.clientIpCidr) {
            schedulerSecurityGroup.connections.allowFrom(aws_ec2_1.Peer.ipv4(props.clientIpCidr), aws_ec2_1.Port.tcp(22));
            schedulerSecurityGroup.connections.allowFrom(aws_ec2_1.Peer.ipv4(props.clientIpCidr), aws_ec2_1.Port.tcp(80));
            schedulerSecurityGroup.connections.allowFrom(aws_ec2_1.Peer.ipv4(props.clientIpCidr), aws_ec2_1.Port.tcp(443));
        }
        /**
         * Allow traffic between Master agent and compute nodes
         */
        computeNodeSecurityGroup.connections.allowFrom(schedulerSecurityGroup, aws_ec2_1.Port.allTraffic());
        /**
         * Allow ELB healtcheck to communicate with web ui on master host
         */
        schedulerSecurityGroup.connections.allowInternally(aws_ec2_1.Port.tcp(8443));
        const storage = new storage_1.EfsStorage(stack, 'EfsStorage', {
            clusterId: network.clusterId,
            vpc: network.vpc,
            schedulerSecurityGroup,
            computeNodeSecurityGroup,
        });
        new scheduler_1.Scheduler(stack, 'Scheduler', {
            s3InstallBucket,
            s3InstallFolder,
            schedulerSecurityGroup,
            network,
            storage,
            instanceType: new aws_ec2_1.InstanceType('m5.large'),
            ldapUserName: (_c = props.ldapUserName) !== null && _c !== void 0 ? _c : 'ldapUserName',
            ldapUserPassword: (_d = props.ldapUserPassword) !== null && _d !== void 0 ? _d : 'ldapUserPassword!123',
        });
    }
}
exports.Workload = Workload;
//# sourceMappingURL=data:application/json;base64,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