"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.VpcProvider = exports.AmiProvider = exports.Cluster = void 0;
const cdk = require("@aws-cdk/core");
const ec2 = require("@aws-cdk/aws-ec2");
const s3 = require("@aws-cdk/aws-s3");
const iam = require("@aws-cdk/aws-iam");
const autoscaling = require("@aws-cdk/aws-autoscaling");
const lambda = require("@aws-cdk/aws-lambda");
const logs = require("@aws-cdk/aws-logs");
const cr = require("@aws-cdk/custom-resources");
const path = require("path");
const DEFAULT_INSTANCE_TYPE = ec2.InstanceType.of(ec2.InstanceClass.M6G, ec2.InstanceSize.MEDIUM);
/**
 * Represents the k3sCluster construct
 */
class Cluster extends cdk.Construct {
    constructor(scope, id, props = {}) {
        var _a, _b, _c, _d, _e;
        super(scope, id);
        // VPC configuration
        const vpc = (_a = props.vpc) !== null && _a !== void 0 ? _a : new ec2.Vpc(this, 'Vpc', { maxAzs: 3, natGateways: 1 });
        // S3 bucket to host K3s token + kubeconfig file 
        const k3sBucket = new s3.Bucket(this, 'k3sBucket', {
            removalPolicy: (_b = props.bucketRemovalPolicy) !== null && _b !== void 0 ? _b : cdk.RemovalPolicy.RETAIN,
        });
        // Delete S3 Object CustomResource
        if (props.bucketRemovalPolicy === cdk.RemovalPolicy.DESTROY) {
            const onEvent = new lambda.Function(this, 'onEventHandler', {
                runtime: lambda.Runtime.PYTHON_3_8,
                code: lambda.Code.fromAsset(path.join(__dirname, '../custom-resource-handler')),
                handler: 'index.on_event',
            });
            const deleteS3ObjectProvider = new cr.Provider(this, 'deleteS3ObjectProvider', {
                onEventHandler: onEvent,
                logRetention: logs.RetentionDays.ONE_DAY,
            });
            const CRdeleteS3ObjectProvider = new cdk.CustomResource(this, 'CRdeleteS3ObjectProvider', {
                serviceToken: deleteS3ObjectProvider.serviceToken,
                properties: {
                    Bucket: k3sBucket.bucketName,
                },
            });
            CRdeleteS3ObjectProvider.node.addDependency(k3sBucket);
            k3sBucket.grantDelete(onEvent);
            k3sBucket.grantReadWrite(onEvent);
        }
        // control plane node Security Group      
        const k3scontrolplanesg = new ec2.SecurityGroup(this, 'k3s-controlplane-SG', { vpc });
        k3scontrolplanesg.addIngressRule(ec2.Peer.anyIpv4(), ec2.Port.tcp(22), 'SSH');
        k3scontrolplanesg.addIngressRule(ec2.Peer.anyIpv4(), ec2.Port.tcp(6443), 'K3s port');
        // worker nodes Security Group      
        const k3sworkersg = new ec2.SecurityGroup(this, 'k3s-worker-SG', { vpc });
        // for this prototype the workers are being placed in a public subnet 
        // ideally they should land on a private subnet 
        /// also ingress traffic - ssh (bastion style) or 6443 - should come from the control plane node only 
        k3sworkersg.addIngressRule(ec2.Peer.anyIpv4(), ec2.Port.tcp(22), 'SSH');
        k3sworkersg.addIngressRule(ec2.Peer.anyIpv4(), ec2.Port.tcp(6443), 'K3s port');
        // check if the user requires a particular instance type for workers and control plane
        // if not, the default instance type is used 
        this.controlPlaneInstanceType = (_c = props.controlPlaneInstanceType) !== null && _c !== void 0 ? _c : DEFAULT_INSTANCE_TYPE;
        this.workerInstanceType = (_d = props.workerInstanceType) !== null && _d !== void 0 ? _d : DEFAULT_INSTANCE_TYPE;
        // create control plane node
        const k3scontrolplane = new ec2.Instance(this, 'k3s-controlplane', {
            instanceType: this.controlPlaneInstanceType,
            machineImage: new AmiProvider().amiId,
            vpc,
            vpcSubnets: {
                subnets: vpc.publicSubnets,
            },
            instanceName: 'k3s-controlplane',
            securityGroup: k3scontrolplanesg,
        });
        k3scontrolplane.addUserData(`
       #!/bin/bash
       curl -L -o k3s https://github.com/rancher/k3s/releases/download/v1.16.9%2Bk3s1/k3s-arm64
       chmod +x k3s
       ./k3s server &
       sleep 30
       ENDPOINT=$(curl http://169.254.169.254/latest/meta-data/public-hostname) 
       cp /etc/rancher/k3s/k3s.yaml /etc/rancher/k3s/kubeconfig.yaml
       sed -i s/127.0.0.1/$ENDPOINT/ /etc/rancher/k3s/kubeconfig.yaml
       aws s3 cp /var/lib/rancher/k3s/server/node-token s3://${k3sBucket.bucketName}/node-token
       aws s3 cp /etc/rancher/k3s/kubeconfig.yaml s3://${k3sBucket.bucketName}/kubeconfig.yaml
     `);
        this.endpointUri = k3scontrolplane.instancePublicIp;
        // create launch template for worker ASG
        // prepare the userData
        const userData = ec2.UserData.forLinux();
        userData.addCommands(`
          #!/bin/bash
          LOGFILE='/var/log/k3s.log'
          curl -L -o k3s https://github.com/rancher/k3s/releases/download/v1.16.13%2Bk3s1/k3s-arm64
          chmod +x k3s
          echo the bucket name is ${k3sBucket.bucketName} 
          aws s3 cp s3://${k3sBucket.bucketName}/node-token /node-token 
          (./k3s agent --server https://${k3scontrolplane.instancePrivateIp}:6443 \
          --token $(cat /node-token) 2>&1 | tee -a $LOGFILE || echo "failed" > $LOGFILE &)
    `);
        const lt = new ec2.CfnLaunchTemplate(this, 'WorkerLaunchTemplate', {
            launchTemplateData: {
                imageId: new AmiProvider().amiId.getImage(this).imageId,
                instanceType: this.workerInstanceType.toString(),
                instanceMarketOptions: {
                    marketType: props.spotWorkerNodes ? 'spot' : undefined,
                    spotOptions: props.spotWorkerNodes ? {
                        spotInstanceType: 'one-time',
                    } : undefined,
                },
                userData: cdk.Fn.base64(userData.render()),
            },
        });
        // create worker ASG
        const workerAsg = new autoscaling.AutoScalingGroup(this, 'WorkerAsg', {
            instanceType: this.workerInstanceType,
            machineImage: new AmiProvider().amiId,
            vpc,
            vpcSubnets: {
                subnetType: ec2.SubnetType.PUBLIC,
            },
            minCapacity: (_e = props.workerMinCapacity) !== null && _e !== void 0 ? _e : 3,
        });
        const cfnAsg = workerAsg.node.tryFindChild('ASG');
        cfnAsg.addPropertyDeletionOverride('LaunchConfigurationName');
        cfnAsg.addPropertyOverride('LaunchTemplate', {
            LaunchTemplateId: lt.ref,
            Version: lt.attrLatestVersionNumber,
        });
        workerAsg.addSecurityGroup(k3sworkersg);
        // enable the SSM session manager
        workerAsg.role.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonSSMManagedInstanceCore'));
        // grant the S3 write permission to the control plane node and read permissions to the worker nodes
        k3sBucket.grantWrite(k3scontrolplane.role);
        k3sBucket.grantRead(workerAsg.role);
        // endpoint info
        new cdk.CfnOutput(this, 'Endpoint', { value: `https://${k3scontrolplane.instancePublicIp}:6443` });
        // kubeconfig.yaml path
        new cdk.CfnOutput(this, 'Kubernetes configuration file', { value: `s3://${k3sBucket.bucketName}/kubeconfig.yaml` });
        workerAsg.node.addDependency(k3scontrolplane);
    }
}
exports.Cluster = Cluster;
/**
 * The AMI provider to get the latest Amazon Linux 2 AMI for ARM64
 */
class AmiProvider {
    get amiId() {
        return ec2.MachineImage.latestAmazonLinux({
            cpuType: ec2.AmazonLinuxCpuType.ARM_64,
            generation: ec2.AmazonLinuxGeneration.AMAZON_LINUX_2,
        });
    }
}
exports.AmiProvider = AmiProvider;
/**
 * The VPC provider to create or import the VPC
 */
class VpcProvider {
    static getOrCreate(scope) {
        const vpc = scope.node.tryGetContext('use_default_vpc') === '1' ?
            ec2.Vpc.fromLookup(scope, 'Vpc', { isDefault: true }) :
            scope.node.tryGetContext('use_vpc_id') ?
                ec2.Vpc.fromLookup(scope, 'Vpc', { vpcId: scope.node.tryGetContext('use_vpc_id') }) :
                new ec2.Vpc(scope, 'Vpc', { maxAzs: 3, natGateways: 1 });
        return vpc;
    }
}
exports.VpcProvider = VpcProvider;
//# sourceMappingURL=data:application/json;base64,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