"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.RefactorSpaces = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const shared_1 = require("../shared");
/**
 * Statement provider for service [refactor-spaces](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsmigrationhubrefactorspaces.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class RefactorSpaces extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [refactor-spaces](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsmigrationhubrefactorspaces.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'refactor-spaces';
        this.accessLevelList = {
            Write: [
                'CreateApplication',
                'CreateEnvironment',
                'CreateRoute',
                'CreateService',
                'DeleteApplication',
                'DeleteEnvironment',
                'DeleteResourcePolicy',
                'DeleteRoute',
                'DeleteService',
                'PutResourcePolicy',
                'UpdateRoute'
            ],
            Read: [
                'GetApplication',
                'GetEnvironment',
                'GetResourcePolicy',
                'GetRoute',
                'GetService',
                'ListApplications',
                'ListEnvironmentVpcs',
                'ListEnvironments',
                'ListRoutes',
                'ListServices',
                'ListTagsForResource'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
    /**
     * Grants permission to create an application within an environment
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApplicationCreatedByAccount()
     * - .ifCreatedByAccountIds()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/migrationhub-refactor-spaces/latest/APIReference/API_CreateApplication.html
     */
    toCreateApplication() {
        return this.to('CreateApplication');
    }
    /**
     * Grants permission to create an environment
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/migrationhub-refactor-spaces/latest/APIReference/API_CreateEnvironment.html
     */
    toCreateEnvironment() {
        return this.to('CreateEnvironment');
    }
    /**
     * Grants permission to create a route within an application
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApplicationCreatedByAccount()
     * - .ifServiceCreatedByAccount()
     * - .ifRouteCreatedByAccount()
     * - .ifCreatedByAccountIds()
     * - .ifSourcePath()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/migrationhub-refactor-spaces/latest/APIReference/API_CreateRoute.html
     */
    toCreateRoute() {
        return this.to('CreateRoute');
    }
    /**
     * Grants permission to create a service within an application
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApplicationCreatedByAccount()
     * - .ifServiceCreatedByAccount()
     * - .ifCreatedByAccountIds()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/migrationhub-refactor-spaces/latest/APIReference/API_CreateService.html
     */
    toCreateService() {
        return this.to('CreateService');
    }
    /**
     * Grants permission to delete an application from an environment
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApplicationCreatedByAccount()
     * - .ifCreatedByAccountIds()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/migrationhub-refactor-spaces/latest/APIReference/API_DeleteApplication.html
     */
    toDeleteApplication() {
        return this.to('DeleteApplication');
    }
    /**
     * Grants permission to delete an environment
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/migrationhub-refactor-spaces/latest/APIReference/API_DeleteEnvironment.html
     */
    toDeleteEnvironment() {
        return this.to('DeleteEnvironment');
    }
    /**
     * Grants permission to delete a resource policy
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/migrationhub-refactor-spaces/latest/APIReference/API_DeleteResourcePolicy.html
     */
    toDeleteResourcePolicy() {
        return this.to('DeleteResourcePolicy');
    }
    /**
     * Grants permission to delete a route from an application
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApplicationCreatedByAccount()
     * - .ifServiceCreatedByAccount()
     * - .ifRouteCreatedByAccount()
     * - .ifCreatedByAccountIds()
     * - .ifSourcePath()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/migrationhub-refactor-spaces/latest/APIReference/API_DeleteRoute.html
     */
    toDeleteRoute() {
        return this.to('DeleteRoute');
    }
    /**
     * Grants permission to delete a service from an application
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApplicationCreatedByAccount()
     * - .ifServiceCreatedByAccount()
     * - .ifCreatedByAccountIds()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/migrationhub-refactor-spaces/latest/APIReference/API_DeleteService.html
     */
    toDeleteService() {
        return this.to('DeleteService');
    }
    /**
     * Grants permission to get more information about an application
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApplicationCreatedByAccount()
     * - .ifCreatedByAccountIds()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/migrationhub-refactor-spaces/latest/APIReference/API_GetApplication.html
     */
    toGetApplication() {
        return this.to('GetApplication');
    }
    /**
     * Grants permission to get more information for an environment
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/migrationhub-refactor-spaces/latest/APIReference/API_GetEnvironment.html
     */
    toGetEnvironment() {
        return this.to('GetEnvironment');
    }
    /**
     * Grants permission to get the details about a resource policy
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/migrationhub-refactor-spaces/latest/APIReference/API_GetResourcePolicy.html
     */
    toGetResourcePolicy() {
        return this.to('GetResourcePolicy');
    }
    /**
     * Grants permission to get more information about a route
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApplicationCreatedByAccount()
     * - .ifServiceCreatedByAccount()
     * - .ifRouteCreatedByAccount()
     * - .ifCreatedByAccountIds()
     * - .ifSourcePath()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/migrationhub-refactor-spaces/latest/APIReference/API_GetRoute.html
     */
    toGetRoute() {
        return this.to('GetRoute');
    }
    /**
     * Grants permission to get more information about a service
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApplicationCreatedByAccount()
     * - .ifServiceCreatedByAccount()
     * - .ifCreatedByAccountIds()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/migrationhub-refactor-spaces/latest/APIReference/API_GetService.html
     */
    toGetService() {
        return this.to('GetService');
    }
    /**
     * Grants permission to list all the applications in an environment
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/migrationhub-refactor-spaces/latest/APIReference/API_ListApplications.html
     */
    toListApplications() {
        return this.to('ListApplications');
    }
    /**
     * Grants permission to list all the VPCs for the environment
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/migrationhub-refactor-spaces/latest/APIReference/API_ListEnvironmentVpcs.html
     */
    toListEnvironmentVpcs() {
        return this.to('ListEnvironmentVpcs');
    }
    /**
     * Grants permission to list all environments
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/migrationhub-refactor-spaces/latest/APIReference/API_ListEnvironments.html
     */
    toListEnvironments() {
        return this.to('ListEnvironments');
    }
    /**
     * Grants permission to list all the routes in an application
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/migrationhub-refactor-spaces/latest/APIReference/API_ListRoutes.html
     */
    toListRoutes() {
        return this.to('ListRoutes');
    }
    /**
     * Grants permission to list all the services in an environment
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/migrationhub-refactor-spaces/latest/APIReference/API_ListServices.html
     */
    toListServices() {
        return this.to('ListServices');
    }
    /**
     * Grants permission to list all the tags for a given resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/migrationhub-refactor-spaces/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to add a resource policy
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/migrationhub-refactor-spaces/latest/APIReference/API_PutResourcePolicy.html
     */
    toPutResourcePolicy() {
        return this.to('PutResourcePolicy');
    }
    /**
     * Grants permission to tag a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifApplicationCreatedByAccount()
     * - .ifServiceCreatedByAccount()
     * - .ifRouteCreatedByAccount()
     * - .ifCreatedByAccountIds()
     * - .ifSourcePath()
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/migrationhub-refactor-spaces/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to remove a tag from a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifApplicationCreatedByAccount()
     * - .ifServiceCreatedByAccount()
     * - .ifRouteCreatedByAccount()
     * - .ifCreatedByAccountIds()
     * - .ifSourcePath()
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/migrationhub-refactor-spaces/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update a route from an application
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApplicationCreatedByAccount()
     * - .ifServiceCreatedByAccount()
     * - .ifRouteCreatedByAccount()
     * - .ifCreatedByAccountIds()
     * - .ifSourcePath()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/migrationhub-refactor-spaces/latest/APIReference/API_UpdateRoute.html
     */
    toUpdateRoute() {
        return this.to('UpdateRoute');
    }
    /**
     * Adds a resource of type environment to the statement
     *
     * https://docs.aws.amazon.com/migrationhub-refactor-spaces/latest/userguide/security_iam_service-with-iam.html#security_iam_service-with-iam-id-based-policies-resources
     *
     * @param environmentId - Identifier for the environmentId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onEnvironment(environmentId, account, region, partition) {
        return this.on(`arn:${partition || RefactorSpaces.defaultPartition}:refactor-spaces:${region || '*'}:${account || '*'}:environment/${environmentId}`);
    }
    /**
     * Adds a resource of type application to the statement
     *
     * https://docs.aws.amazon.com/migrationhub-refactor-spaces/latest/userguide/security_iam_service-with-iam.html#security_iam_service-with-iam-id-based-policies-resources
     *
     * @param environmentId - Identifier for the environmentId.
     * @param applicationId - Identifier for the applicationId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifApplicationCreatedByAccount()
     * - .ifCreatedByAccountIds()
     */
    onApplication(environmentId, applicationId, account, region, partition) {
        return this.on(`arn:${partition || RefactorSpaces.defaultPartition}:refactor-spaces:${region || '*'}:${account || '*'}:environment/${environmentId}/application/${applicationId}`);
    }
    /**
     * Adds a resource of type service to the statement
     *
     * https://docs.aws.amazon.com/migrationhub-refactor-spaces/latest/userguide/security_iam_service-with-iam.html#security_iam_service-with-iam-id-based-policies-resources
     *
     * @param environmentId - Identifier for the environmentId.
     * @param applicationId - Identifier for the applicationId.
     * @param serviceId - Identifier for the serviceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifApplicationCreatedByAccount()
     * - .ifCreatedByAccountIds()
     * - .ifServiceCreatedByAccount()
     */
    onService(environmentId, applicationId, serviceId, account, region, partition) {
        return this.on(`arn:${partition || RefactorSpaces.defaultPartition}:refactor-spaces:${region || '*'}:${account || '*'}:environment/${environmentId}/application/${applicationId}/service/${serviceId}`);
    }
    /**
     * Adds a resource of type route to the statement
     *
     * https://docs.aws.amazon.com/migrationhub-refactor-spaces/latest/userguide/security_iam_service-with-iam.html#security_iam_service-with-iam-id-based-policies-resources
     *
     * @param environmentId - Identifier for the environmentId.
     * @param applicationId - Identifier for the applicationId.
     * @param routeId - Identifier for the routeId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifApplicationCreatedByAccount()
     * - .ifCreatedByAccountIds()
     * - .ifRouteCreatedByAccount()
     * - .ifServiceCreatedByAccount()
     * - .ifSourcePath()
     */
    onRoute(environmentId, applicationId, routeId, account, region, partition) {
        return this.on(`arn:${partition || RefactorSpaces.defaultPartition}:refactor-spaces:${region || '*'}:${account || '*'}:environment/${environmentId}/application/${applicationId}/route/${routeId}`);
    }
    /**
     * Filters access by restricting the action to only those accounts that created the application within an environment
     *
     * https://docs.aws.amazon.com/migrationhub-refactor-spaces/latest/userguide/security_iam_service-with-iam.html#security_iam_service-with-iam-id-based-policies-conditionkeys
     *
     * Applies to actions:
     * - .toCreateApplication()
     * - .toCreateRoute()
     * - .toCreateService()
     * - .toDeleteApplication()
     * - .toDeleteRoute()
     * - .toDeleteService()
     * - .toGetApplication()
     * - .toGetRoute()
     * - .toGetService()
     * - .toTagResource()
     * - .toUntagResource()
     * - .toUpdateRoute()
     *
     * Applies to resource types:
     * - application
     * - service
     * - route
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifApplicationCreatedByAccount(value, operator) {
        return this.if(`ApplicationCreatedByAccount`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the accounts that created the resource
     *
     * https://docs.aws.amazon.com/migrationhub-refactor-spaces/latest/userguide/security_iam_service-with-iam.html#security_iam_service-with-iam-id-based-policies-conditionkeys
     *
     * Applies to actions:
     * - .toCreateApplication()
     * - .toCreateRoute()
     * - .toCreateService()
     * - .toDeleteApplication()
     * - .toDeleteRoute()
     * - .toDeleteService()
     * - .toGetApplication()
     * - .toGetRoute()
     * - .toGetService()
     * - .toTagResource()
     * - .toUntagResource()
     * - .toUpdateRoute()
     *
     * Applies to resource types:
     * - application
     * - service
     * - route
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifCreatedByAccountIds(value, operator) {
        return this.if(`CreatedByAccountIds`, value, operator || 'StringLike');
    }
    /**
     * Filters access by restricting the action to only those accounts that created the route within an application
     *
     * https://docs.aws.amazon.com/migrationhub-refactor-spaces/latest/userguide/security_iam_service-with-iam.html#security_iam_service-with-iam-id-based-policies-conditionkeys
     *
     * Applies to actions:
     * - .toCreateRoute()
     * - .toDeleteRoute()
     * - .toGetRoute()
     * - .toTagResource()
     * - .toUntagResource()
     * - .toUpdateRoute()
     *
     * Applies to resource types:
     * - route
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifRouteCreatedByAccount(value, operator) {
        return this.if(`RouteCreatedByAccount`, value, operator || 'StringLike');
    }
    /**
     * Filters access by restricting the action to only those accounts that created the service within an application
     *
     * https://docs.aws.amazon.com/migrationhub-refactor-spaces/latest/userguide/security_iam_service-with-iam.html#security_iam_service-with-iam-id-based-policies-conditionkeys
     *
     * Applies to actions:
     * - .toCreateRoute()
     * - .toCreateService()
     * - .toDeleteRoute()
     * - .toDeleteService()
     * - .toGetRoute()
     * - .toGetService()
     * - .toTagResource()
     * - .toUntagResource()
     * - .toUpdateRoute()
     *
     * Applies to resource types:
     * - service
     * - route
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifServiceCreatedByAccount(value, operator) {
        return this.if(`ServiceCreatedByAccount`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the path of the route
     *
     * https://docs.aws.amazon.com/migrationhub-refactor-spaces/latest/userguide/security_iam_service-with-iam.html#security_iam_service-with-iam-id-based-policies-conditionkeys
     *
     * Applies to actions:
     * - .toCreateRoute()
     * - .toDeleteRoute()
     * - .toGetRoute()
     * - .toTagResource()
     * - .toUntagResource()
     * - .toUpdateRoute()
     *
     * Applies to resource types:
     * - route
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifSourcePath(value, operator) {
        return this.if(`SourcePath`, value, operator || 'StringLike');
    }
}
exports.RefactorSpaces = RefactorSpaces;
_a = JSII_RTTI_SYMBOL_1;
RefactorSpaces[_a] = { fqn: "iam-floyd.RefactorSpaces", version: "0.458.0" };
//# sourceMappingURL=data:application/json;base64,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