"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.JobQueue = void 0;
const core_1 = require("@aws-cdk/core");
const batch_generated_1 = require("./batch.generated");
/**
 * Batch Job Queue.
 *
 * Defines a batch job queue to define how submitted batch jobs
 * should be ran based on specified batch compute environments.
 */
class JobQueue extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.jobQueueName,
        });
        if (props.computeEnvironments.length === 0) {
            throw new Error('computeEnvironments must be non-empty');
        }
        const jobQueue = new batch_generated_1.CfnJobQueue(this, 'Resource', {
            computeEnvironmentOrder: props.computeEnvironments.map(cp => ({
                computeEnvironment: cp.computeEnvironment.computeEnvironmentArn,
                order: cp.order,
            })),
            jobQueueName: this.physicalName,
            priority: props.priority || 1,
            state: props.enabled === undefined ? 'ENABLED' : (props.enabled ? 'ENABLED' : 'DISABLED'),
        });
        this.jobQueueArn = this.getResourceArnAttribute(jobQueue.ref, {
            service: 'batch',
            resource: 'job-queue',
            resourceName: this.physicalName,
        });
        this.jobQueueName = this.getResourceNameAttribute(jobQueue.ref);
    }
    /**
     * Fetches an existing batch job queue by its amazon resource name.
     *
     * @param scope
     * @param id
     * @param jobQueueArn
     */
    static fromJobQueueArn(scope, id, jobQueueArn) {
        const stack = core_1.Stack.of(scope);
        const jobQueueName = stack.parseArn(jobQueueArn).resourceName;
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.jobQueueArn = jobQueueArn;
                this.jobQueueName = jobQueueName;
            }
        }
        return new Import(scope, id);
    }
}
exports.JobQueue = JobQueue;
//# sourceMappingURL=data:application/json;base64,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