import logging
from abc import ABC, abstractmethod
from typing import TYPE_CHECKING

from autosweep import sweep
from autosweep.data_types.metadata import DUTInfo

if TYPE_CHECKING:
    from pathlib import Path

    from autosweep.exec_helpers.reporter import ResultsHold
    from autosweep.instruments.instrument_manager import InstrumentManager


class AbsTest(ABC):
    """
    This is abstract base class for all tests that are used within the TestExec and called by the test recipes.
    """

    # This needs to be true, otherwise it will not be registered as a test by the registrar.
    _ta_test = True

    def __init__(self, dut_info: "DUTInfo", results: "ResultsHold", save_path: "Path"):
        """

        :param dut_info: The information about the device-under-test
        :type dut_info: autosweep.utils.data_types.metadata.DUTInfo
        :param results: Any test results generated by the test are held in this object
        :type results: autosweep.utils.exec_helpers.reporter.ResultsHold
        :param save_path: The path to the folder in which any generated files, like raw data will be saved.
        :type save_path: pathlib.Path
        """
        self.logger = logging.getLogger(self.__class__.__name__)

        self.dut_info = dut_info

        self.instrument_manager = None

        self.save_path = save_path
        self.raw_data_fname = "raw_data.json"

        self._raw_data = False
        self.metadata = None
        self.sweeps = None

        self.results = results

    @abstractmethod
    def run_acquire(self, instr_mgr: "InstrumentManager") -> None:
        """
        This method is called by the TestExec for data acquisition. It must be overwritten.

        :param instr_mgr: An instrument manager with the appropriate instruments
        :type instr_mgr: autosweep.instruments.instrument_manager.InstrumentManager
        :return: None
        """
        raise NotImplementedError

    def save_data(
        self, sweeps: dict | None = None, metadata: dict | None = None
    ) -> None:
        """
        A helper method that can be called from within 'run_acquire()' to save data in a standardized way.

        :param sweeps: A collection of Sweep instances
        :type sweeps: dict, optional
        :param metadata: Any global metadata needing saving
        :type metadata: dict, optional
        :return: None
        """
        for key, s in sweeps.items():
            if not isinstance(key, str):
                msg = f"The 'sweep' key, '{key}' should be a str"
                raise TypeError(msg)

            if not isinstance(s, sweep.Sweep):
                msg = f"The 'sweep' value for key '{key}' should be a autosweep.sweep.Sweep class instance"
                raise TypeError(msg)

        self.sweeps = sweeps
        self.metadata = metadata if metadata else {}

        sweep.io.to_json(
            sweeps=self.sweeps,
            metadata=self.metadata,
            dut_info=self.dut_info,
            path=self.save_path / self.raw_data_fname,
        )

        self._raw_data = True

    @abstractmethod
    def run_analysis(self, report_headings: list) -> None:
        """
        This method is called by the TestExec for data analysis. It must be overwritten.

        :param report_headings: A collection of headings used in the HTML report
        :type report_headings: list
        :return: None
        """
        raise NotImplementedError

    def load_data(self) -> None:
        """
        Used in conjunction with the 'save_data()' method. Call this within 'run_analysis()' to load saved raw data.
        """
        # no point reading in data if it's already in memory
        if not self._raw_data:
            self.sweeps, self.metadata, _ = sweep.io.read_json(
                path=self.save_path / self.raw_data_fname
            )
