import os
from typing import Callable, Dict, Tuple

from streamlit import session_state as _state

# import streamlit.components.v1 as components
from streamlit.components.v1 import components as _components

from pollination_streamlit.interactors import ApiClient

from .components_callbacks import register_callback

# get_host
parent_dir = os.path.dirname(os.path.abspath(__file__))
build_dir = os.path.join(parent_dir, "./get_host/frontend/build")
_get_host = _components.declare_component("get_host", path=build_dir)


def get_host(key='foo'):
    """Create a new instance of "get_host".

    Parameters
    ----------
    key: str or None
        An optional key that uniquely identifies this component. If this is
        None, and the component's arguments are changed, the component will
        be re-mounted in the Streamlit frontend and lose its current state.

    Returns
    -------
    host: 'web' | 'rhino' | 'revit' | 'sketchup'
    """

    get_host = _get_host(key=key)

    return get_host


# get_geometry
parent_dir = os.path.dirname(os.path.abspath(__file__))
build_dir = os.path.join(parent_dir, "./get_geometry/frontend/build")
_get_geometry = _components.declare_component("get_geometry", path=build_dir)


def get_geometry(
    key: str = 'foo', *, 
    label: str = None, 
    options: dict = None,
    on_change:  Callable = None,
    args: Tuple = None,
    kwargs: Dict = None,
):
    """Create a new instance of "get_geometry".

    Parameters
    ----------
    key: str or None
        An optional key that uniquely identifies this component. If this is
        None, and the component's arguments are changed, the component will
        be re-mounted in the Streamlit frontend and lose its current state.
    label: str or None
        A string that will be displayed on the button. Defaults to "Get Geometry"
    options: dictionary or None
        A dictionary to show / hide button options. Defaults each option to visible & unselected.

        {
            "subscribe" : {
                "show": True or False,
                "selected": True or False
            },
            "selection" : {
                "show": True or False,
                "selected": True or False
            }
        }

    on_change: Callable or None
        An optional on_change function that will be called when hbjson input
        is changed.
    args: Tuple or None
        An optional tuple of positional arguments that will be passed to the
        on_change function.
    kwargs: dictionary or None
        An optional dictionary of kwargs that will be passed to the on_change
        function.

    Returns
    -------
    dict
        A dictionary with the following structure

        {
            'geometry': List[dict]
        }

        where
            'geometry': List of ladybug geometries as dictionary
    """
    
    if on_change is not None:
        args = args or []
        kwargs = kwargs or {}
        register_callback(key, on_change, *args, **kwargs)

    get_geometry = _get_geometry(key=key, button_label=label, options_config=options)

    return get_geometry


# get_hbjson
parent_dir = os.path.dirname(os.path.abspath(__file__))
build_dir = os.path.join(parent_dir, "./get_hbjson/frontend/build")
_get_hbjson = _components.declare_component("get_hbjson", path=build_dir)


def get_hbjson(
    key: str = 'foo', *,
    label: str = None,
    options: Dict = None,
    on_change:  Callable = None,
    args: Tuple = None,
    kwargs: Dict = None,
):
    """Create a new instance of "get_hbjson".

    Parameters
    ----------
    key: str or None
        An optional key that uniquely identifies this component. If this is
        None, and the component's arguments are changed, the component will
        be re-mounted in the Streamlit frontend and lose its current state.
    label: str or None
        A string that will be displayed on the button. Defaults to "Get Geometry"
    options: dictionary or None
        A dictionary to show / hide button options. Defaults each option to visible & unselected.

        {
            "subscribe" : {
                "show": True or False,
                "selected": True or False
            },
            "selection" : {
                "show": True or False,
                "selected": True or False
            }
        }

    on_change: Callable or None
        An optional on_change function that will be called when hbjson input
        is changed.
    args: Tuple or None
        An optional tuple of positional arguments that will be passed to the
        on_change function.
    kwargs: dictionary or None
        An optional dictionary of kwargs that will be passed to the on_change
        function.

    Returns
    -------
    dict
        A dictionary with the following structure

        {
            'hbjson': dict
        }

        where
            'hbjson': hbjson model as dictionary
    """

    if on_change is not None:
        args = args or []
        kwargs = kwargs or {}
        register_callback(key, on_change, *args, **kwargs)

    get_hbjson = _get_hbjson(key=key, button_label=label, options_config=options)

    return get_hbjson


# send_geometry
parent_dir = os.path.dirname(os.path.abspath(__file__))
build_dir = os.path.join(parent_dir, "./send_geometry/frontend/build")
_send_geometry = _components.declare_component("send_geometry", path=build_dir)


def send_geometry(
    key='foo', *, 
    geometry={}, 
    label: str = None, 
    option='preview', 
    options: dict = None ):

    """Create a new instance of "send_geometry".

    Parameters
    ----------
    key: str or None
        An optional key that uniquely identifies this component. If this is
        None, and the component's arguments are changed, the component will
        be re-mounted in the Streamlit frontend and lose its current state.
    geometry: dictionary
        A dictionary of ladybug geometries.
    option: 'add' | 'preview' | 'clear' | 'subscribe-preview'
        An option that modifies the action to be taken in the host CAD platform.
    label: str or None
        A string that will be displayed on the button. Defaults to "Get Geometry"
    options: dictionary or None
        A dictionary to show / hide button options. Defaults to show all.

        {
            'add', = True or False
            'delete', = True or False
            'preview', = True or False
            'clear', = True or False
            'subscribe-preview' = True or False
        }
    
    Returns
    -------
    """

    send_geometry = _send_geometry( key=key, geometry=geometry, option=option, button_label=label, options_config = options)

    return send_geometry


# send_hbjson
parent_dir = os.path.dirname(os.path.abspath(__file__))
build_dir = os.path.join(parent_dir, "./send_hbjson/frontend/build")
_send_hbjson = _components.declare_component("send_hbjson", path=build_dir)


def send_hbjson(key='foo', *, 
    hbjson={}, 
    option='preview',
    label: str = None,
    options: dict = None):

    """Create a new instance of "send_hbjson".

    Parameters
    ----------
    key: str or None
        An optional key that uniquely identifies this component. If this is
        None, and the component's arguments are changed, the component will
        be re-mounted in the Streamlit frontend and lose its current state.
    hbjson: dictionary
        An hbjson model as a dictionary.
    option: 'add' | 'preview' | 'clear' | 'subscribe-preview'
        An option that modifies the action to be taken in the host CAD platform.
    label: str or None
        A string that will be displayed on the button. Defaults to "Get Geometry"
    options: dictionary or None
        A dictionary to show / hide button options. Defaults to show all.

        {
            'add', = True or False
            'delete', = True or False
            'preview', = True or False
            'clear', = True or False
            'subscribe-preview' = True or False
        }
    Returns
    -------
    """

    send_hbjson = _send_hbjson(
        key=key,
        hbjson=hbjson,
        option=option,
        button_label=label,
        options_config=options
    )

    return send_hbjson


# send_results
parent_dir = os.path.dirname(os.path.abspath(__file__))
build_dir = os.path.join(parent_dir, "./send_results/frontend/build")
_send_results = _components.declare_component("send_results", path=build_dir)


def send_results(key='foo', *, 
    results={}, 
    option='preview', 
    options=None, 
    label=None ):
    
    """Create a new instance of "send_results".

    Parameters
    ----------
    key: str or None
        An optional key that uniquely identifies this component. If this is
        None, and the component's arguments are changed, the component will
        be re-mounted in the Streamlit frontend and lose its current state.
    results: dictionary
        If host === 'rhino' use new GraphicContainer object
        If host === 'revit' dictionary with fields: 'results', 'sensorGrids' and 'configs.
    option: 'add' | 'delete' | 'preview' | 'clear' | 'subscribe-preview'
        An option that modifies the action to be taken in the host CAD platform.
    label: str or None
        A string that will be displayed on the button. Defaults to "Get Geometry"
    options: dictionary or None
        A dictionary to show / hide button options. Defaults to show all.

        {
            'add', = True or False
            'delete', = True or False
            'preview', = True or False
            'clear', = True or False
            'subscribe-preview' = True or False
        }
        
    Returns
    -------
    """

    send_results = _send_results(
        key=key, 
        results=results, 
        option=option, 
        options_config=options, 
        button_label=label
    )

    return send_results


# auth_user
parent_dir = os.path.dirname(os.path.abspath(__file__))
build_dir = os.path.join(parent_dir, "./auth_user/frontend/build")
_auth_user = _components.declare_component("auth_user", path=build_dir)


def auth_user(key='foo', api_client: ApiClient = None):
    """Create a new instance of "auth_user".

    Parameters
    ----------
    key: str or None
        An optional key that uniquely identifies this component. If this is
        None, and the component's arguments are changed, the component will
        be re-mounted in the Streamlit frontend and lose its current state.
    api_client: ApiClient
        ApiClient is a class returned by get_api_client(), part of the 
        pollination_streamlit package.
        ```
        from pollination_streamlit.selectors import get_api_client
        ...
        api_client = get_api_client
        ```

    Returns
    -------
    dict
        A dictionary with the following structure

        {
            'id': string
            'email': string
            'name': string
            'username': string
            'description': string or None
            'picture': string or None
        }
    """

    client = api_client.__dict__

    auth_user = _auth_user(key=key,
                           access_token=client["_jwt_token"],
                           api_key=client["_api_token"],
                           base_path=client["_host"])

    return auth_user


# select_account
parent_dir = os.path.dirname(os.path.abspath(__file__))
build_dir = os.path.join(parent_dir, "./select_account/frontend/build")
_select_account = _components.declare_component(
    "select_account", path=build_dir)


def select_account(
    key='foo',
    api_client: ApiClient = None, *,
    on_change:  Callable = None,
    args: Tuple = None,
    kwargs: Dict = None,
):
    """Create a new instance of "select_account".

    Parameters
    ----------
    key: str or None
        An optional key that uniquely identifies this component. If this is
        None, and the component's arguments are changed, the component will
        be re-mounted in the Streamlit frontend and lose its current state.
    api_client: ApiClient
        ApiClient is a class returned by get_api_client(), part of the 
        pollination_streamlit package.
        ```
        from pollination_streamlit.selectors import get_api_client
        ...
        api_client = get_api_client
        ```
    on_change: Callable or None
        An optional on_change function that will be called when hbjson input
        is changed.
    args: Tuple or None
        An optional tuple of positional arguments that will be passed to the
        on_change function.
    kwargs: dictionary or None
        An optional dictionary of kwargs that will be passed to the on_change
        function.
    
    Returns
    -------
    dict
        A dictionary with the following structure

        {
            'id': string
            'email': string or None
            'contact_email': string or None
            'name': string
            'username': string or None
            'account_name': string or None
            'description': string or None
            'picture': string or None
            'picture_url': string or None
            'owner': dictionary or None
            'role': string or None
            'member_count': number or None
            'team_count': number or None
        }

        where

            'owner': dictionary with the following structure

            {
              'id': string
              'type': 'org' | 'user'
              'name': string
            }
    """
    client = api_client.__dict__

    if on_change is not None:
        args = args or []
        kwargs = kwargs or {}
        register_callback(key, on_change, *args, **kwargs)

    select_account = _select_account(key=key,
                                     access_token=client["_jwt_token"],
                                     api_key=client["_api_token"],
                                     base_path=client["_host"])

    return select_account


# # select_project
parent_dir = os.path.dirname(os.path.abspath(__file__))
build_dir = os.path.join(parent_dir, "./select_project/frontend/build")
_select_project = _components.declare_component(
    "select_project", path=build_dir)


def select_project(
    key='foo', 
    api_client: ApiClient = None, *,
    project_owner=None,
    on_change:  Callable = None,
    args: Tuple = None,
    kwargs: Dict = None,
):
    """Create a new instance of "select_project".

    Parameters
    ----------
    key: str or None
        An optional key that uniquely identifies this component. If this is
        None, and the component's arguments are changed, the component will
        be re-mounted in the Streamlit frontend and lose its current state.
    api_client: ApiClient
        ApiClient is a class returned by get_api_client(), part of the 
        pollination_streamlit package.
        ```
        from pollination_streamlit.selectors import get_api_client
        ...
        api_client = get_api_client
        ```
    project_owner: str or None
        username of project owner
    on_change: Callable or None
        An optional on_change function that will be called when hbjson input
        is changed.
    args: Tuple or None
        An optional tuple of positional arguments that will be passed to the
        on_change function.
    kwargs: dictionary or None
        An optional dictionary of kwargs that will be passed to the on_change
        function.

    Returns
    -------
    dict
        A dictionary with the following structure

            {
                'name': string,
                'description': string or None,
                'public': boolean,
                'id': string,
                'owner': dictionary,
                'permissions': dictionary,
                'slug': string,
                'usage': dictionary,            
            }

            where

                'owner' is a dictionary with the following structure

                    {
                        "id": string,
                        "account_type": string,
                        "name": string,
                        "display_name": string,
                        "description": string or None,
                        "picture_url": string or None
                    }

                'permissions' is a dictionary with the following structure

                    {
                        "admin": boolean,
                        "write": boolean,
                        "read": boolean
                    }

                'usage' is a dictionary with the following structure

                    {
                        "start": string,
                        "end": string,
                        "cpu": int,
                        "memory": int,
                        "succeeded": int,
                        "failed": int,
                        "daily_usage": tuple
                    }
    """

    if on_change is not None:
        args = args or []
        kwargs = kwargs or {}
        register_callback(key, on_change, *args, **kwargs)
    
    client = api_client.__dict__

    select_project = _select_project(key=key,
                                     project_owner=project_owner,
                                     access_token=client["_jwt_token"],
                                     api_key=client["_api_token"],
                                     base_path=client["_host"])

    return select_project


# select_recipe
parent_dir = os.path.dirname(os.path.abspath(__file__))
build_dir = os.path.join(parent_dir, "./select_recipe/frontend/build")
_select_recipe = _components.declare_component(
    "select_recipe", path=build_dir)


def select_recipe(
    key='foo', 
    api_client: ApiClient = None, *, 
    project_name=None, 
    project_owner=None,
    on_change:  Callable = None,
    args: Tuple = None,
    kwargs: Dict = None,
):
    """Create a new instance of "select_recipe".

    Parameters
    ----------
    key: str or None
        An optional key that uniquely identifies this component. If this is
        None, and the component's arguments are changed, the component will
        be re-mounted in the Streamlit frontend and lose its current state.
    api_client: ApiClient
        ApiClient is a class returned by get_api_client(), part of the 
        pollination_streamlit package.
        ```
        from pollination_streamlit.selectors import get_api_client
        ...
        api_client = get_api_client
        ```
    project_name: str or None
        project name
    project_owner: str or None
        username of the project owner
    on_change: Callable or None
        An optional on_change function that will be called when hbjson input
        is changed.
    args: Tuple or None
        An optional tuple of positional arguments that will be passed to the
        on_change function.
    kwargs: dictionary or None
        An optional dictionary of kwargs that will be passed to the on_change
        function.

    Returns
    -------
    dict
        A dictionary with the structure described here:

            https://api.pollination.cloud/docs#/Projects/get_project_recipes

    """
    if on_change is not None:
        args = args or []
        kwargs = kwargs or {}
        register_callback(key, on_change, *args, **kwargs)

    client = api_client.__dict__

    select_recipe = _select_recipe(
        key=key,
        project_name=project_name,
        project_owner=project_owner,
        access_token=client["_jwt_token"],
        api_key=client["_api_token"],
        base_path=client["_host"])

    return select_recipe


# select_study
parent_dir = os.path.dirname(os.path.abspath(__file__))
build_dir = os.path.join(parent_dir, "./select_study/frontend/build")
_select_study = _components.declare_component(
    "select_study", path=build_dir)


def select_study(
    key='foo', api_client: ApiClient = None, *,
    project_name=None,
    project_owner=None,
    on_change:  Callable = None,
    args: Tuple = None,
    kwargs: Dict = None,
):
    """Create a new instance of "select_study".

    Parameters
    ----------
    key: str or None
        An optional key that uniquely identifies this component. If this is
        None, and the component's arguments are changed, the component will
        be re-mounted in the Streamlit frontend and lose its current state.
    api_client: ApiClient
        ApiClient is a class returned by get_api_client(), part of the 
        pollination_streamlit package.
        ```
        from pollination_streamlit.selectors import get_api_client
        ...
        api_client = get_api_client
        ```
    project_name: str or None
        name of project
    project_owner: str or None
        username of project owner
    on_change: Callable or None
        An optional on_change function that will be called when hbjson input
        is changed.
    args: Tuple or None
        An optional tuple of positional arguments that will be passed to the
        on_change function.
    kwargs: dictionary or None
        An optional dictionary of kwargs that will be passed to the on_change
        function.

    Returns
    -------
    dict
        A dictionary with the structure documented here:

            https://api.pollination.cloud/docs#/Jobs/list_jobs
    """
    if on_change is not None:
        args = args or []
        kwargs = kwargs or {}
        register_callback(key, on_change, *args, **kwargs)
    
    client = api_client.__dict__

    select_study = _select_study(
        key=key,
        project_name=project_name,
        project_owner=project_owner,
        access_token=client["_jwt_token"],
        api_key=client["_api_token"],
        base_path=client["_host"])

    return select_study


# select_run
parent_dir = os.path.dirname(os.path.abspath(__file__))
build_dir = os.path.join(parent_dir, "./select_run/frontend/build")
_select_run = _components.declare_component(
    "select_run", path=build_dir)


def select_run(
    key='foo', api_client: ApiClient = None, *,
    project_name=None,
    project_owner=None,
    job_id=None,
    on_change:  Callable = None,
    args: Tuple = None,
    kwargs: Dict = None
):
    """Create a new instance of "select_run".

    Parameters
    ----------
    key: str or None
        An optional key that uniquely identifies this component. If this is
        None, and the component's arguments are changed, the component will
        be re-mounted in the Streamlit frontend and lose its current state.
    api_client: ApiClient
        ApiClient is a class returned by get_api_client(), part of the 
        pollination_streamlit package.
        ```
        from pollination_streamlit.selectors import get_api_client
        ...
        api_client = get_api_client
        ```
    project_owner: str or None
        username of project owner
    job_id: str or None
        id of study
    on_change: Callable or None
        An optional on_change function that will be called when hbjson input
        is changed.
    args: Tuple or None
        An optional tuple of positional arguments that will be passed to the
        on_change function.
    kwargs: dictionary or None
        An optional dictionary of kwargs that will be passed to the on_change
        function.
        
    Returns
    -------
    dict
        A dictionary with the sturcture described here:

            https://api.pollination.cloud/docs#/Runs/list_runs
    """

    if on_change is not None:
        args = args or []
        kwargs = kwargs or {}
        register_callback(key, on_change, *args, **kwargs)

    client = api_client.__dict__

    select_run = _select_run(
        key=key,
        project_name=project_name,
        project_owner=project_owner,
        job_id=job_id,
        access_token=client["_jwt_token"],
        api_key=client["_api_token"],
        base_path=client["_host"])

    return select_run
