/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
import { Component, IgnoreFile, JsonFile, Project } from "projen";
import { Obj } from "projen/lib/util";
import { Nx } from "../nx-types";
/** Default NX outputs to cache */
export declare const NX_DEFAULT_BUILD_OUTPUTS: string[];
/**
 * Component which manages the workspace specific NX Config for the root monorepo.
 * @experimental
 */
export declare class NxWorkspace extends Component {
    /**
     * Retrieves the singleton instance associated with project root.
     *
     * @param scope project instance.
     */
    static of(scope: Project): NxWorkspace | undefined;
    /**
     * Raw nx.json file to support overrides that aren't handled
     * directly.
     *
     * **Attention:** any overrides applied here will not be visible
     * in the properties and only included in final synthesized output,
     * and likely to override native handling.
     * @advanced
     */
    readonly nxJson: JsonFile;
    /**
     * .nxignore file
     */
    readonly nxIgnore: IgnoreFile;
    /**
     * Automatically infer NxProject targets based on project type.
     * @experimental
     */
    autoInferProjectTargets: boolean;
    /**
     * Override the default nx cacheDirectory
     * @experimental May cause `Could not find 'nx' module in this workspace.` [issue](https://github.com/nrwl/nx/issues/8929). If you experience this issue, please remove `cacheDirectory` override.
     */
    cacheDirectory?: string;
    /**
     * Indicates if non-native nx hasher will be used.
     *
     * If true, the NX_NON_NATIVE_HASHER env var will be set
     * to true for all project tasks.
     */
    nonNativeHasher: boolean;
    /**
     * List of cacheable operations.
     */
    cacheableOperations: string[];
    /**
     * Some presets use the extends property to hide some default options in a separate json file.
     * The json file specified in the extends property is located in your node_modules folder.
     * The Nx preset files are specified in the nx package.
     *
     * @default "nx/presets/npm.json"
     */
    extends: string;
    /**
     * Tells Nx what prefix to use when generating library imports.
     */
    npmScope: string;
    /**
     * Where new apps + libs should be placed
     */
    workspaceLayout?: Nx.IWorkspaceLayout;
    /**
     * Plugins for extending the project graph
     */
    plugins: string[];
    /**
     * Configuration for Nx Plugins
     */
    pluginsConfig: Obj<any>;
    /**
     * Default options for `nx affected`
     */
    affected: Nx.INxAffectedConfig;
    /**
     * Named inputs
     * @see https://nx.dev/reference/nx-json#inputs-&-namedinputs
     */
    namedInputs: Nx.INamedInputs;
    /**
     * Dependencies between different target names across all projects
     *
     * @see https://nx.dev/reference/nx-json#target-defaults
     */
    targetDefaults: Nx.ITargetDefaults;
    /**
     * Default task runner
     */
    defaultTaskRunner: string;
    /**
     * Default task runner options
     */
    defaultTaskRunnerOptions: Obj<any>;
    /**
     * Task runner options.
     */
    tasksRunnerOptions: Obj<any>;
    constructor(project: Project);
    /**
     * Setup workspace to use nx-cloud
     * @param readOnlyAccessToken
     */
    useNxCloud(readOnlyAccessToken: string): void;
    /**
     * Set `namedInput` value helper
     */
    setNamedInput(name: string, inputs: string[]): void;
    /** Set `targetDefaults` helper */
    setTargetDefault(name: string, target: Nx.IProjectTarget, merge?: boolean): void;
    /** @internal */
    protected _recursivelyApplyToProject(project: Project, fn: (project: Project) => void): void;
    /** @internal */
    protected _applyNonNativeHasher(project: Project): void;
    /** @internal */
    protected _applyCacheDirectory(project: Project): void;
    /** @inheritdoc */
    preSynthesize(): void;
    /** @inheritdoc */
    synthesize(): void;
}
