"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DependencyLink = exports.ReferenceLink = exports.ChildLink = exports.Link = exports.Edge = exports.BaseEdge = void 0;
const Dot = require("ts-graphviz");
const subgraphs_1 = require("./subgraphs");
const theme_1 = require("../theme");
/**
 * BaseEdge class is the base class for defining a {@link Dot.Edge}.
 * @internal
 */
class BaseEdge extends Dot.Edge {
    /** Indicates if edge is considered **extraneous** */
    get isExtraneous() {
        return this.isSynthetic;
    }
    /** Indicates if edge is considered **verbose** */
    get isVerbose() {
        return this.isExtraneous || this.isClosedLoop;
    }
    /** Indicates if edge is a *closed loop*, meaning its *leaf* and *head* are the same entity */
    get isClosedLoop() {
        const [from, to] = this.targets;
        if (from.id === to.id)
            return true;
        if (from.id === this.attributes.get("lhead"))
            return true;
        if (to.id === this.attributes.get("ltail"))
            return true;
        return false;
    }
    /** @internal */
    constructor(from, to // TODO: support edge chain (need to handle subgraph case) // ...rest: EdgeTarget[]
    ) {
        const _attributes = {};
        let isCompound = false;
        let isSynthetic = false;
        // [compound](https://graphviz.org/docs/attrs/compound/) support - referencing container rather than node
        let _from;
        if (from instanceof subgraphs_1.Container) {
            isCompound = true;
            _from = from.nodes[0];
            if (_from == null) {
                isSynthetic = true;
                _from = {
                    id: `synthetic_${from.id}`,
                    style: "none",
                    shape: "none",
                    label: "",
                };
            }
            // https://graphviz.org/docs/attrs/ltail/
            _attributes.ltail = from.id;
        }
        else {
            _from = from;
        }
        let _to;
        if (to instanceof subgraphs_1.Container) {
            _to = to.nodes[0];
            isCompound = true;
            if (_to == null) {
                isSynthetic = true;
                _to = {
                    id: `synthetic_${to.id}`,
                    style: "none",
                    shape: "none",
                    label: "",
                };
            }
            // https://graphviz.org/docs/attrs/lhead/
            _attributes.lhead = to.id;
        }
        else {
            _to = to;
        }
        super([_from, _to], _attributes);
        this.from = from;
        this.to = to;
        this._extraneous = false;
        this.isCompound = isCompound;
        this.isSynthetic = isSynthetic;
        // Experimental: attempting to improve layout control of subgraphs (disabled by default)
        if (theme_1.GraphTheme.instance.rendering.unconstrainedCrossClusterEdges) {
            // Prevent cross-cluster edges from mangling the cluster layout
            if (this.isClosedLoop ||
                this.isCompound ||
                this.isSynthetic ||
                this.from.graphNode.findAncestor({
                    filter: (node) => node.isCluster,
                }) !==
                    this.to.graphNode.findAncestor({ filter: (node) => node.isCluster })) {
                this.attributes.set("constraint", false);
                if (!this.isClosedLoop &&
                    this.from instanceof Dot.Node &&
                    this.to instanceof Dot.Node) {
                    const group = "group_" +
                        this.from.graphNode.getNearestAncestor(this.to.graphNode).uuid;
                    this.from.attributes.set("group", group);
                    this.to.attributes.set("group", group);
                }
            }
        }
    }
}
exports.BaseEdge = BaseEdge;
/**
 * Edge class is the base class for {@link Graph.Edge} based {@link Dot.Edge} entities
 * @internal
 */
class Edge extends BaseEdge {
    /** @internal */
    constructor(edge, from, to
    // TODO: support edge chain (need to handle subgraph case)
    // Need to have specific use case before implementing this, but Dot.Edge supports chaining
    // ...rest: EdgeTarget[]
    ) {
        super(from, to);
        this.attributes.set("id", edge.uuid);
        this.graphEdge = edge;
        this._extraneous = edge.isExtraneous;
    }
}
exports.Edge = Edge;
/**
 * Link class defines a {@link Graph.Edge} defined by a {@link Graph.Node}
 * @internal
 */
class Link extends Edge {
    /** @internal */
    constructor(edge, from, to) {
        super(edge, from, to);
    }
}
exports.Link = Link;
/**
 * ChildLink class defines a {@link Dot.Edge} for a {@link Graph.Edge} that describes a parent-child {@link Graph.Node} relationship
 * @internal
 */
class ChildLink extends BaseEdge {
    /** @internal */
    constructor(from, to) {
        super(from, to);
        this.attributes.apply(theme_1.GraphTheme.instance.childLink);
    }
}
exports.ChildLink = ChildLink;
/**
 * ReferenceLink class defines a {@link Dot.Edge} for a {@link Graph.Reference} edge
 * @internal
 */
class ReferenceLink extends Link {
    /** @internal */
    constructor(edge, from, to) {
        super(edge, from, to);
        this.attributes.apply(theme_1.GraphTheme.instance.referenceLink);
    }
}
exports.ReferenceLink = ReferenceLink;
/**
 * DependencyLink class defines a {@link Dot.Edge} for a {@link Graph.Dependency} edge
 * @internal
 */
class DependencyLink extends Link {
    /** @internal */
    constructor(edge, from, to) {
        super(edge, from, to);
        this.attributes.apply(theme_1.GraphTheme.instance.dependencyLink);
    }
}
exports.DependencyLink = DependencyLink;
//# sourceMappingURL=data:application/json;base64,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