"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Diagram = void 0;
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const Dot = require("ts-graphviz");
const theme_1 = require("../theme");
/**
 * Diagram class defines the root {@link Dot.Digraph} of the diagram
 * @internal
 */
class Diagram extends Dot.Digraph {
    /** @internal */
    constructor(label, imagepath) {
        const { graph, node, edge } = theme_1.GraphTheme.instance;
        super("ROOT", {
            label,
            imagepath,
        });
        this._trackedImages = new Set();
        this.imagepath = imagepath;
        this.apply(graph);
        if (theme_1.GraphTheme.instance.rendering.layout === "vertical") {
            this.set("rankdir", "LR");
        }
        // NB: do not apply "subgraph" attributes as there seems to be bug where it overrides the diagram attributes
        // The subgraph class will apply them directly so not a concern
        this.attributes.node.apply(node);
        this.attributes.edge.apply(edge);
    }
    /**
     * Track image used in the graph for downstream tooling support integration.
     *
     * @see [dot-wasm options](https://hpcc-systems.github.io/hpcc-js-wasm/classes/graphviz.Graphviz.html#layout)
     * @param image Image to track
     */
    trackImage(image) {
        this._trackedImages.add(image);
    }
    /** Get list of all tracked images */
    getTrackedImages() {
        return Array.from(this._trackedImages);
    }
    /**
     * Converts the diagram to **dot** string format
     * @see {@link Dot.toDot}
     */
    toDot() {
        return Dot.toDot(this);
    }
}
exports.Diagram = Diagram;
//# sourceMappingURL=data:application/json;base64,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