"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CdkGraph = exports.CdkGraphContext = exports.CdkGraphArtifacts = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const chalk = require("chalk"); // eslint-disable-line @typescript-eslint/no-require-imports
const constructs_1 = require("constructs");
const fs = require("fs-extra");
const cdk_internals_1 = require("./cdk-internals");
const config_1 = require("./config");
const core_1 = require("./core");
const GRAPH_ARTIFACT_ID = "GRAPH";
/** CdkGraph core artifacts */
var CdkGraphArtifacts;
(function (CdkGraphArtifacts) {
    CdkGraphArtifacts["GRAPH_METADATA"] = "graph-metadata.json";
    CdkGraphArtifacts["GRAPH"] = "graph.json";
})(CdkGraphArtifacts || (exports.CdkGraphArtifacts = CdkGraphArtifacts = {}));
/** CdkGraph context */
class CdkGraphContext {
    constructor(store, outdir) {
        this.store = store;
        this.outdir = outdir;
        /** @internal */
        this._artifacts = {};
    }
    /**
     * Get CdkGraph artifact by id
     * @throws Error is artifact does not exist
     */
    getArtifact(id) {
        const artifact = this._artifacts[id];
        if (artifact) {
            return artifact;
        }
        throw new Error(`Graph artifact ${id} does not exist`);
    }
    /** Get CdkGraph core `graph.json` artifact */
    get graphJson() {
        return this.getArtifact(GRAPH_ARTIFACT_ID);
    }
    /** Indicates if context has an artifact with *filename* defined */
    hasArtifactFile(filename) {
        return !!Object.values(this._artifacts).find((artifact) => artifact.filename === filename);
    }
    /** Get record of all graph artifacts keyed by artifact id */
    get artifacts() {
        return this._artifacts;
    }
    /**
     * Logs an artifact entry. In general this should not be called directly, as `writeArtifact` should be utilized
     * to perform writing and logging artifacts. However some plugins utilize other tools that generate the artifacts,
     * in which case the plugin would call this method to log the entry.
     * @param source The source of the artifact, such as the name of plugin
     * @param id Unique id of the artifact
     * @param filepath Full path where the artifact is stored
     * @param description Description of the artifact
     * @returns
     * @throws Error is artifact id or filename already exists
     */
    logArtifact(source, id, filepath, description) {
        if (id in this._artifacts) {
            throw new Error(`Graph artifact ${id} already defined`);
        }
        if (this.hasArtifactFile(filepath)) {
            throw new Error(`Graph artifact "${filepath}" already defined`);
        }
        const filename = path.relative(this.outdir, filepath);
        if (!(source instanceof CdkGraph)) {
            if (Object.keys(CdkGraphArtifacts).includes(id)) {
                throw new Error(`Graph artifact id ${id} is reserved`);
            }
            if (Object.values(CdkGraphArtifacts).includes(filename)) {
                throw new Error(`Graph artifact file ${filename} is reserved`);
            }
        }
        const artifact = {
            id,
            filepath,
            description,
            filename,
            source: source instanceof CdkGraph
                ? `${CdkGraph.ID}`
                : `plugin:${source.id}@${source.version}`,
        };
        this._artifacts[id] = artifact;
        console.info(chalk.cyanBright(`[CdkGraph] Artifact ${id} written to \x1B]8;;file://${artifact.filepath}\x1B\\${artifact.filename}\x1B]8;;\x1B\\ (${artifact.source})`));
        return artifact;
    }
    /**
     * Writes artifact data to outdir and logs the entry.
     * @param source The source of the artifact, such as the name of plugin
     * @param id Unique id of the artifact
     * @param filename Relative name of the file
     * @param description Description of the artifact
     * @returns
     */
    writeArtifact(source, id, filename, data, description) {
        const filepath = path.join(this.outdir, filename);
        const artifact = this.logArtifact(source, id, filepath, description);
        fs.ensureDirSync(path.dirname(filepath));
        fs.writeFileSync(filepath, data, { encoding: "utf-8" });
        return artifact;
    }
}
exports.CdkGraphContext = CdkGraphContext;
_a = JSII_RTTI_SYMBOL_1;
CdkGraphContext[_a] = { fqn: "@aws/pdk.cdk_graph.CdkGraphContext", version: "0.23.9" };
/**
 * CdkGraph construct is the cdk-graph framework controller that is responsible for
 * computing the graph, storing serialized graph, and instrumenting plugins per the
 * plugin contract.
 */
class CdkGraph extends constructs_1.Construct {
    /**
     * Get the context for the graph instance.
     *
     * This will be `undefined` before construct synthesis has initiated.
     */
    get graphContext() {
        return this._context;
    }
    constructor(root, props = {}) {
        super(root, CdkGraph.ID);
        this.root = root;
        this.config = (0, config_1.resolveConfig)();
        this.plugins = props.plugins || [];
        // TODO: verify plugin deps via semver
        // bind all plugins to this instance of the graph
        this.plugins.forEach((plugin) => {
            plugin.bind(this);
        });
        // Apply Aspect for each plugin that supports "inspect" phase
        this.plugins.forEach((plugin) => {
            if (plugin.inspect) {
                aws_cdk_lib_1.Aspects.of(this.root).add({
                    visit: plugin.inspect,
                });
            }
        });
        (0, cdk_internals_1.addCustomSynthesis)(this, {
            onSynthesize: (session) => {
                this._synthesize(session);
            },
        });
    }
    /** @internal */
    _synthesize(session) {
        const store = (0, core_1.computeGraph)(this.root);
        const outdir = (0, config_1.resolveOutdir)(session.outdir, this.config.outdir);
        const context = new CdkGraphContext(store, outdir);
        context.writeArtifact(this, GRAPH_ARTIFACT_ID, CdkGraphArtifacts.GRAPH, JSON.stringify(context.store.serialize(), null, 2), "Serialized graph");
        this.plugins.forEach((plugin) => {
            plugin.synthesize && plugin.synthesize(context);
        });
        fs.writeFileSync(path.join(outdir, CdkGraphArtifacts.GRAPH_METADATA), JSON.stringify({
            version: CdkGraph.VERSION,
            artifacts: context.artifacts,
        }, null, 2), { encoding: "utf-8" });
        // store context for reporting
        this._context = context;
    }
    /**
     * Asynchronous report generation. This operation enables running expensive and non-synchronous
     * report generation by plugins post synthesis.
     *
     * If a given plugin requires performing asynchronous operations or is general expensive, it should
     * utilize `report` rather than `synthesize`.
     */
    async report() {
        if (this._context == null) {
            // TODO: support deserializing pdk-graph to generate store/context
            console.warn(chalk.yellowBright("[CdkGraph] In the near future, reports will be runnable outside of cdk synth"));
            throw new Error("CdkGraph report called outside of cdk synth process");
        }
        for (const plugin of this.plugins) {
            plugin.report && (await plugin.report(this._context));
        }
    }
}
exports.CdkGraph = CdkGraph;
_b = JSII_RTTI_SYMBOL_1;
CdkGraph[_b] = { fqn: "@aws/pdk.cdk_graph.CdkGraph", version: "0.23.9" };
/** Fixed CdkGraph construct id */
CdkGraph.ID = core_1.GRAPH_ID;
/** Current CdkGraph semantic version */
CdkGraph.VERSION = "0.0.0"; // TODO: make dynamic from package
//# sourceMappingURL=data:application/json;base64,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