"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.compactFilterPreset = exports.nonExtraneousFilterPreset = exports.focusFilter = void 0;
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const filters_1 = require("./filters");
/**
 * Focus the graph on a specific node.
 * @throws Error if store is not filterable
 * @destructive
 */
function focusFilter(store, plan) {
    filters_1.Filters.verifyFilterable(store);
    if (plan.focus == null)
        return; // noop
    let focusedNode;
    let hoist = true;
    const { filter: _filter, noHoist: _noHoist } = plan.focus;
    focusedNode = _filter.filter(store);
    hoist = !_noHoist;
    if (focusedNode === store.root)
        return; // noop
    const ancestors = focusedNode.scopes.slice();
    // remove the actual store.root from ancestors (we can't destroy that)
    const rootAncestor = ancestors.shift();
    if (rootAncestor !== store.root) {
        throw new Error(`${focusedNode} is not within the store root graph: it has root of ${rootAncestor}`);
    }
    if (hoist) {
        // Move focused node as direct child of store root and prune all other ancestors
        focusedNode.mutateHoist(store.root);
        // Only need to destroy to first non-root ancestor to prune the ancestral tree
        if (ancestors.length) {
            ancestors[0].mutateDestroy();
        }
        // prune all other root children (unless preserved)
        store.root.children.forEach((child) => {
            if (child !== focusedNode) {
                child.mutateDestroy();
            }
        });
    }
    else {
        // keep the focused node in place, but prune non-direct ancestor chain nodes
        // the direct ancestor chain is only the nodes scopes
        ancestors.reverse().forEach((ancestor) => {
            ancestor.siblings.forEach((ancestorSibling) => {
                ancestorSibling.mutateDestroy();
            });
        });
        // prune all planRoot siblings
        focusedNode.siblings.forEach((sibling) => {
            sibling.mutateDestroy();
        });
    }
}
exports.focusFilter = focusFilter;
/**
 * Preset that performs {@link Filters.pruneExtraneous} filter on store.
 * @throws Error if store is not filterable
 * @destructive
 */
function nonExtraneousFilterPreset(store) {
    return filters_1.Filters.pruneExtraneous().filter(store);
}
exports.nonExtraneousFilterPreset = nonExtraneousFilterPreset;
/**
 * Preset that performs {@link Filters.compact} filter on store.
 * @throws Error if store is not filterable
 * @destructive
 */
function compactFilterPreset(store) {
    filters_1.Filters.compact().filter(store);
}
exports.compactFilterPreset = compactFilterPreset;
//# sourceMappingURL=data:application/json;base64,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