"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CloudscapeReactTsWebsiteProject = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const fs = require("fs");
const path = require("path");
const Mustache = require("mustache");
const projen_1 = require("projen");
const javascript_1 = require("projen/lib/javascript");
const web_1 = require("projen/lib/web");
/**
 * Synthesizes a Cloudscape React Typescript Website Project.
 *
 * @pjid cloudscape-react-ts-website
 */
class CloudscapeReactTsWebsiteProject extends web_1.ReactTypeScriptProject {
    constructor(options) {
        super({
            ...options,
            defaultReleaseBranch: options.defaultReleaseBranch ?? "main",
            name: options.name,
            sampleCode: false,
            prettier: options.prettier || true,
            packageManager: options.parent && options.parent instanceof javascript_1.NodeProject
                ? options.parent.package.packageManager
                : options.packageManager,
            readme: {
                contents: fs
                    .readFileSync(path.resolve(__dirname, "../samples/cloudscape-react-ts-website/README.md"))
                    .toString(),
            },
            gitignore: [
                "public/runtime-config.json",
                "public/api.json",
                "public/*/api.json",
                ...(options.gitignore || []),
            ],
        });
        this.typeSafeApis = [
            ...(options.typeSafeApis || []),
            ...(options.typeSafeApi ? [options.typeSafeApi] : []),
        ];
        this.addDeps("@aws-northstar/ui", "@cloudscape-design/components", "@cloudscape-design/board-components", "react-router-dom");
        this.testTask.reset();
        const lintTask = this.tasks.tryFind("eslint");
        lintTask && this.testTask.spawn(lintTask);
        this.testTask.exec("react-scripts test --watchAll=false --passWithNoTests");
        this.applicationName = options.applicationName ?? "Sample App";
        this.publicDir = options.publicDir ?? "public";
        const srcDir = path.resolve(__dirname, "../samples/cloudscape-react-ts-website/src");
        const publicDir = path.resolve(__dirname, "../samples/cloudscape-react-ts-website/public");
        this.typeSafeApis.forEach((typeSafeApi) => {
            const hooks = typeSafeApi.library?.typescriptReactQueryHooks;
            const libraryHooksPackage = hooks?.package?.packageName;
            const libraryHooksPackageVersion = hooks?.package?.manifest.version;
            if (!libraryHooksPackage) {
                throw new Error("Cannot pass in a Type Safe Api without React Hooks Library configured!");
            }
            this.addDeps(`${libraryHooksPackage}@${libraryHooksPackageVersion}`);
            this.setupSwaggerUi(typeSafeApi);
        });
        const apis = this.typeSafeApis.map((tsApi, idx) => ({
            apiName: tsApi.model.apiName,
            isLast: idx === this.typeSafeApis.length - 1,
            apiNameSafe: tsApi.model.apiName
                ?.replace(/[^a-z0-9_]+/gi, "")
                .replace(/^[0-9]+/gi, ""),
            hooksPackage: tsApi.library?.typescriptReactQueryHooks?.package?.packageName,
        }));
        const mustacheConfig = {
            applicationName: this.applicationName,
            typeSafeApis: apis,
            typeSafeApisReversed: [...apis].reverse(),
        };
        new projen_1.SampleDir(this, this.srcdir, {
            files: {
                ...Object.fromEntries(this.buildSampleDirEntries(srcDir, [], mustacheConfig)),
            },
        });
        new projen_1.SampleDir(this, this.publicDir, {
            files: {
                ...Object.fromEntries(this.buildSampleDirEntries(publicDir, [], mustacheConfig)),
            },
        });
        // Linting is managed as part of the test task already, so disable react-scripts running eslint again
        this.tasks.addEnvironment("DISABLE_ESLINT_PLUGIN", "true");
        // Relax EsLint and TSC for dev
        this.tasks.tryFind("dev")?.env("ESLINT_NO_DEV_ERRORS", "true");
        this.tasks.tryFind("dev")?.env("TSC_COMPILE_ON_ERROR", "true");
    }
    setupSwaggerUi(tsApi) {
        this.addDevDeps("@types/swagger-ui-react");
        this.addDeps("swagger-ui-react@5.5.0", "aws4fetch");
        const targetApiSpecFolder = `public/${tsApi.model.apiName}`;
        const targetApiSpecPath = `${targetApiSpecFolder}/api.json`;
        this.postCompileTask.exec(`rm -rf ${targetApiSpecFolder}`);
        this.postCompileTask.exec(`mkdir -p ${targetApiSpecFolder} && cp ${path.relative(this.outdir, tsApi.model.outdir)}/.api.json ${targetApiSpecPath}`);
    }
    buildSampleDirEntries(dir, pathPrefixes = [], mustacheConfig) {
        return fs
            .readdirSync(dir, { withFileTypes: true })
            .filter((f) => mustacheConfig.typeSafeApis.length > 0 ||
            (!`${pathPrefixes.join("/")}${f.name}`.includes("TypeSafeApiClient") &&
                !`${pathPrefixes.join("/")}${f.name}`.includes("ApiExplorer")))
            .flatMap((f) => f.isDirectory()
            ? this.buildSampleDirEntries(`${dir}/${f.name}`, [...pathPrefixes, f.name], mustacheConfig)
            : [
                [
                    `${path.join(...pathPrefixes, f.name.replace(".mustache", ""))}`,
                    Mustache.render(fs.readFileSync(`${dir}/${f.name}`).toString(), mustacheConfig),
                ],
            ]);
    }
}
exports.CloudscapeReactTsWebsiteProject = CloudscapeReactTsWebsiteProject;
_a = JSII_RTTI_SYMBOL_1;
CloudscapeReactTsWebsiteProject[_a] = { fqn: "@aws/pdk.cloudscape_react_ts_website.CloudscapeReactTsWebsiteProject", version: "0.23.9" };
//# sourceMappingURL=data:application/json;base64,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