"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.InfrastructureCommands = void 0;
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const monorepo_1 = require("../../monorepo");
const projen_1 = require("projen");
/**
 * Common commands for infrastructure projects
 */
class InfrastructureCommands extends projen_1.Component {
    /**
     * Retrieves an instance of InfrastructureCommands if one is associated to the given project
     * @param project project instance
     */
    static of(project) {
        return project.components.find((c) => monorepo_1.ProjectUtils.isNamedInstanceOf(c, InfrastructureCommands));
    }
    /**
     * Retrieves an instance of InfrastructureCommands if one is associated to the given project,
     * otherwise creates an InfrastructureCommands instance for the project.
     * @param project project instance
     */
    static ensure(project) {
        return (InfrastructureCommands.of(project) || new InfrastructureCommands(project));
    }
    constructor(project) {
        super(project);
        // Add a development deployment task which uses hotswap for faster deployments
        // See: https://aws.amazon.com/blogs/developer/increasing-development-speed-with-cdk-watch/
        const deployDevTask = project.addTask("deploy:dev", {
            receiveArgs: true,
            description: "Performs a hotswap CDK deployment, useful for faster development cycles",
        });
        // --hotswap-fallback falls back to a regular deployment if there are resources which have
        // changed that cannot be hotswapped
        deployDevTask.exec("cdk deploy --hotswap-fallback --require-approval never", {
            receiveArgs: true,
        });
    }
}
exports.InfrastructureCommands = InfrastructureCommands;
//# sourceMappingURL=data:application/json;base64,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