"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.InfrastructureJavaProject = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const fs = require("fs");
const path = require("path");
const monorepo_1 = require("../../../monorepo");
const Mustache = require("mustache");
const projen_1 = require("projen");
const awscdk_1 = require("projen/lib/awscdk");
const infrastructure_commands_1 = require("../../components/infrastructure-commands");
const consts_1 = require("../../consts");
/**
 * Synthesizes a Infrastructure Typescript Project.
 */
class InfrastructureJavaProject extends awscdk_1.AwsCdkJavaApp {
    constructor(options) {
        const groupId = options.groupId ?? "software.aws.infra";
        const artifactId = options.artifactId ?? "infra";
        super({
            ...options,
            cdkVersion: options.cdkVersion ?? "2.1.0",
            sample: false,
            junit: false,
            groupId,
            artifactId,
            mainClass: `${groupId}.Main`,
            version: options.version ?? "0.0.0",
            name: options.name,
            readme: {
                contents: fs
                    .readFileSync(path.resolve(__dirname, "../../../samples/infrastructure/java/README.md"))
                    .toString(),
            },
        });
        infrastructure_commands_1.InfrastructureCommands.ensure(this);
        this.pom.addPlugin("org.apache.maven.plugins/maven-surefire-plugin@3.1.2");
        this.pom.addPlugin("org.apache.maven.plugins/maven-compiler-plugin@3.8.1", {
            configuration: {
                release: "11",
            },
        });
        if (options.junit !== false) {
            [
                "org.junit.jupiter/junit-jupiter-api@^5",
                "org.junit.jupiter/junit-jupiter-engine@^5",
                "io.github.origin-energy/java-snapshot-testing-junit5@^4.0.6",
                "io.github.origin-energy/java-snapshot-testing-plugin-jackson@^4.0.6",
                "org.slf4j/slf4j-simple@2.0.0-alpha0",
            ].forEach((d) => this.addTestDependency(d));
            this.testTask.exec("mvn test");
        }
        this.addDependency("software.aws/pdk@^0");
        const srcDir = path.resolve(__dirname, "../../../samples/infrastructure/java/src");
        const testDir = path.resolve(__dirname, "../../../samples/infrastructure/java/test");
        const typeSafeApis = [
            ...(options.typeSafeApis || []),
            ...(options.typeSafeApi ? [options.typeSafeApi] : []),
        ];
        const cloudscapeReactTsWebsites = [
            ...(options.cloudscapeReactTsWebsites || []),
            ...(options.cloudscapeReactTsWebsite
                ? [options.cloudscapeReactTsWebsite]
                : []),
        ];
        typeSafeApis.forEach((tsApi) => {
            if (!tsApi.infrastructure.java) {
                throw new Error("Cannot pass in a Type Safe Api without Java Infrastructure configured!");
            }
            monorepo_1.NxProject.ensure(this).addJavaDependency(tsApi.infrastructure.java);
            // Ensure handlers are built before infra
            tsApi.all.handlers?.forEach((handler) => {
                monorepo_1.NxProject.ensure(this).addImplicitDependency(handler);
            });
        });
        cloudscapeReactTsWebsites.forEach((csWebsite) => {
            // Ensure website is built before infra
            monorepo_1.NxProject.ensure(this).addImplicitDependency(csWebsite);
        });
        const mustacheConfig = {
            stackName: options.stackName || consts_1.DEFAULT_STACK_NAME,
            groupId,
            typeSafeApis: this.generateTypeSafeMustacheConfig(groupId, typeSafeApis),
            cloudscapeReactTsWebsites: cloudscapeReactTsWebsites.map((csWebsite) => {
                const websiteName = this.capitalize(csWebsite.package.packageName
                    .replace(/[^a-z0-9_]+/gi, "")
                    .replace(/^[0-9]+/gi, ""));
                return {
                    websiteName,
                    websiteNameLowercase: websiteName.toLowerCase(),
                    groupId,
                    websiteDistRelativePath: path.relative(this.outdir, `${csWebsite.outdir}/build`),
                    typeSafeApis: this.generateTypeSafeMustacheConfig(groupId, csWebsite.typeSafeApis),
                };
            }),
        };
        options.sample !== false &&
            this.emitSampleFiles(srcDir, ["src", "main"], mustacheConfig);
        options.sample !== false &&
            this.emitSampleFiles(testDir, ["src", "test"], mustacheConfig);
    }
    generateTypeSafeMustacheConfig(groupId, typeSafeApis) {
        return typeSafeApis?.map((tsApi, idx) => {
            const apiName = this.capitalize(tsApi.model
                .apiName.replace(/[^a-z0-9_]+/gi, "")
                .replace(/^[0-9]+/gi, ""));
            return {
                apiName,
                apiNameLowercase: apiName?.toLowerCase(),
                groupId,
                infraPackage: `${tsApi?.infrastructure.java?.pom.groupId}.${tsApi?.infrastructure.java?.pom.name}.infra`,
                isLast: idx === typeSafeApis.length - 1,
            };
        });
    }
    capitalize(word) {
        return word.charAt(0).toUpperCase() + word.slice(1);
    }
    emitSampleFiles(dir, pathPrefixes = [], mustacheConfig) {
        fs.readdirSync(dir, { withFileTypes: true }).forEach((f) => {
            if (f.isDirectory()) {
                return this.emitSampleFiles(`${dir}/${f.name}`, [
                    ...pathPrefixes,
                    ...(f.name === "groupId"
                        ? mustacheConfig.groupId.split(".")
                        : [f.name]),
                ], mustacheConfig);
            }
            else if (f.name.endsWith("ApiConstruct.java.mustache")) {
                mustacheConfig.typeSafeApis.forEach((tsApi) => {
                    new projen_1.SampleFile(this, `${path.join(...pathPrefixes, `${tsApi.apiName}.java`)}`, {
                        contents: Mustache.render(fs.readFileSync(`${dir}/${f.name}`).toString(), tsApi),
                    });
                });
            }
            else if (f.name.endsWith("WebsiteConstruct.java.mustache")) {
                mustacheConfig.cloudscapeReactTsWebsites.forEach((csWebsite) => {
                    new projen_1.SampleFile(this, `${path.join(...pathPrefixes, `${csWebsite.websiteName}.java`)}`, {
                        contents: Mustache.render(fs.readFileSync(`${dir}/${f.name}`).toString(), csWebsite),
                    });
                });
            }
            else {
                const contents = Mustache.render(fs.readFileSync(`${dir}/${f.name}`).toString(), mustacheConfig);
                return new projen_1.SampleFile(this, `${path.join(...pathPrefixes, f.name.replace(".mustache", ""))}`, {
                    contents,
                    sourcePath: (!contents && `${dir}/${f.name}`) || undefined,
                });
            }
        });
    }
}
exports.InfrastructureJavaProject = InfrastructureJavaProject;
_a = JSII_RTTI_SYMBOL_1;
InfrastructureJavaProject[_a] = { fqn: "@aws/pdk.infrastructure.InfrastructureJavaProject", version: "0.23.9" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiaW5mcmFzdHJ1Y3R1cmUtamF2YS1wcm9qZWN0LmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiaW5mcmFzdHJ1Y3R1cmUtamF2YS1wcm9qZWN0LnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7Ozs7O0FBQUE7c0NBQ3NDO0FBQ3RDLHlCQUF5QjtBQUN6Qiw2QkFBNkI7QUFFN0IsNENBQTBDO0FBRTFDLHFDQUFxQztBQUNyQyxtQ0FBb0M7QUFDcEMsOENBQWtEO0FBRWxELHNGQUFrRjtBQUNsRix5Q0FBa0Q7QUFvQ2xEOztHQUVHO0FBQ0gsTUFBYSx5QkFBMEIsU0FBUSxzQkFBYTtJQUMxRCxZQUFZLE9BQXlDO1FBQ25ELE1BQU0sT0FBTyxHQUFHLE9BQU8sQ0FBQyxPQUFPLElBQUksb0JBQW9CLENBQUM7UUFDeEQsTUFBTSxVQUFVLEdBQUcsT0FBTyxDQUFDLFVBQVUsSUFBSSxPQUFPLENBQUM7UUFFakQsS0FBSyxDQUFDO1lBQ0osR0FBRyxPQUFPO1lBQ1YsVUFBVSxFQUFFLE9BQU8sQ0FBQyxVQUFVLElBQUksT0FBTztZQUN6QyxNQUFNLEVBQUUsS0FBSztZQUNiLEtBQUssRUFBRSxLQUFLO1lBQ1osT0FBTztZQUNQLFVBQVU7WUFDVixTQUFTLEVBQUUsR0FBRyxPQUFPLE9BQU87WUFDNUIsT0FBTyxFQUFFLE9BQU8sQ0FBQyxPQUFPLElBQUksT0FBTztZQUNuQyxJQUFJLEVBQUUsT0FBTyxDQUFDLElBQUk7WUFDbEIsTUFBTSxFQUFFO2dCQUNOLFFBQVEsRUFBRSxFQUFFO3FCQUNULFlBQVksQ0FDWCxJQUFJLENBQUMsT0FBTyxDQUNWLFNBQVMsRUFDVCxnREFBZ0QsQ0FDakQsQ0FDRjtxQkFDQSxRQUFRLEVBQUU7YUFDZDtTQUNGLENBQUMsQ0FBQztRQUVILGdEQUFzQixDQUFDLE1BQU0sQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUVwQyxJQUFJLENBQUMsR0FBRyxDQUFDLFNBQVMsQ0FBQyxzREFBc0QsQ0FBQyxDQUFDO1FBQzNFLElBQUksQ0FBQyxHQUFHLENBQUMsU0FBUyxDQUFDLHNEQUFzRCxFQUFFO1lBQ3pFLGFBQWEsRUFBRTtnQkFDYixPQUFPLEVBQUUsSUFBSTthQUNkO1NBQ0YsQ0FBQyxDQUFDO1FBRUgsSUFBSSxPQUFPLENBQUMsS0FBSyxLQUFLLEtBQUssRUFBRTtZQUMzQjtnQkFDRSx3Q0FBd0M7Z0JBQ3hDLDJDQUEyQztnQkFDM0MsNkRBQTZEO2dCQUM3RCxxRUFBcUU7Z0JBQ3JFLHFDQUFxQzthQUN0QyxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUMsRUFBRSxFQUFFLENBQUMsSUFBSSxDQUFDLGlCQUFpQixDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUM7WUFDNUMsSUFBSSxDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDLENBQUM7U0FDaEM7UUFFRCxJQUFJLENBQUMsYUFBYSxDQUFDLHFCQUFxQixDQUFDLENBQUM7UUFFMUMsTUFBTSxNQUFNLEdBQUcsSUFBSSxDQUFDLE9BQU8sQ0FDekIsU0FBUyxFQUNULDBDQUEwQyxDQUMzQyxDQUFDO1FBQ0YsTUFBTSxPQUFPLEdBQUcsSUFBSSxDQUFDLE9BQU8sQ0FDMUIsU0FBUyxFQUNULDJDQUEyQyxDQUM1QyxDQUFDO1FBRUYsTUFBTSxZQUFZLEdBQUc7WUFDbkIsR0FBRyxDQUFDLE9BQU8sQ0FBQyxZQUFZLElBQUksRUFBRSxDQUFDO1lBQy9CLEdBQUcsQ0FBQyxPQUFPLENBQUMsV0FBVyxDQUFDLENBQUMsQ0FBQyxDQUFDLE9BQU8sQ0FBQyxXQUFXLENBQUMsQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDO1NBQ3RELENBQUM7UUFDRixNQUFNLHlCQUF5QixHQUFHO1lBQ2hDLEdBQUcsQ0FBQyxPQUFPLENBQUMseUJBQXlCLElBQUksRUFBRSxDQUFDO1lBQzVDLEdBQUcsQ0FBQyxPQUFPLENBQUMsd0JBQXdCO2dCQUNsQyxDQUFDLENBQUMsQ0FBQyxPQUFPLENBQUMsd0JBQXdCLENBQUM7Z0JBQ3BDLENBQUMsQ0FBQyxFQUFFLENBQUM7U0FDUixDQUFDO1FBRUYsWUFBWSxDQUFDLE9BQU8sQ0FBQyxDQUFDLEtBQUssRUFBRSxFQUFFO1lBQzdCLElBQUksQ0FBQyxLQUFLLENBQUMsY0FBYyxDQUFDLElBQUksRUFBRTtnQkFDOUIsTUFBTSxJQUFJLEtBQUssQ0FDYix3RUFBd0UsQ0FDekUsQ0FBQzthQUNIO1lBQ0Qsb0JBQVMsQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDLENBQUMsaUJBQWlCLENBQUMsS0FBSyxDQUFDLGNBQWMsQ0FBQyxJQUFJLENBQUMsQ0FBQztZQUNwRSx5Q0FBeUM7WUFDekMsS0FBSyxDQUFDLEdBQUcsQ0FBQyxRQUFRLEVBQUUsT0FBTyxDQUFDLENBQUMsT0FBTyxFQUFFLEVBQUU7Z0JBQ3RDLG9CQUFTLENBQUMsTUFBTSxDQUFDLElBQUksQ0FBQyxDQUFDLHFCQUFxQixDQUFDLE9BQU8sQ0FBQyxDQUFDO1lBQ3hELENBQUMsQ0FBQyxDQUFDO1FBQ0wsQ0FBQyxDQUFDLENBQUM7UUFFSCx5QkFBeUIsQ0FBQyxPQUFPLENBQUMsQ0FBQyxTQUFTLEVBQUUsRUFBRTtZQUM5Qyx1Q0FBdUM7WUFDdkMsb0JBQVMsQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDLENBQUMscUJBQXFCLENBQUMsU0FBUyxDQUFDLENBQUM7UUFDMUQsQ0FBQyxDQUFDLENBQUM7UUFFSCxNQUFNLGNBQWMsR0FBRztZQUNyQixTQUFTLEVBQUUsT0FBTyxDQUFDLFNBQVMsSUFBSSwyQkFBa0I7WUFDbEQsT0FBTztZQUNQLFlBQVksRUFBRSxJQUFJLENBQUMsOEJBQThCLENBQUMsT0FBTyxFQUFFLFlBQVksQ0FBQztZQUN4RSx5QkFBeUIsRUFBRSx5QkFBeUIsQ0FBQyxHQUFHLENBQUMsQ0FBQyxTQUFTLEVBQUUsRUFBRTtnQkFDckUsTUFBTSxXQUFXLEdBQUcsSUFBSSxDQUFDLFVBQVUsQ0FDakMsU0FBUyxDQUFDLE9BQU8sQ0FBQyxXQUFXO3FCQUMxQixPQUFPLENBQUMsZUFBZSxFQUFFLEVBQUUsQ0FBQztxQkFDNUIsT0FBTyxDQUFDLFdBQVcsRUFBRSxFQUFFLENBQUMsQ0FDNUIsQ0FBQztnQkFDRixPQUFPO29CQUNMLFdBQVc7b0JBQ1gsb0JBQW9CLEVBQUUsV0FBVyxDQUFDLFdBQVcsRUFBRTtvQkFDL0MsT0FBTztvQkFDUCx1QkFBdUIsRUFBRSxJQUFJLENBQUMsUUFBUSxDQUNwQyxJQUFJLENBQUMsTUFBTSxFQUNYLEdBQUcsU0FBUyxDQUFDLE1BQU0sUUFBUSxDQUM1QjtvQkFDRCxZQUFZLEVBQUUsSUFBSSxDQUFDLDhCQUE4QixDQUMvQyxPQUFPLEVBQ1AsU0FBUyxDQUFDLFlBQVksQ0FDdkI7aUJBQ0YsQ0FBQztZQUNKLENBQUMsQ0FBQztTQUNILENBQUM7UUFFRixPQUFPLENBQUMsTUFBTSxLQUFLLEtBQUs7WUFDdEIsSUFBSSxDQUFDLGVBQWUsQ0FBQyxNQUFNLEVBQUUsQ0FBQyxLQUFLLEVBQUUsTUFBTSxDQUFDLEVBQUUsY0FBYyxDQUFDLENBQUM7UUFDaEUsT0FBTyxDQUFDLE1BQU0sS0FBSyxLQUFLO1lBQ3RCLElBQUksQ0FBQyxlQUFlLENBQUMsT0FBTyxFQUFFLENBQUMsS0FBSyxFQUFFLE1BQU0sQ0FBQyxFQUFFLGNBQWMsQ0FBQyxDQUFDO0lBQ25FLENBQUM7SUFFTyw4QkFBOEIsQ0FDcEMsT0FBZSxFQUNmLFlBQW1DO1FBRW5DLE9BQU8sWUFBWSxFQUFFLEdBQUcsQ0FBQyxDQUFDLEtBQUssRUFBRSxHQUFHLEVBQUUsRUFBRTtZQUN0QyxNQUFNLE9BQU8sR0FBRyxJQUFJLENBQUMsVUFBVSxDQUM3QixLQUFLLENBQUMsS0FBSztpQkFDUixPQUFRLENBQUMsT0FBTyxDQUFDLGVBQWUsRUFBRSxFQUFFLENBQUM7aUJBQ3JDLE9BQU8sQ0FBQyxXQUFXLEVBQUUsRUFBRSxDQUFDLENBQzVCLENBQUM7WUFDRixPQUFPO2dCQUNMLE9BQU87Z0JBQ1AsZ0JBQWdCLEVBQUUsT0FBTyxFQUFFLFdBQVcsRUFBRTtnQkFDeEMsT0FBTztnQkFDUCxZQUFZLEVBQUUsR0FBRyxLQUFLLEVBQUUsY0FBYyxDQUFDLElBQUksRUFBRSxHQUFHLENBQUMsT0FBTyxJQUFJLEtBQUssRUFBRSxjQUFjLENBQUMsSUFBSSxFQUFFLEdBQUcsQ0FBQyxJQUFJLFFBQVE7Z0JBQ3hHLE1BQU0sRUFBRSxHQUFHLEtBQUssWUFBWSxDQUFDLE1BQU0sR0FBRyxDQUFDO2FBQ3hDLENBQUM7UUFDSixDQUFDLENBQUMsQ0FBQztJQUNMLENBQUM7SUFFTyxVQUFVLENBQUMsSUFBWTtRQUM3QixPQUFPLElBQUksQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDLENBQUMsV0FBVyxFQUFFLEdBQUcsSUFBSSxDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUMsQ0FBQztJQUN0RCxDQUFDO0lBRU8sZUFBZSxDQUNyQixHQUFXLEVBQ1gsZUFBeUIsRUFBRSxFQUMzQixjQUFtQjtRQUVuQixFQUFFLENBQUMsV0FBVyxDQUFDLEdBQUcsRUFBRSxFQUFFLGFBQWEsRUFBRSxJQUFJLEVBQUUsQ0FBQyxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUMsRUFBRSxFQUFFO1lBQ3pELElBQUksQ0FBQyxDQUFDLFdBQVcsRUFBRSxFQUFFO2dCQUNuQixPQUFPLElBQUksQ0FBQyxlQUFlLENBQ3pCLEdBQUcsR0FBRyxJQUFJLENBQUMsQ0FBQyxJQUFJLEVBQUUsRUFDbEI7b0JBQ0UsR0FBRyxZQUFZO29CQUNmLEdBQUcsQ0FBQyxDQUFDLENBQUMsSUFBSSxLQUFLLFNBQVM7d0JBQ3RCLENBQUMsQ0FBQyxjQUFjLENBQUMsT0FBTyxDQUFDLEtBQUssQ0FBQyxHQUFHLENBQUM7d0JBQ25DLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsQ0FBQztpQkFDZCxFQUNELGNBQWMsQ0FDZixDQUFDO2FBQ0g7aUJBQU0sSUFBSSxDQUFDLENBQUMsSUFBSSxDQUFDLFFBQVEsQ0FBQyw0QkFBNEIsQ0FBQyxFQUFFO2dCQUN4RCxjQUFjLENBQUMsWUFBWSxDQUFDLE9BQU8sQ0FBQyxDQUFDLEtBQVUsRUFBRSxFQUFFO29CQUNqRCxJQUFJLG1CQUFVLENBQ1osSUFBSSxFQUNKLEdBQUcsSUFBSSxDQUFDLElBQUksQ0FBQyxHQUFHLFlBQVksRUFBRSxHQUFHLEtBQUssQ0FBQyxPQUFPLE9BQU8sQ0FBQyxFQUFFLEVBQ3hEO3dCQUNFLFFBQVEsRUFBRSxRQUFRLENBQUMsTUFBTSxDQUN2QixFQUFFLENBQUMsWUFBWSxDQUFDLEdBQUcsR0FBRyxJQUFJLENBQUMsQ0FBQyxJQUFJLEVBQUUsQ0FBQyxDQUFDLFFBQVEsRUFBRSxFQUM5QyxLQUFLLENBQ047cUJBQ0YsQ0FDRixDQUFDO2dCQUNKLENBQUMsQ0FBQyxDQUFDO2FBQ0o7aUJBQU0sSUFBSSxDQUFDLENBQUMsSUFBSSxDQUFDLFFBQVEsQ0FBQyxnQ0FBZ0MsQ0FBQyxFQUFFO2dCQUM1RCxjQUFjLENBQUMseUJBQXlCLENBQUMsT0FBTyxDQUFDLENBQUMsU0FBYyxFQUFFLEVBQUU7b0JBQ2xFLElBQUksbUJBQVUsQ0FDWixJQUFJLEVBQ0osR0FBRyxJQUFJLENBQUMsSUFBSSxDQUFDLEdBQUcsWUFBWSxFQUFFLEdBQUcsU0FBUyxDQUFDLFdBQVcsT0FBTyxDQUFDLEVBQUUsRUFDaEU7d0JBQ0UsUUFBUSxFQUFFLFFBQVEsQ0FBQyxNQUFNLENBQ3ZCLEVBQUUsQ0FBQyxZQUFZLENBQUMsR0FBRyxHQUFHLElBQUksQ0FBQyxDQUFDLElBQUksRUFBRSxDQUFDLENBQUMsUUFBUSxFQUFFLEVBQzlDLFNBQVMsQ0FDVjtxQkFDRixDQUNGLENBQUM7Z0JBQ0osQ0FBQyxDQUFDLENBQUM7YUFDSjtpQkFBTTtnQkFDTCxNQUFNLFFBQVEsR0FBRyxRQUFRLENBQUMsTUFBTSxDQUM5QixFQUFFLENBQUMsWUFBWSxDQUFDLEdBQUcsR0FBRyxJQUFJLENBQUMsQ0FBQyxJQUFJLEVBQUUsQ0FBQyxDQUFDLFFBQVEsRUFBRSxFQUM5QyxjQUFjLENBQ2YsQ0FBQztnQkFDRixPQUFPLElBQUksbUJBQVUsQ0FDbkIsSUFBSSxFQUNKLEdBQUcsSUFBSSxDQUFDLElBQUksQ0FBQyxHQUFHLFlBQVksRUFBRSxDQUFDLENBQUMsSUFBSSxDQUFDLE9BQU8sQ0FBQyxXQUFXLEVBQUUsRUFBRSxDQUFDLENBQUMsRUFBRSxFQUNoRTtvQkFDRSxRQUFRO29CQUNSLFVBQVUsRUFBRSxDQUFDLENBQUMsUUFBUSxJQUFJLEdBQUcsR0FBRyxJQUFJLENBQUMsQ0FBQyxJQUFJLEVBQUUsQ0FBQyxJQUFJLFNBQVM7aUJBQzNELENBQ0YsQ0FBQzthQUNIO1FBQ0gsQ0FBQyxDQUFDLENBQUM7SUFDTCxDQUFDOztBQXpNSCw4REEwTUMiLCJzb3VyY2VzQ29udGVudCI6WyIvKiEgQ29weXJpZ2h0IFtBbWF6b24uY29tXShodHRwOi8vYW1hem9uLmNvbS8pLCBJbmMuIG9yIGl0cyBhZmZpbGlhdGVzLiBBbGwgUmlnaHRzIFJlc2VydmVkLlxuU1BEWC1MaWNlbnNlLUlkZW50aWZpZXI6IEFwYWNoZS0yLjAgKi9cbmltcG9ydCAqIGFzIGZzIGZyb20gXCJmc1wiO1xuaW1wb3J0ICogYXMgcGF0aCBmcm9tIFwicGF0aFwiO1xuaW1wb3J0IHsgQ2xvdWRzY2FwZVJlYWN0VHNXZWJzaXRlUHJvamVjdCB9IGZyb20gXCJAYXdzL2Nsb3Vkc2NhcGUtcmVhY3QtdHMtd2Vic2l0ZVwiO1xuaW1wb3J0IHsgTnhQcm9qZWN0IH0gZnJvbSBcIkBhd3MvbW9ub3JlcG9cIjtcbmltcG9ydCB7IFR5cGVTYWZlQXBpUHJvamVjdCB9IGZyb20gXCJAYXdzL3R5cGUtc2FmZS1hcGlcIjtcbmltcG9ydCAqIGFzIE11c3RhY2hlIGZyb20gXCJtdXN0YWNoZVwiO1xuaW1wb3J0IHsgU2FtcGxlRmlsZSB9IGZyb20gXCJwcm9qZW5cIjtcbmltcG9ydCB7IEF3c0Nka0phdmFBcHAgfSBmcm9tIFwicHJvamVuL2xpYi9hd3NjZGtcIjtcbmltcG9ydCB7IEF3c0Nka0phdmFBcHBPcHRpb25zIH0gZnJvbSBcIi4vYXdzLWNkay1qYXZhLWFwcC1vcHRpb25zXCI7XG5pbXBvcnQgeyBJbmZyYXN0cnVjdHVyZUNvbW1hbmRzIH0gZnJvbSBcIi4uLy4uL2NvbXBvbmVudHMvaW5mcmFzdHJ1Y3R1cmUtY29tbWFuZHNcIjtcbmltcG9ydCB7IERFRkFVTFRfU1RBQ0tfTkFNRSB9IGZyb20gXCIuLi8uLi9jb25zdHNcIjtcblxuLyoqXG4gKiBDb25maWd1cmF0aW9uIG9wdGlvbnMgZm9yIHRoZSBJbmZyYXN0cnVjdHVyZUphdmFQcm9qZWN0LlxuICovXG5leHBvcnQgaW50ZXJmYWNlIEluZnJhc3RydWN0dXJlSmF2YVByb2plY3RPcHRpb25zIGV4dGVuZHMgQXdzQ2RrSmF2YUFwcE9wdGlvbnMge1xuICAvKipcbiAgICogU3RhY2sgbmFtZS5cbiAgICpcbiAgICogQGRlZmF1bHQgaW5mcmEtZGV2XG4gICAqL1xuICByZWFkb25seSBzdGFja05hbWU/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFR5cGVTYWZlQXBpIGluc3RhbmNlIHRvIHVzZSB3aGVuIHNldHRpbmcgdXAgdGhlIGluaXRpYWwgcHJvamVjdCBzYW1wbGUgY29kZS5cbiAgICogQGRlcHJlY2F0ZWQgdXNlIHR5cGVTYWZlQXBpc1xuICAgKi9cbiAgcmVhZG9ubHkgdHlwZVNhZmVBcGk/OiBUeXBlU2FmZUFwaVByb2plY3Q7XG5cbiAgLyoqXG4gICAqIENsb3Vkc2NhcGVSZWFjdFRzV2Vic2l0ZVByb2plY3QgaW5zdGFuY2UgdG8gdXNlIHdoZW4gc2V0dGluZyB1cCB0aGUgaW5pdGlhbCBwcm9qZWN0IHNhbXBsZSBjb2RlLlxuICAgKiBAZGVwcmVjYXRlZCB1c2UgY2xvdWRzY2FwZVJlYWN0VHNXZWJzaXRlc1xuICAgKi9cbiAgcmVhZG9ubHkgY2xvdWRzY2FwZVJlYWN0VHNXZWJzaXRlPzogQ2xvdWRzY2FwZVJlYWN0VHNXZWJzaXRlUHJvamVjdDtcblxuICAvKipcbiAgICogVHlwZVNhZmVBcGkgaW5zdGFuY2VzIHRvIHVzZSB3aGVuIHNldHRpbmcgdXAgdGhlIGluaXRpYWwgcHJvamVjdCBzYW1wbGUgY29kZS5cbiAgICovXG4gIHJlYWRvbmx5IHR5cGVTYWZlQXBpcz86IFR5cGVTYWZlQXBpUHJvamVjdFtdO1xuXG4gIC8qKlxuICAgKiBDbG91ZHNjYXBlUmVhY3RUc1dlYnNpdGVQcm9qZWN0IGluc3RhbmNlcyB0byB1c2Ugd2hlbiBzZXR0aW5nIHVwIHRoZSBpbml0aWFsIHByb2plY3Qgc2FtcGxlIGNvZGUuXG4gICAqL1xuICByZWFkb25seSBjbG91ZHNjYXBlUmVhY3RUc1dlYnNpdGVzPzogQ2xvdWRzY2FwZVJlYWN0VHNXZWJzaXRlUHJvamVjdFtdO1xufVxuXG4vKipcbiAqIFN5bnRoZXNpemVzIGEgSW5mcmFzdHJ1Y3R1cmUgVHlwZXNjcmlwdCBQcm9qZWN0LlxuICovXG5leHBvcnQgY2xhc3MgSW5mcmFzdHJ1Y3R1cmVKYXZhUHJvamVjdCBleHRlbmRzIEF3c0Nka0phdmFBcHAge1xuICBjb25zdHJ1Y3RvcihvcHRpb25zOiBJbmZyYXN0cnVjdHVyZUphdmFQcm9qZWN0T3B0aW9ucykge1xuICAgIGNvbnN0IGdyb3VwSWQgPSBvcHRpb25zLmdyb3VwSWQgPz8gXCJzb2Z0d2FyZS5hd3MuaW5mcmFcIjtcbiAgICBjb25zdCBhcnRpZmFjdElkID0gb3B0aW9ucy5hcnRpZmFjdElkID8/IFwiaW5mcmFcIjtcblxuICAgIHN1cGVyKHtcbiAgICAgIC4uLm9wdGlvbnMsXG4gICAgICBjZGtWZXJzaW9uOiBvcHRpb25zLmNka1ZlcnNpb24gPz8gXCIyLjEuMFwiLFxuICAgICAgc2FtcGxlOiBmYWxzZSxcbiAgICAgIGp1bml0OiBmYWxzZSxcbiAgICAgIGdyb3VwSWQsXG4gICAgICBhcnRpZmFjdElkLFxuICAgICAgbWFpbkNsYXNzOiBgJHtncm91cElkfS5NYWluYCxcbiAgICAgIHZlcnNpb246IG9wdGlvbnMudmVyc2lvbiA/PyBcIjAuMC4wXCIsXG4gICAgICBuYW1lOiBvcHRpb25zLm5hbWUsXG4gICAgICByZWFkbWU6IHtcbiAgICAgICAgY29udGVudHM6IGZzXG4gICAgICAgICAgLnJlYWRGaWxlU3luYyhcbiAgICAgICAgICAgIHBhdGgucmVzb2x2ZShcbiAgICAgICAgICAgICAgX19kaXJuYW1lLFxuICAgICAgICAgICAgICBcIi4uLy4uLy4uL3NhbXBsZXMvaW5mcmFzdHJ1Y3R1cmUvamF2YS9SRUFETUUubWRcIlxuICAgICAgICAgICAgKVxuICAgICAgICAgIClcbiAgICAgICAgICAudG9TdHJpbmcoKSxcbiAgICAgIH0sXG4gICAgfSk7XG5cbiAgICBJbmZyYXN0cnVjdHVyZUNvbW1hbmRzLmVuc3VyZSh0aGlzKTtcblxuICAgIHRoaXMucG9tLmFkZFBsdWdpbihcIm9yZy5hcGFjaGUubWF2ZW4ucGx1Z2lucy9tYXZlbi1zdXJlZmlyZS1wbHVnaW5AMy4xLjJcIik7XG4gICAgdGhpcy5wb20uYWRkUGx1Z2luKFwib3JnLmFwYWNoZS5tYXZlbi5wbHVnaW5zL21hdmVuLWNvbXBpbGVyLXBsdWdpbkAzLjguMVwiLCB7XG4gICAgICBjb25maWd1cmF0aW9uOiB7XG4gICAgICAgIHJlbGVhc2U6IFwiMTFcIixcbiAgICAgIH0sXG4gICAgfSk7XG5cbiAgICBpZiAob3B0aW9ucy5qdW5pdCAhPT0gZmFsc2UpIHtcbiAgICAgIFtcbiAgICAgICAgXCJvcmcuanVuaXQuanVwaXRlci9qdW5pdC1qdXBpdGVyLWFwaUBeNVwiLFxuICAgICAgICBcIm9yZy5qdW5pdC5qdXBpdGVyL2p1bml0LWp1cGl0ZXItZW5naW5lQF41XCIsXG4gICAgICAgIFwiaW8uZ2l0aHViLm9yaWdpbi1lbmVyZ3kvamF2YS1zbmFwc2hvdC10ZXN0aW5nLWp1bml0NUBeNC4wLjZcIixcbiAgICAgICAgXCJpby5naXRodWIub3JpZ2luLWVuZXJneS9qYXZhLXNuYXBzaG90LXRlc3RpbmctcGx1Z2luLWphY2tzb25AXjQuMC42XCIsXG4gICAgICAgIFwib3JnLnNsZjRqL3NsZjRqLXNpbXBsZUAyLjAuMC1hbHBoYTBcIixcbiAgICAgIF0uZm9yRWFjaCgoZCkgPT4gdGhpcy5hZGRUZXN0RGVwZW5kZW5jeShkKSk7XG4gICAgICB0aGlzLnRlc3RUYXNrLmV4ZWMoXCJtdm4gdGVzdFwiKTtcbiAgICB9XG5cbiAgICB0aGlzLmFkZERlcGVuZGVuY3koXCJzb2Z0d2FyZS5hd3MvcGRrQF4wXCIpO1xuXG4gICAgY29uc3Qgc3JjRGlyID0gcGF0aC5yZXNvbHZlKFxuICAgICAgX19kaXJuYW1lLFxuICAgICAgXCIuLi8uLi8uLi9zYW1wbGVzL2luZnJhc3RydWN0dXJlL2phdmEvc3JjXCJcbiAgICApO1xuICAgIGNvbnN0IHRlc3REaXIgPSBwYXRoLnJlc29sdmUoXG4gICAgICBfX2Rpcm5hbWUsXG4gICAgICBcIi4uLy4uLy4uL3NhbXBsZXMvaW5mcmFzdHJ1Y3R1cmUvamF2YS90ZXN0XCJcbiAgICApO1xuXG4gICAgY29uc3QgdHlwZVNhZmVBcGlzID0gW1xuICAgICAgLi4uKG9wdGlvbnMudHlwZVNhZmVBcGlzIHx8IFtdKSxcbiAgICAgIC4uLihvcHRpb25zLnR5cGVTYWZlQXBpID8gW29wdGlvbnMudHlwZVNhZmVBcGldIDogW10pLFxuICAgIF07XG4gICAgY29uc3QgY2xvdWRzY2FwZVJlYWN0VHNXZWJzaXRlcyA9IFtcbiAgICAgIC4uLihvcHRpb25zLmNsb3Vkc2NhcGVSZWFjdFRzV2Vic2l0ZXMgfHwgW10pLFxuICAgICAgLi4uKG9wdGlvbnMuY2xvdWRzY2FwZVJlYWN0VHNXZWJzaXRlXG4gICAgICAgID8gW29wdGlvbnMuY2xvdWRzY2FwZVJlYWN0VHNXZWJzaXRlXVxuICAgICAgICA6IFtdKSxcbiAgICBdO1xuXG4gICAgdHlwZVNhZmVBcGlzLmZvckVhY2goKHRzQXBpKSA9PiB7XG4gICAgICBpZiAoIXRzQXBpLmluZnJhc3RydWN0dXJlLmphdmEpIHtcbiAgICAgICAgdGhyb3cgbmV3IEVycm9yKFxuICAgICAgICAgIFwiQ2Fubm90IHBhc3MgaW4gYSBUeXBlIFNhZmUgQXBpIHdpdGhvdXQgSmF2YSBJbmZyYXN0cnVjdHVyZSBjb25maWd1cmVkIVwiXG4gICAgICAgICk7XG4gICAgICB9XG4gICAgICBOeFByb2plY3QuZW5zdXJlKHRoaXMpLmFkZEphdmFEZXBlbmRlbmN5KHRzQXBpLmluZnJhc3RydWN0dXJlLmphdmEpO1xuICAgICAgLy8gRW5zdXJlIGhhbmRsZXJzIGFyZSBidWlsdCBiZWZvcmUgaW5mcmFcbiAgICAgIHRzQXBpLmFsbC5oYW5kbGVycz8uZm9yRWFjaCgoaGFuZGxlcikgPT4ge1xuICAgICAgICBOeFByb2plY3QuZW5zdXJlKHRoaXMpLmFkZEltcGxpY2l0RGVwZW5kZW5jeShoYW5kbGVyKTtcbiAgICAgIH0pO1xuICAgIH0pO1xuXG4gICAgY2xvdWRzY2FwZVJlYWN0VHNXZWJzaXRlcy5mb3JFYWNoKChjc1dlYnNpdGUpID0+IHtcbiAgICAgIC8vIEVuc3VyZSB3ZWJzaXRlIGlzIGJ1aWx0IGJlZm9yZSBpbmZyYVxuICAgICAgTnhQcm9qZWN0LmVuc3VyZSh0aGlzKS5hZGRJbXBsaWNpdERlcGVuZGVuY3koY3NXZWJzaXRlKTtcbiAgICB9KTtcblxuICAgIGNvbnN0IG11c3RhY2hlQ29uZmlnID0ge1xuICAgICAgc3RhY2tOYW1lOiBvcHRpb25zLnN0YWNrTmFtZSB8fCBERUZBVUxUX1NUQUNLX05BTUUsXG4gICAgICBncm91cElkLFxuICAgICAgdHlwZVNhZmVBcGlzOiB0aGlzLmdlbmVyYXRlVHlwZVNhZmVNdXN0YWNoZUNvbmZpZyhncm91cElkLCB0eXBlU2FmZUFwaXMpLFxuICAgICAgY2xvdWRzY2FwZVJlYWN0VHNXZWJzaXRlczogY2xvdWRzY2FwZVJlYWN0VHNXZWJzaXRlcy5tYXAoKGNzV2Vic2l0ZSkgPT4ge1xuICAgICAgICBjb25zdCB3ZWJzaXRlTmFtZSA9IHRoaXMuY2FwaXRhbGl6ZShcbiAgICAgICAgICBjc1dlYnNpdGUucGFja2FnZS5wYWNrYWdlTmFtZVxuICAgICAgICAgICAgLnJlcGxhY2UoL1teYS16MC05X10rL2dpLCBcIlwiKVxuICAgICAgICAgICAgLnJlcGxhY2UoL15bMC05XSsvZ2ksIFwiXCIpXG4gICAgICAgICk7XG4gICAgICAgIHJldHVybiB7XG4gICAgICAgICAgd2Vic2l0ZU5hbWUsXG4gICAgICAgICAgd2Vic2l0ZU5hbWVMb3dlcmNhc2U6IHdlYnNpdGVOYW1lLnRvTG93ZXJDYXNlKCksXG4gICAgICAgICAgZ3JvdXBJZCxcbiAgICAgICAgICB3ZWJzaXRlRGlzdFJlbGF0aXZlUGF0aDogcGF0aC5yZWxhdGl2ZShcbiAgICAgICAgICAgIHRoaXMub3V0ZGlyLFxuICAgICAgICAgICAgYCR7Y3NXZWJzaXRlLm91dGRpcn0vYnVpbGRgXG4gICAgICAgICAgKSxcbiAgICAgICAgICB0eXBlU2FmZUFwaXM6IHRoaXMuZ2VuZXJhdGVUeXBlU2FmZU11c3RhY2hlQ29uZmlnKFxuICAgICAgICAgICAgZ3JvdXBJZCxcbiAgICAgICAgICAgIGNzV2Vic2l0ZS50eXBlU2FmZUFwaXNcbiAgICAgICAgICApLFxuICAgICAgICB9O1xuICAgICAgfSksXG4gICAgfTtcblxuICAgIG9wdGlvbnMuc2FtcGxlICE9PSBmYWxzZSAmJlxuICAgICAgdGhpcy5lbWl0U2FtcGxlRmlsZXMoc3JjRGlyLCBbXCJzcmNcIiwgXCJtYWluXCJdLCBtdXN0YWNoZUNvbmZpZyk7XG4gICAgb3B0aW9ucy5zYW1wbGUgIT09IGZhbHNlICYmXG4gICAgICB0aGlzLmVtaXRTYW1wbGVGaWxlcyh0ZXN0RGlyLCBbXCJzcmNcIiwgXCJ0ZXN0XCJdLCBtdXN0YWNoZUNvbmZpZyk7XG4gIH1cblxuICBwcml2YXRlIGdlbmVyYXRlVHlwZVNhZmVNdXN0YWNoZUNvbmZpZyhcbiAgICBncm91cElkOiBzdHJpbmcsXG4gICAgdHlwZVNhZmVBcGlzPzogVHlwZVNhZmVBcGlQcm9qZWN0W11cbiAgKSB7XG4gICAgcmV0dXJuIHR5cGVTYWZlQXBpcz8ubWFwKCh0c0FwaSwgaWR4KSA9PiB7XG4gICAgICBjb25zdCBhcGlOYW1lID0gdGhpcy5jYXBpdGFsaXplKFxuICAgICAgICB0c0FwaS5tb2RlbFxuICAgICAgICAgIC5hcGlOYW1lIS5yZXBsYWNlKC9bXmEtejAtOV9dKy9naSwgXCJcIilcbiAgICAgICAgICAucmVwbGFjZSgvXlswLTldKy9naSwgXCJcIilcbiAgICAgICk7XG4gICAgICByZXR1cm4ge1xuICAgICAgICBhcGlOYW1lLFxuICAgICAgICBhcGlOYW1lTG93ZXJjYXNlOiBhcGlOYW1lPy50b0xvd2VyQ2FzZSgpLFxuICAgICAgICBncm91cElkLFxuICAgICAgICBpbmZyYVBhY2thZ2U6IGAke3RzQXBpPy5pbmZyYXN0cnVjdHVyZS5qYXZhPy5wb20uZ3JvdXBJZH0uJHt0c0FwaT8uaW5mcmFzdHJ1Y3R1cmUuamF2YT8ucG9tLm5hbWV9LmluZnJhYCxcbiAgICAgICAgaXNMYXN0OiBpZHggPT09IHR5cGVTYWZlQXBpcy5sZW5ndGggLSAxLFxuICAgICAgfTtcbiAgICB9KTtcbiAgfVxuXG4gIHByaXZhdGUgY2FwaXRhbGl6ZSh3b3JkOiBzdHJpbmcpIHtcbiAgICByZXR1cm4gd29yZC5jaGFyQXQoMCkudG9VcHBlckNhc2UoKSArIHdvcmQuc2xpY2UoMSk7XG4gIH1cblxuICBwcml2YXRlIGVtaXRTYW1wbGVGaWxlcyhcbiAgICBkaXI6IHN0cmluZyxcbiAgICBwYXRoUHJlZml4ZXM6IHN0cmluZ1tdID0gW10sXG4gICAgbXVzdGFjaGVDb25maWc6IGFueVxuICApIHtcbiAgICBmcy5yZWFkZGlyU3luYyhkaXIsIHsgd2l0aEZpbGVUeXBlczogdHJ1ZSB9KS5mb3JFYWNoKChmKSA9PiB7XG4gICAgICBpZiAoZi5pc0RpcmVjdG9yeSgpKSB7XG4gICAgICAgIHJldHVybiB0aGlzLmVtaXRTYW1wbGVGaWxlcyhcbiAgICAgICAgICBgJHtkaXJ9LyR7Zi5uYW1lfWAsXG4gICAgICAgICAgW1xuICAgICAgICAgICAgLi4ucGF0aFByZWZpeGVzLFxuICAgICAgICAgICAgLi4uKGYubmFtZSA9PT0gXCJncm91cElkXCJcbiAgICAgICAgICAgICAgPyBtdXN0YWNoZUNvbmZpZy5ncm91cElkLnNwbGl0KFwiLlwiKVxuICAgICAgICAgICAgICA6IFtmLm5hbWVdKSxcbiAgICAgICAgICBdLFxuICAgICAgICAgIG11c3RhY2hlQ29uZmlnXG4gICAgICAgICk7XG4gICAgICB9IGVsc2UgaWYgKGYubmFtZS5lbmRzV2l0aChcIkFwaUNvbnN0cnVjdC5qYXZhLm11c3RhY2hlXCIpKSB7XG4gICAgICAgIG11c3RhY2hlQ29uZmlnLnR5cGVTYWZlQXBpcy5mb3JFYWNoKCh0c0FwaTogYW55KSA9PiB7XG4gICAgICAgICAgbmV3IFNhbXBsZUZpbGUoXG4gICAgICAgICAgICB0aGlzLFxuICAgICAgICAgICAgYCR7cGF0aC5qb2luKC4uLnBhdGhQcmVmaXhlcywgYCR7dHNBcGkuYXBpTmFtZX0uamF2YWApfWAsXG4gICAgICAgICAgICB7XG4gICAgICAgICAgICAgIGNvbnRlbnRzOiBNdXN0YWNoZS5yZW5kZXIoXG4gICAgICAgICAgICAgICAgZnMucmVhZEZpbGVTeW5jKGAke2Rpcn0vJHtmLm5hbWV9YCkudG9TdHJpbmcoKSxcbiAgICAgICAgICAgICAgICB0c0FwaVxuICAgICAgICAgICAgICApLFxuICAgICAgICAgICAgfVxuICAgICAgICAgICk7XG4gICAgICAgIH0pO1xuICAgICAgfSBlbHNlIGlmIChmLm5hbWUuZW5kc1dpdGgoXCJXZWJzaXRlQ29uc3RydWN0LmphdmEubXVzdGFjaGVcIikpIHtcbiAgICAgICAgbXVzdGFjaGVDb25maWcuY2xvdWRzY2FwZVJlYWN0VHNXZWJzaXRlcy5mb3JFYWNoKChjc1dlYnNpdGU6IGFueSkgPT4ge1xuICAgICAgICAgIG5ldyBTYW1wbGVGaWxlKFxuICAgICAgICAgICAgdGhpcyxcbiAgICAgICAgICAgIGAke3BhdGguam9pbiguLi5wYXRoUHJlZml4ZXMsIGAke2NzV2Vic2l0ZS53ZWJzaXRlTmFtZX0uamF2YWApfWAsXG4gICAgICAgICAgICB7XG4gICAgICAgICAgICAgIGNvbnRlbnRzOiBNdXN0YWNoZS5yZW5kZXIoXG4gICAgICAgICAgICAgICAgZnMucmVhZEZpbGVTeW5jKGAke2Rpcn0vJHtmLm5hbWV9YCkudG9TdHJpbmcoKSxcbiAgICAgICAgICAgICAgICBjc1dlYnNpdGVcbiAgICAgICAgICAgICAgKSxcbiAgICAgICAgICAgIH1cbiAgICAgICAgICApO1xuICAgICAgICB9KTtcbiAgICAgIH0gZWxzZSB7XG4gICAgICAgIGNvbnN0IGNvbnRlbnRzID0gTXVzdGFjaGUucmVuZGVyKFxuICAgICAgICAgIGZzLnJlYWRGaWxlU3luYyhgJHtkaXJ9LyR7Zi5uYW1lfWApLnRvU3RyaW5nKCksXG4gICAgICAgICAgbXVzdGFjaGVDb25maWdcbiAgICAgICAgKTtcbiAgICAgICAgcmV0dXJuIG5ldyBTYW1wbGVGaWxlKFxuICAgICAgICAgIHRoaXMsXG4gICAgICAgICAgYCR7cGF0aC5qb2luKC4uLnBhdGhQcmVmaXhlcywgZi5uYW1lLnJlcGxhY2UoXCIubXVzdGFjaGVcIiwgXCJcIikpfWAsXG4gICAgICAgICAge1xuICAgICAgICAgICAgY29udGVudHMsXG4gICAgICAgICAgICBzb3VyY2VQYXRoOiAoIWNvbnRlbnRzICYmIGAke2Rpcn0vJHtmLm5hbWV9YCkgfHwgdW5kZWZpbmVkLFxuICAgICAgICAgIH1cbiAgICAgICAgKTtcbiAgICAgIH1cbiAgICB9KTtcbiAgfVxufVxuIl19