"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.InfrastructurePyProject = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const fs = require("fs");
const path = require("path");
const monorepo_1 = require("../../../monorepo");
const Mustache = require("mustache");
const projen_1 = require("projen");
const awscdk_1 = require("projen/lib/awscdk");
const infrastructure_commands_1 = require("../../components/infrastructure-commands");
const consts_1 = require("../../consts");
/**
 * Synthesizes a Infrastructure Python Project.
 */
class InfrastructurePyProject extends awscdk_1.AwsCdkPythonApp {
    constructor(options) {
        const moduleName = options.moduleName ?? "infra";
        super({
            ...options,
            cdkVersion: options.cdkVersion ?? "2.1.0",
            sample: false,
            poetry: true,
            moduleName,
            appEntrypoint: "main.py",
            pytest: options.pytest ?? false,
            version: options.version ?? "0.0.0",
            authorName: options.authorName ?? "pdkuser",
            authorEmail: options.authorEmail ?? "user@pdk.com",
            name: options.name,
            readme: {
                contents: fs
                    .readFileSync(path.resolve(__dirname, "../../../samples/infrastructure/python/README.md"))
                    .toString(),
            },
        });
        infrastructure_commands_1.InfrastructureCommands.ensure(this);
        ["pytest@^7", "syrupy@^4"].forEach((devDep) => this.addDevDependency(devDep));
        ["aws_pdk@^0", "python@^3.9"].forEach((dep) => this.addDependency(dep));
        const srcDir = path.resolve(__dirname, "../../../samples/infrastructure/python/src");
        const testDir = path.resolve(__dirname, "../../../samples/infrastructure/python/test");
        const typeSafeApis = [
            ...(options.typeSafeApis || []),
            ...(options.typeSafeApi ? [options.typeSafeApi] : []),
        ];
        const cloudscapeReactTsWebsites = [
            ...(options.cloudscapeReactTsWebsites || []),
            ...(options.cloudscapeReactTsWebsite
                ? [options.cloudscapeReactTsWebsite]
                : []),
        ];
        typeSafeApis.forEach((tsApi) => {
            if (!tsApi.infrastructure.python) {
                throw new Error("Cannot pass in a Type Safe Api without Python Infrastructure configured!");
            }
            monorepo_1.NxProject.ensure(this).addPythonPoetryDependency(tsApi.infrastructure.python);
            // Ensure handlers are built before infra
            tsApi.all.handlers?.forEach((handler) => {
                monorepo_1.NxProject.ensure(this).addImplicitDependency(handler);
            });
        });
        cloudscapeReactTsWebsites.forEach((csWebsite) => {
            // Ensure website is built before infra
            monorepo_1.NxProject.ensure(this).addImplicitDependency(csWebsite);
        });
        const mustacheConfig = {
            stackName: options.stackName || consts_1.DEFAULT_STACK_NAME,
            moduleName,
            typeSafeApis: this.generateTypeSafeMustacheConfig(moduleName, typeSafeApis),
            cloudscapeReactTsWebsites: cloudscapeReactTsWebsites.map((csWebsite) => {
                const websiteName = this.capitalize(csWebsite.package.packageName
                    .replace(/[^a-z0-9_]+/gi, "")
                    .replace(/^[0-9]+/gi, ""));
                return {
                    websiteName,
                    websiteNameLowercase: websiteName.toLowerCase(),
                    websiteDistRelativePath: path.relative(this.outdir, `${csWebsite.outdir}/build`),
                    typeSafeApis: this.generateTypeSafeMustacheConfig(moduleName, csWebsite.typeSafeApis),
                };
            }),
        };
        const tstDir = "tests";
        options.sample !== false &&
            this.emitSampleFiles(srcDir, [this.moduleName], mustacheConfig);
        options.sample !== false &&
            this.emitSampleFiles(testDir, [tstDir], mustacheConfig);
        this.testTask.reset(`poetry run pytest ${tstDir}/ \${CI:-'--snapshot-update'}`);
    }
    generateTypeSafeMustacheConfig(moduleName, typeSafeApis) {
        return typeSafeApis?.map((tsApi, idx) => {
            const apiName = this.capitalize(tsApi.model
                .apiName.replace(/[^a-z0-9_]+/gi, "")
                .replace(/^[0-9]+/gi, ""));
            return {
                apiName,
                apiNameLowercase: apiName?.toLowerCase(),
                infraPackage: tsApi.infrastructure.python?.moduleName,
                moduleName,
                isLast: idx === typeSafeApis.length - 1,
            };
        });
    }
    capitalize(word) {
        return word.charAt(0).toUpperCase() + word.slice(1);
    }
    emitSampleFiles(dir, pathPrefixes = [], mustacheConfig) {
        fs.readdirSync(dir, { withFileTypes: true }).forEach((f) => {
            if (f.isDirectory()) {
                return this.emitSampleFiles(`${dir}/${f.name}`, [...pathPrefixes, f.name], mustacheConfig);
            }
            else if (f.name.endsWith("api.py.mustache")) {
                mustacheConfig.typeSafeApis.forEach((tsApi) => {
                    new projen_1.SampleFile(this, `${path.join(...pathPrefixes, `${tsApi.apiNameLowercase}.py`)}`, {
                        contents: Mustache.render(fs.readFileSync(`${dir}/${f.name}`).toString(), tsApi),
                    });
                });
            }
            else if (f.name.endsWith("website.py.mustache")) {
                mustacheConfig.cloudscapeReactTsWebsites.forEach((csWebsite) => {
                    new projen_1.SampleFile(this, `${path.join(...pathPrefixes, `${csWebsite.websiteNameLowercase}.py`)}`, {
                        contents: Mustache.render(fs.readFileSync(`${dir}/${f.name}`).toString(), csWebsite),
                    });
                });
            }
            else {
                const contents = Mustache.render(fs.readFileSync(`${dir}/${f.name}`).toString(), mustacheConfig);
                return new projen_1.SampleFile(this, `${path.join(...(f.name !== "main.py.mustache" ? pathPrefixes : []), // emit at the root so package imports work correctly :(
                f.name.replace(".mustache", ""))}`, {
                    contents,
                    sourcePath: (!contents && `${dir}/${f.name}`) || undefined,
                });
            }
        });
    }
}
exports.InfrastructurePyProject = InfrastructurePyProject;
_a = JSII_RTTI_SYMBOL_1;
InfrastructurePyProject[_a] = { fqn: "@aws/pdk.infrastructure.InfrastructurePyProject", version: "0.23.9" };
//# sourceMappingURL=data:application/json;base64,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