"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.InfrastructureTsProject = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const fs = require("fs");
const path = require("path");
const monorepo_1 = require("../../../monorepo");
const Mustache = require("mustache");
const projen_1 = require("projen");
const awscdk_1 = require("projen/lib/awscdk");
const javascript_1 = require("projen/lib/javascript");
const infrastructure_commands_1 = require("../../components/infrastructure-commands");
const consts_1 = require("../../consts");
/**
 * Synthesizes a Infrastructure Typescript Project.
 */
class InfrastructureTsProject extends awscdk_1.AwsCdkTypeScriptApp {
    constructor(options) {
        super({
            ...options,
            defaultReleaseBranch: options.defaultReleaseBranch ?? "main",
            prettier: options.prettier || true,
            packageManager: options.parent && options.parent instanceof javascript_1.NodeProject
                ? options.parent.package.packageManager
                : options.packageManager,
            cdkVersion: options.cdkVersion ?? "2.1.0",
            name: options.name,
            sampleCode: false,
            readme: {
                contents: fs
                    .readFileSync(path.resolve(__dirname, "../../../samples/infrastructure/typescript/README.md"))
                    .toString(),
            },
        });
        infrastructure_commands_1.InfrastructureCommands.ensure(this);
        this.addDeps("@aws/pdk");
        const srcDir = path.resolve(__dirname, "../../../samples/infrastructure/typescript/src");
        const testDir = path.resolve(__dirname, "../../../samples/infrastructure/typescript/test");
        const typeSafeApis = [
            ...(options.typeSafeApis || []),
            ...(options.typeSafeApi ? [options.typeSafeApi] : []),
        ];
        const cloudscapeReactTsWebsites = [
            ...(options.cloudscapeReactTsWebsites || []),
            ...(options.cloudscapeReactTsWebsite
                ? [options.cloudscapeReactTsWebsite]
                : []),
        ];
        typeSafeApis.forEach((tsApi) => {
            if (!tsApi.infrastructure.typescript) {
                throw new Error("Cannot pass in a Type Safe Api without Typescript Infrastructure configured!");
            }
            this.addDeps(`${tsApi.infrastructure.typescript?.package.packageName}@${tsApi.infrastructure.typescript?.package.manifest.version}`);
            // Ensure handlers are built before infra
            tsApi.all.handlers?.forEach((handler) => {
                monorepo_1.NxProject.ensure(this).addImplicitDependency(handler);
            });
        });
        cloudscapeReactTsWebsites.forEach((csWebsite) => {
            // Ensure website is built before infra
            this.addDevDeps(`${csWebsite.package.packageName}@${csWebsite.package.manifest.version}`);
        });
        const mustacheConfig = {
            stackName: options.stackName || consts_1.DEFAULT_STACK_NAME,
            typeSafeApis: this.generateTypeSafeMustacheConfig(typeSafeApis),
            cloudscapeReactTsWebsites: cloudscapeReactTsWebsites.map((csWebsite) => {
                const websiteName = this.capitalize(csWebsite.package.packageName
                    .replace(/[^a-z0-9_]+/gi, "")
                    .replace(/^[0-9]+/gi, ""));
                return {
                    websiteName,
                    websiteNameLowercase: websiteName.toLowerCase(),
                    websiteDistRelativePath: path.relative(this.outdir, `${csWebsite.outdir}/build`),
                    typeSafeApis: this.generateTypeSafeMustacheConfig(csWebsite.typeSafeApis),
                };
            }),
        };
        options.sampleCode !== false &&
            this.emitSampleFiles(srcDir, ["src"], mustacheConfig);
        options.sampleCode !== false &&
            this.emitSampleFiles(testDir, ["test"], mustacheConfig);
        const eslintTask = this.tasks.tryFind("eslint");
        this.testTask.reset();
        this.testTask.exec("jest --passWithNoTests ${CI:-'--updateSnapshot'}");
        eslintTask && this.testTask.spawn(eslintTask);
    }
    generateTypeSafeMustacheConfig(typeSafeApis) {
        return typeSafeApis?.map((tsApi, idx) => {
            const apiName = this.capitalize(tsApi.model
                .apiName.replace(/[^a-z0-9_]+/gi, "")
                .replace(/^[0-9]+/gi, ""));
            return {
                apiName,
                apiNameLowercase: apiName?.toLowerCase(),
                infraPackage: tsApi.infrastructure.typescript?.package.packageName,
                isLast: idx === typeSafeApis.length - 1,
            };
        });
    }
    capitalize(word) {
        return word.charAt(0).toUpperCase() + word.slice(1);
    }
    emitSampleFiles(dir, pathPrefixes = [], mustacheConfig) {
        fs.readdirSync(dir, { withFileTypes: true }).forEach((f) => {
            if (f.isDirectory()) {
                this.emitSampleFiles(`${dir}/${f.name}`, [...pathPrefixes, f.name], mustacheConfig);
            }
            else if (f.name.endsWith("api.ts.mustache")) {
                mustacheConfig.typeSafeApis.forEach((tsApi) => {
                    new projen_1.SampleFile(this, `${path.join(...pathPrefixes, `${tsApi.apiNameLowercase}.ts`)}`, {
                        contents: Mustache.render(fs.readFileSync(`${dir}/${f.name}`).toString(), tsApi),
                    });
                });
            }
            else if (f.name.endsWith("website.ts.mustache")) {
                mustacheConfig.cloudscapeReactTsWebsites.forEach((csWebsite) => {
                    new projen_1.SampleFile(this, `${path.join(...pathPrefixes, `${csWebsite.websiteNameLowercase}.ts`)}`, {
                        contents: Mustache.render(fs.readFileSync(`${dir}/${f.name}`).toString(), csWebsite),
                    });
                });
            }
            else {
                new projen_1.SampleFile(this, `${path.join(...pathPrefixes, f.name.replace(".mustache", ""))}`, {
                    contents: Mustache.render(fs.readFileSync(`${dir}/${f.name}`).toString(), mustacheConfig),
                });
            }
        });
    }
}
exports.InfrastructureTsProject = InfrastructureTsProject;
_a = JSII_RTTI_SYMBOL_1;
InfrastructureTsProject[_a] = { fqn: "@aws/pdk.infrastructure.InfrastructureTsProject", version: "0.23.9" };
//# sourceMappingURL=data:application/json;base64,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