"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.NxConfigurator = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const path = require("path");
const projen_1 = require("projen");
const javascript_1 = require("projen/lib/javascript");
const python_1 = require("projen/lib/python");
const nx_project_1 = require("./nx-project");
const nx_workspace_1 = require("./nx-workspace");
const utils_1 = require("../utils");
/**
 * Configues common NX related tasks and methods.
 */
class NxConfigurator extends projen_1.Component {
    constructor(project, options) {
        super(project);
        this.nxPlugins = {};
        project.addGitIgnore(".nx/cache");
        project.addTask("run-many", {
            receiveArgs: true,
            exec: utils_1.NodePackageUtils.command.exec(utils_1.NodePackageUtils.getNodePackageManager(this.project, javascript_1.NodePackageManager.NPM), "nx", "run-many"),
            description: "Run task against multiple workspace projects",
        });
        project.addTask("graph", {
            receiveArgs: true,
            exec: utils_1.NodePackageUtils.command.exec(utils_1.NodePackageUtils.getNodePackageManager(this.project, javascript_1.NodePackageManager.NPM), "nx", "graph"),
            description: "Generate dependency graph for monorepo",
        });
        this.nx = nx_workspace_1.NxWorkspace.of(project) || new nx_workspace_1.NxWorkspace(project);
        this.nx.affected.defaultBase = options?.defaultReleaseBranch ?? "mainline";
    }
    patchPoetryEnv(project) {
        // Since the root monorepo is a poetry project and sets the VIRTUAL_ENV, and poetry env info -p will print
        // the virtual env set in the VIRTUAL_ENV variable if set, we need to unset it to ensure the local project's
        // env is used.
        if (utils_1.ProjectUtils.isNamedInstanceOf(project.depsManager, python_1.Poetry)) {
            project.tasks.addEnvironment("VIRTUAL_ENV", "$(env -u VIRTUAL_ENV poetry env info -p || echo '')");
            project.tasks.addEnvironment("PATH", "$(echo $(env -u VIRTUAL_ENV poetry env info -p || echo '')/bin:$PATH)");
        }
    }
    patchPythonProjects(projects) {
        projects.forEach((p) => {
            if (utils_1.ProjectUtils.isNamedInstanceOf(p, python_1.PythonProject)) {
                this.patchPoetryEnv(p);
            }
            this.patchPythonProjects(p.subprojects);
        });
    }
    /**
     * Overrides "build" related project tasks (build, compile, test, etc.) with `npx nx run-many` format.
     * @param task - The task or task name to override
     * @param options - Nx run-many options
     * @param overrideOptions - Options for overriding the task
     * @returns - The task that was overridden
     * @internal
     */
    _overrideNxBuildTask(task, options, overrideOptions) {
        if (typeof task === "string") {
            task = this.project.tasks.tryFind(task);
        }
        if (task == null) {
            return;
        }
        if (overrideOptions?.force) {
            // @ts-ignore - private property
            task._locked = false;
        }
        task.reset(this.execNxRunManyCommand(options), {
            receiveArgs: true,
        });
        task.description += " for all affected projects";
        if (overrideOptions?.disableReset) {
            // Prevent any further resets of the task to force it to remain as the overridden nx build task
            task.reset = () => { };
        }
        return task;
    }
    /**
     * Adds a command to upgrade all python subprojects to the given task
     * @param monorepo the monorepo project
     * @param task the task to add the command to
     * @internal
     */
    _configurePythonSubprojectUpgradeDeps(monorepo, task) {
        // Upgrade deps for
        const pythonSubprojects = monorepo.subprojects.filter((p) => utils_1.ProjectUtils.isNamedInstanceOf(p, python_1.PythonProject));
        if (pythonSubprojects.length > 0) {
            task.exec(this.execNxRunManyCommand({
                target: "install",
                projects: pythonSubprojects.map((p) => p.name),
            }), { receiveArgs: true });
        }
    }
    /**
     * Returns the install task or creates one with nx installation command added.
     *
     * Note: this should only be called from non-node projects
     *
     * @param nxPlugins additional plugins to install
     * @returns install task
     */
    ensureNxInstallTask(nxPlugins) {
        this.nxPlugins = nxPlugins;
        const installTask = this.project.tasks.tryFind("install") ?? this.project.addTask("install");
        installTask.exec("yarn install --check-files");
        (this.project.tasks.tryFind("install:ci") ??
            this.project.addTask("install:ci")).exec("yarn install --check-files --frozen-lockfile");
        return installTask;
    }
    /**
     * Helper to format `npx nx run-many ...` style command execution in package manager.
     * @param options
     */
    execNxRunManyCommand(options) {
        return utils_1.NodePackageUtils.command.exec(utils_1.NodePackageUtils.getNodePackageManager(this.project, javascript_1.NodePackageManager.NPM), ...this.composeNxRunManyCommand(options));
    }
    /**
     * Helper to format `npx nx run-many ...` style command
     * @param options
     */
    composeNxRunManyCommand(options) {
        const args = [];
        if (options.configuration) {
            args.push(`--configuration=${options.configuration}`);
        }
        if (options.runner) {
            args.push(`--runner=${options.runner}`);
        }
        if (options.parallel) {
            args.push(`--parallel=${options.parallel}`);
        }
        if (options.skipCache) {
            args.push("--skip-nx-cache");
        }
        if (options.ignoreCycles) {
            args.push("--nx-ignore-cycles");
        }
        if (options.noBail !== true) {
            args.push("--nx-bail");
        }
        if (options.projects && options.projects.length) {
            args.push(`--projects=${options.projects.join(",")}`);
        }
        if (options.exclude) {
            args.push(`--exclude=${options.exclude}`);
        }
        if (options.verbose) {
            args.push("--verbose");
        }
        return [
            "nx",
            "run-many",
            `--target=${options.target}`,
            `--output-style=${options.outputStyle || "stream"}`,
            ...args,
        ];
    }
    /**
     * Add project task that executes `npx nx run-many ...` style command.
     */
    addNxRunManyTask(name, options) {
        return this.project.addTask(name, {
            receiveArgs: true,
            exec: this.execNxRunManyCommand(options),
        });
    }
    /**
     * Create an implicit dependency between two Projects. This is typically
     * used in polygot repos where a Typescript project wants a build dependency
     * on a Python project as an example.
     *
     * @param dependent project you want to have the dependency.
     * @param dependee project you wish to depend on.
     * @throws error if this is called on a dependent which does not have a NXProject component attached.
     */
    addImplicitDependency(dependent, dependee) {
        nx_project_1.NxProject.ensure(dependent).addImplicitDependency(dependee);
    }
    /**
     * Adds a dependency between two Java Projects in the monorepo.
     * @param dependent project you want to have the dependency
     * @param dependee project you wish to depend on
     */
    addJavaDependency(dependent, dependee) {
        nx_project_1.NxProject.ensure(dependent).addJavaDependency(dependee);
    }
    /**
     * Adds a dependency between two Python Projects in the monorepo. The dependent must have Poetry enabled.
     * @param dependent project you want to have the dependency (must be a Poetry Python Project)
     * @param dependee project you wish to depend on
     * @throws error if the dependent does not have Poetry enabled
     */
    addPythonPoetryDependency(dependent, dependee) {
        nx_project_1.NxProject.ensure(dependent).addPythonPoetryDependency(dependee);
    }
    /**
     * Ensures that all non-root projects have NxProject applied.
     * @internal
     */
    _ensureNxProjectGraph() {
        function _ensure(_project) {
            if (_project.root === _project)
                return;
            nx_project_1.NxProject.ensure(_project);
            _project.subprojects.forEach((p) => {
                _ensure(p);
            });
        }
        this.project.subprojects.forEach(_ensure);
    }
    /**
     * Emits package.json for non-node NX monorepos.
     * @internal
     */
    _emitPackageJson() {
        if (!utils_1.ProjectUtils.isNamedInstanceOf(this.project, javascript_1.NodeProject) &&
            !this.project.tryFindFile("package.json")) {
            new projen_1.JsonFile(this.project, "package.json", {
                obj: {
                    devDependencies: {
                        ...this.nxPlugins,
                        nx: "^16",
                        "@nx/devkit": "^16",
                    },
                    private: true,
                    workspaces: this.project.subprojects
                        .filter((p) => utils_1.ProjectUtils.isNamedInstanceOf(p, javascript_1.NodeProject))
                        .map((p) => path.relative(this.project.outdir, p.outdir)),
                },
            }).synthesize();
        }
    }
    preSynthesize() {
        // Calling before super() to ensure proper pre-synth of NxProject component and its nested components
        this._ensureNxProjectGraph();
        this._emitPackageJson();
        this.patchPythonProjects([this.project]);
    }
    /**
     * @inheritDoc
     */
    synth() {
        this.resetDefaultTask();
    }
    /**
     * Ensures subprojects don't have a default task
     */
    resetDefaultTask() {
        this.project.subprojects.forEach((subProject) => {
            // Disable default task on subprojects as this isn't supported in a monorepo
            subProject.defaultTask?.reset();
        });
    }
}
exports.NxConfigurator = NxConfigurator;
_a = JSII_RTTI_SYMBOL_1;
NxConfigurator[_a] = { fqn: "@aws/pdk.monorepo.NxConfigurator", version: "0.23.9" };
//# sourceMappingURL=data:application/json;base64,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