"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.MonorepoPythonProject = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const projen_1 = require("projen");
const javascript_1 = require("projen/lib/javascript");
const python_1 = require("projen/lib/python");
const nx_configurator_1 = require("../../components/nx-configurator");
const nx_project_1 = require("../../components/nx-project");
const projen_dependency_1 = require("../../components/projen-dependency");
const utils_1 = require("../../utils");
/**
 * This project type will bootstrap a NX based monorepo with support for polygot
 * builds, build caching, dependency graph visualization and much more.
 *
 * @pjid monorepo-py
 */
class MonorepoPythonProject extends python_1.PythonProject {
    constructor(options) {
        super({
            ...options,
            sample: false,
            poetry: true,
            pytest: options.pytest ?? false,
            version: options.version ?? "0.0.0",
            authorName: options.authorName ?? "pdkuser",
            authorEmail: options.authorEmail ?? "user@pdk.com",
            projenrcPythonOptions: {
                ...options.projenrcPythonOptions,
                projenVersion: options.projenrcPythonOptions?.projenVersion ??
                    projen_dependency_1.DEFAULT_PROJEN_VERSION,
            },
        });
        this.projenVersion =
            options.projenrcPythonOptions?.projenVersion ?? projen_dependency_1.DEFAULT_PROJEN_VERSION;
        // Remove dependency on typescript package which projen incorrectly adds to initial .projenrc.py
        // See: https://github.com/projen/projen/issues/2475
        this.deps.removeDependency("@aws/pdk");
        this.addDevDependency("aws_pdk@^0");
        this.nxConfigurator = new nx_configurator_1.NxConfigurator(this, {
            defaultReleaseBranch: options.defaultReleaseBranch ?? "main",
        });
        // Setup python NX plugin
        this.nx.plugins.push("@nxlv/python");
        this.installTask = this.nxConfigurator.ensureNxInstallTask({
            "@nxlv/python": "^16",
        });
        // Map tasks to nx run-many
        this.nxConfigurator._overrideNxBuildTask(this.buildTask, { target: "build" }, { force: true });
        this.nxConfigurator._overrideNxBuildTask(this.preCompileTask, {
            target: "pre-compile",
        });
        this.nxConfigurator._overrideNxBuildTask(this.compileTask, {
            target: "compile",
        });
        this.nxConfigurator._overrideNxBuildTask(this.postCompileTask, {
            target: "post-compile",
        });
        this.nxConfigurator._overrideNxBuildTask(this.testTask, {
            target: "test",
        });
        this.nxConfigurator._overrideNxBuildTask(this.packageTask, {
            target: "package",
        });
    }
    /**
     * @inheritdoc
     */
    get nx() {
        return this.nxConfigurator.nx;
    }
    /**
     * @inheritdoc
     */
    execNxRunManyCommand(options) {
        return this.nxConfigurator.execNxRunManyCommand(options);
    }
    /**
     * @inheritdoc
     */
    composeNxRunManyCommand(options) {
        return this.nxConfigurator.composeNxRunManyCommand(options);
    }
    /**
     * @inheritdoc
     */
    addNxRunManyTask(name, options) {
        return this.nxConfigurator.addNxRunManyTask(name, options);
    }
    /**
     * @inheritdoc
     */
    addImplicitDependency(dependent, dependee) {
        this.nxConfigurator.addImplicitDependency(dependent, dependee);
    }
    /**
     * @inheritdoc
     */
    addJavaDependency(dependent, dependee) {
        this.nxConfigurator.addJavaDependency(dependent, dependee);
    }
    /**
     * @inheritdoc
     */
    addPythonPoetryDependency(dependent, dependee) {
        this.nxConfigurator.addPythonPoetryDependency(dependent, dependee);
    }
    addUpgradeDepsTask() {
        const upgradeDepsTask = new projen_1.Task("upgrade-deps", {
            description: "Upgrade dependencies in the monorepo",
        });
        this.nxConfigurator._overrideNxBuildTask(upgradeDepsTask, {
            target: "upgrade",
        });
        // Spawn the install task for python projects since this will update the lockfile to the latest versions satisfying
        // the pyproject.toml file
        // TODO: remove in favour of the "upgrade" task if ever implemented for python
        upgradeDepsTask.spawn(this.depsManager.installTask);
        this.nxConfigurator._configurePythonSubprojectUpgradeDeps(this, upgradeDepsTask);
    }
    /**
     * @inheritdoc
     */
    preSynthesize() {
        // Calling before super() to ensure proper pre-synth of NxProject component and its nested components
        this.nxConfigurator.preSynthesize();
        this.addUpgradeDepsTask();
        super.preSynthesize();
        (0, projen_dependency_1.syncProjenVersions)(this.subprojects, this.projenVersion);
    }
    /**
     * @inheritDoc
     */
    synth() {
        this.validateSubProjects();
        this.installPythonSubprojects();
        this.nxConfigurator.synth();
        super.synth();
    }
    /**
     * @inheritdoc
     *
     * NOTE: Be sure to ensure the VIRTUAL_ENV is unset during postSynthesize as the individual poetry envs will only be created if a existing VIRTUAL_ENV cannot be found.
     */
    postSynthesize() {
        const vEnv = process.env.VIRTUAL_ENV;
        process.env.VIRTUAL_ENV = "";
        super.postSynthesize();
        process.env.VIRTUAL_ENV = vEnv;
    }
    /**
     * Ensures all python subprojects have their install target called after the monorepo install task.
     */
    installPythonSubprojects() {
        const installProjects = this.subprojects.filter((project) => isPythonProject(project));
        if (installProjects.length > 0) {
            const nxRunManyInstall = this.composeNxRunManyCommand({
                target: "install",
                projects: installProjects.map((project) => project.name),
                parallel: 1,
            });
            this.installTask.exec(utils_1.NodePackageUtils.command.exec(javascript_1.NodePackageManager.NPM, ...nxRunManyInstall));
            // Ensure that install-py follows dependency order
            installProjects.forEach((p) => {
                nx_project_1.NxProject.ensure(p).setTarget("install", {
                    dependsOn: ["^install"],
                });
            });
        }
    }
    /**
     * Ensures subprojects don't have a default task and that all Python subpackages are configured to use Poetry.
     */
    validateSubProjects() {
        this.subprojects.forEach((subProject) => {
            // Disable default task on subprojects as this isn't supported in a monorepo
            subProject.defaultTask?.reset();
            if (isPythonProject(subProject) && !isPoetryConfigured(subProject)) {
                throw new Error(`${subProject.name} must use poetry to manage dependencies when using NXMonorepoPythonProject`);
            }
        });
    }
}
exports.MonorepoPythonProject = MonorepoPythonProject;
_a = JSII_RTTI_SYMBOL_1;
MonorepoPythonProject[_a] = { fqn: "@aws/pdk.monorepo.MonorepoPythonProject", version: "0.23.9" };
/**
 * Determines if the passed in project is of type PythonProject.
 *
 * @param project Project instance.
 * @returns true if the project instance is of type PythonProject.
 */
function isPythonProject(project) {
    return utils_1.ProjectUtils.isNamedInstanceOf(project, python_1.PythonProject);
}
/**
 * Determines if the passed in project uses Poetry.
 *
 * @param project PythonProject instance.
 * @returns true if the project uses Poetry.
 */
function isPoetryConfigured(project) {
    return (project.components.find((c) => utils_1.ProjectUtils.isNamedInstanceOf(c, python_1.Poetry)) !== undefined);
}
//# sourceMappingURL=data:application/json;base64,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