"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.MonorepoTsProject = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const path = require("path");
const projen_1 = require("projen");
const javascript_1 = require("projen/lib/javascript");
const typescript_1 = require("projen/lib/typescript");
const syncpack_options_1 = require("./syncpack-options");
const nx_configurator_1 = require("../../components/nx-configurator");
const nx_project_1 = require("../../components/nx-project");
const projen_dependency_1 = require("../../components/projen-dependency");
const utils_1 = require("../../utils");
/**
 * This project type will bootstrap a monorepo with support for polygot
 * builds, build caching, dependency graph visualization and much more.
 *
 * @pjid monorepo-ts
 */
class MonorepoTsProject extends typescript_1.TypeScriptProject {
    constructor(options) {
        const defaultReleaseBranch = options.defaultReleaseBranch ?? "main";
        super({
            ...options,
            github: options.github ?? false,
            package: options.package ?? false,
            projenCommand: options.packageManager
                ? utils_1.NodePackageUtils.command.projen(options.packageManager)
                : undefined,
            prettier: options.prettier ?? true,
            projenrcTs: true,
            release: options.release ?? false,
            jest: options.jest ?? false,
            sampleCode: false,
            gitignore: [".tmp", ...(options.gitignore ?? [])],
            defaultReleaseBranch,
            eslintOptions: options.eslintOptions ?? {
                dirs: ["."],
                ignorePatterns: ["packages/**/*.*"],
            },
            tsconfig: options.tsconfig ?? {
                compilerOptions: {
                    rootDir: ".",
                },
                include: ["**/*.ts", ".projenrc.ts"],
            },
            peerDeps: ["nx@^16", ...(options.peerDeps || [])],
            devDeps: ["nx@^16", "@aws/pdk@^0", ...(options.devDeps || [])],
            deps: [
                "aws-cdk-lib",
                "cdk-nag",
                "@aws-cdk/aws-cognito-identitypool-alpha@latest",
                ...(options.deps || []),
            ],
            projenVersion: options.projenVersion ?? projen_dependency_1.DEFAULT_PROJEN_VERSION,
        });
        this.subNodeProjectResolves = [];
        this.projenVersion = options.projenVersion ?? projen_dependency_1.DEFAULT_PROJEN_VERSION;
        this.nxConfigurator = new nx_configurator_1.NxConfigurator(this, {
            defaultReleaseBranch,
        });
        this._options = options;
        // engines
        this.package.addEngine("node", ">=16");
        switch (this.package.packageManager) {
            case javascript_1.NodePackageManager.BUN: {
                this.package.addEngine("bun", ">=1");
                break;
            }
            case javascript_1.NodePackageManager.PNPM: {
                // https://pnpm.io/package_json
                // https://github.com/pnpm/pnpm/releases/tag/v8.0.0
                this.package.addEngine("pnpm", ">=8");
                break;
            }
            case javascript_1.NodePackageManager.YARN_CLASSIC:
            case javascript_1.NodePackageManager.YARN: {
                this.package.addEngine("yarn", ">=1 <2");
                break;
            }
            case javascript_1.NodePackageManager.YARN_BERRY:
            case javascript_1.NodePackageManager.YARN2: {
                this.package.addEngine("yarn", ">=2");
                this.gitignore.addPatterns(".yarn/*", ".pnp.cjs", "!.yarn/releases", "!.yarn/plugins");
                break;
            }
            case javascript_1.NodePackageManager.NPM: {
                // Allow older versions of peer deps to resolv compatibility issues
                this.tasks.tryFind("install")?.reset("npm install --legacy-peer-deps");
                this.tasks.tryFind("install:ci")?.reset("npm ci --legacy-peer-deps");
                break;
            }
        }
        this.package.setScript("install:ci", utils_1.NodePackageUtils.command.exec(this.package.packageManager, "projen install:ci"));
        this.workspaceConfig = options.workspaceConfig;
        this.workspacePackages = options.workspaceConfig?.additionalPackages ?? [];
        // Never publish a monorepo root package.
        this.package.addField("private", true);
        // Add alias task for "projen" to synthesize workspace
        this.package.setScript("synth-workspace", utils_1.NodePackageUtils.command.projen(this.package.packageManager));
        // Map tasks to nx run-many
        if (options.scripts == null || options.scripts.build == null) {
            this.nxConfigurator._overrideNxBuildTask(this.buildTask, { target: "build" }, { force: true });
        }
        if (options.scripts == null || options.scripts["pre-compile"] == null) {
            this.nxConfigurator._overrideNxBuildTask(this.preCompileTask, {
                target: "pre-compile",
            });
        }
        if (options.scripts == null || options.scripts.compile == null) {
            this.nxConfigurator._overrideNxBuildTask(this.compileTask, {
                target: "compile",
            });
        }
        if (options.scripts == null || options.scripts["post-compile"] == null) {
            this.nxConfigurator._overrideNxBuildTask(this.postCompileTask, {
                target: "post-compile",
            });
        }
        if (options.scripts == null || options.scripts.test == null) {
            this.nxConfigurator._overrideNxBuildTask(this.testTask, {
                target: "test",
            });
        }
        if (options.scripts == null || options.scripts.eslint == null) {
            // The Projenrc component of TypeScriptProject resets the eslint task as part of preSynthesize which would undo
            // our changes, so we disable further resets.
            this.nxConfigurator._overrideNxBuildTask(this.eslint?.eslintTask, { target: "eslint" }, { disableReset: true });
        }
        if (options.scripts == null || options.scripts.package == null) {
            this.nxConfigurator._overrideNxBuildTask(this.packageTask, {
                target: "package",
            });
        }
        if (options.scripts == null || options.scripts.prepare == null) {
            this.nxConfigurator._overrideNxBuildTask("prepare", {
                target: "prepare",
            });
        }
        if (options.scripts == null || options.scripts.watch == null) {
            this.nxConfigurator._overrideNxBuildTask(this.watchTask, {
                target: "watch",
                noBail: false,
                ignoreCycles: true,
                skipCache: true,
                outputStyle: "stream",
            });
        }
        this.package.addPackageResolutions("@types/babel__traverse@7.18.2", "wrap-ansi@^7.0.0", "@zkochan/js-yaml@npm:js-yaml@4.1.0");
    }
    /**
     * @inheritdoc
     */
    get nx() {
        return this.nxConfigurator.nx;
    }
    /**
     * @inheritdoc
     */
    execNxRunManyCommand(options) {
        return this.nxConfigurator.execNxRunManyCommand(options);
    }
    /**
     * @inheritdoc
     */
    composeNxRunManyCommand(options) {
        return this.nxConfigurator.composeNxRunManyCommand(options);
    }
    /**
     * @inheritdoc
     */
    addNxRunManyTask(name, options) {
        return this.nxConfigurator.addNxRunManyTask(name, options);
    }
    /**
     * @inheritdoc
     */
    addImplicitDependency(dependent, dependee) {
        this.nxConfigurator.addImplicitDependency(dependent, dependee);
    }
    /**
     * @inheritdoc
     */
    addJavaDependency(dependent, dependee) {
        this.nxConfigurator.addJavaDependency(dependent, dependee);
    }
    /**
     * @inheritdoc
     */
    addPythonPoetryDependency(dependent, dependee) {
        this.nxConfigurator.addPythonPoetryDependency(dependent, dependee);
    }
    /**
     * Add one or more additional package globs to the workspace.
     * @param packageGlobs paths to the package to include in the workspace (for example packages/my-package)
     */
    addWorkspacePackages(...packageGlobs) {
        // Any subprojects that were added since the last call to this method need to be added first, in order to ensure
        // we add the workspace packages in a sane order.
        const relativeSubProjectWorkspacePackages = this.sortedSubProjects
            .filter((s) => utils_1.ProjectUtils.isNamedInstanceOf(s, javascript_1.NodeProject))
            .map((project) => path.relative(this.outdir, project.outdir));
        const existingWorkspacePackages = new Set(this.workspacePackages);
        this.workspacePackages.push(...relativeSubProjectWorkspacePackages.filter((pkg) => !existingWorkspacePackages.has(pkg)));
        // Add the additional packages next
        this.workspacePackages.push(...packageGlobs);
    }
    /** Get consistently sorted list of subprojects */
    get sortedSubProjects() {
        return this.subprojects
            .slice()
            .sort((a, b) => a.name.localeCompare(b.name));
    }
    /**
     * Create symbolic links to all local workspace bins. This enables the usage of bins the same
     * way as consumers of the packages have when installing from the registry.
     */
    linkLocalWorkspaceBins() {
        const bins = [];
        this.subprojects.forEach((subProject) => {
            if (utils_1.ProjectUtils.isNamedInstanceOf(subProject, javascript_1.NodeProject) &&
                subProject.name !== "@aws/pdk") {
                const pkgBins = subProject.package.manifest.bin() || {};
                bins.push(...Object.entries(pkgBins).map(([cmd, bin]) => {
                    const resolvedBin = path.join("$PWD", path.relative(this.outdir, subProject.outdir), bin);
                    return [cmd, resolvedBin];
                }));
            }
        });
        const linkTask = this.addTask("workspace:bin:link", {
            steps: bins.map(([cmd, bin]) => ({
                exec: `ln -s ${bin} ${utils_1.NodePackageUtils.command.bin(this.package.packageManager, cmd)} &>/dev/null; exit 0;`,
            })),
        });
        (this.tasks.tryFind("prepare") || this.addTask("prepare")).spawn(linkTask);
    }
    addUpgradeDepsTask() {
        if (this._options.monorepoUpgradeDeps !== false) {
            this.addDevDeps("npm-check-updates", "syncpack@^12");
            const upgradeDepsTask = this.addTask(this._options.monorepoUpgradeDepsOptions?.taskName || "upgrade-deps", {
                description: "Upgrade dependencies in the monorepo",
            });
            // Run the upgrade task for any non-node subprojects
            const nonNodeSubprojects = this.subprojects.filter((p) => !utils_1.NodePackageUtils.isNodeProject(p));
            if (nonNodeSubprojects.length > 0) {
                this.nxConfigurator._overrideNxBuildTask(upgradeDepsTask, {
                    target: "upgrade",
                    projects: nonNodeSubprojects.map((p) => p.name),
                });
            }
            this.nxConfigurator._configurePythonSubprojectUpgradeDeps(this, upgradeDepsTask);
            // Upgrade node subprojects together
            upgradeDepsTask.exec(utils_1.NodePackageUtils.command.exec(this.package.packageManager, "npm-check-updates", "--deep", "--rejectVersion", "0.0.0", "-u", "--dep", "prod,dev,peer,optional,bundle", "--target=minor"));
            // Sync dependency versions across node subprojects
            upgradeDepsTask.exec(utils_1.NodePackageUtils.command.exec(this.package.packageManager, "syncpack", "fix-mismatches"));
            upgradeDepsTask.exec(`rm ${this.package.lockFile}`);
            upgradeDepsTask.exec(utils_1.NodePackageUtils.command.install(this.package.packageManager));
            upgradeDepsTask.exec(utils_1.NodePackageUtils.command.exec(this.package.packageManager, "projen"));
            new projen_1.JsonFile(this, ".syncpackrc.json", {
                obj: this._options.monorepoUpgradeDepsOptions?.syncpackConfig ||
                    syncpack_options_1.Syncpack.DEFAULT_CONFIG,
                readonly: true,
            });
        }
    }
    preSynthesize() {
        this.addUpgradeDepsTask();
        utils_1.NodePackageUtils.removeProjenScript(this);
        this.nxConfigurator.preSynthesize();
        super.preSynthesize();
        if (this._options.workspaceConfig?.linkLocalWorkspaceBins === true) {
            this.linkLocalWorkspaceBins();
        }
        if (this.package.packageManager === javascript_1.NodePackageManager.PNPM) {
            // PNPM hoisting hides transitive bundled dependencies which results in
            // transitive dependencies being packed incorrectly.
            this.subprojects.forEach((subProject) => {
                if (utils_1.NodePackageUtils.isNodeProject(subProject) &&
                    getBundledDeps(subProject).length) {
                    const pkgFolder = path.relative(this.root.outdir, subProject.outdir);
                    // Create a symlink in the sub-project node_modules for all transitive deps
                    // before running "package" task
                    subProject.packageTask.prependExec(`monorepo.pnpm-link-bundled-transitive-deps ${pkgFolder}`);
                }
            });
        }
        this.subprojects.forEach((subProject) => {
            if (utils_1.NodePackageUtils.isNodeProject(subProject)) {
                // Remove any subproject .npmrc files since only the root one matters
                subProject.tryRemoveFile(".npmrc");
                utils_1.NodePackageUtils.removeProjenScript(subProject);
            }
        });
        (0, projen_dependency_1.syncProjenVersions)(this.subprojects, this.projenVersion);
    }
    /**
     * @inheritDoc
     */
    synth() {
        this.validateSubProjects();
        this.updateWorkspace();
        this.installNonNodeDependencies();
        this.resolveLocalBunDependencies();
        // Disable node warnings if configured
        if (this._options.disableNodeWarnings) {
            this.disableNodeWarnings();
        }
        // Prevent sub NodeProject packages from `postSynthesis` which will cause individual/extraneous installs.
        // The workspace package install will handle all the sub NodeProject packages automatically.
        this.subprojects.forEach((subProject) => {
            if (utils_1.NodePackageUtils.isNodeProject(subProject)) {
                const subNodeProject = subProject;
                const subNodeProjectResolver = 
                // @ts-ignore - `resolveDepsAndWritePackageJson` is private
                subNodeProject.package.resolveDepsAndWritePackageJson;
                // @ts-ignore - `installDependencies` is private
                subNodeProject.package.installDependencies = () => {
                    this.subNodeProjectResolves.push(() => subNodeProjectResolver.apply(subNodeProject.package));
                };
                // @ts-ignore - `resolveDepsAndWritePackageJson` is private
                subNodeProject.package.resolveDepsAndWritePackageJson = () => { };
            }
        });
        this.nxConfigurator.synth();
        super.synth();
    }
    /**
     * @inheritDoc
     */
    postSynthesize() {
        super.postSynthesize();
        this.resolveSubNodeProjects();
    }
    /**
     * Resolve sub `NodePackage` dependencies.
     */
    resolveSubNodeProjects() {
        if (this.subNodeProjectResolves.length) {
            if (!this.package.file.changed) {
                // Force workspace install deps since it would not have been invoked during `postSynthesis`.
                // @ts-ignore - `installDependencies` is private
                this.package.installDependencies();
            }
            const completedResolves = this.subNodeProjectResolves.map((resolve) => resolve());
            if (completedResolves.some(Boolean)) {
                // Indicates that a subproject dependency has been resolved from '*', so update the lockfile.
                // @ts-ignore - `installDependencies` is private
                this.package.installDependencies();
            }
        }
        this.subNodeProjectResolves = [];
    }
    /**
     * Ensures subprojects don't have a default task and that all packages use the same package manager.
     */
    validateSubProjects() {
        this.subprojects.forEach((subProject) => {
            // Disable default task on subprojects as this isn't supported in a monorepo
            subProject.defaultTask?.reset();
            if (utils_1.NodePackageUtils.isNodeProject(subProject) &&
                subProject.package.packageManager !== this.package.packageManager) {
                throw new Error(`${subProject.name} packageManager does not match the monorepo packageManager: ${this.package.packageManager}.`);
            }
        });
    }
    /**
     * Add a submodule entry to the appropriate workspace file.
     */
    updateWorkspace() {
        // A final call to addWorkspacePackages will update the list of workspace packages with any subprojects that have
        // not yet been added, in the correct order
        this.addWorkspacePackages();
        let noHoist = this.workspaceConfig?.yarn?.noHoist;
        // Automatically add all sub-project "bundledDependencies" to workspace "hohoist", otherwise they are not bundled in npm package
        if (this.workspaceConfig?.yarn?.disableNoHoistBundled !== true &&
            [
                javascript_1.NodePackageManager.YARN,
                javascript_1.NodePackageManager.YARN2,
                javascript_1.NodePackageManager.YARN_BERRY,
                javascript_1.NodePackageManager.YARN_CLASSIC,
            ].includes(this.package.packageManager)) {
            const noHoistBundled = this.subprojects.flatMap((sub) => {
                if (utils_1.ProjectUtils.isNamedInstanceOf(sub, javascript_1.NodeProject)) {
                    return getBundledDeps(sub).flatMap((dep) => [
                        `${sub.name}/${dep.name}`,
                        `${sub.name}/${dep.name}/*`,
                    ]);
                }
                return [];
            });
            if (noHoistBundled.length) {
                noHoist = [...(noHoist || []), ...noHoistBundled];
            }
        }
        // Add workspaces for each subproject
        if (this.package.packageManager === javascript_1.NodePackageManager.PNPM) {
            new projen_1.YamlFile(this, "pnpm-workspace.yaml", {
                readonly: true,
                obj: {
                    packages: this.workspacePackages,
                },
            });
        }
        this.package.addField("workspaces", {
            packages: this.workspacePackages,
            nohoist: noHoist,
        });
    }
    /**
     * Ensures the install task for non-node projects is executed postinstall.
     *
     * @private
     */
    installNonNodeDependencies() {
        const installProjects = this.subprojects.filter((project) => !utils_1.ProjectUtils.isNamedInstanceOf(project, javascript_1.NodeProject) &&
            project.tasks.tryFind("install"));
        if (installProjects.length > 0) {
            // TODO: Install error on clean repo for postinstall (https://cloud.nx.app/runs/MptQr0BxgF) (https://github.com/nrwl/nx/issues/11210)
            const postinstallTask = this.tasks.tryFind("postinstall") ?? this.addTask("postinstall");
            const nxRunManyInstall = this.composeNxRunManyCommand({
                target: "install",
                projects: installProjects.map((project) => project.name),
                parallel: 1,
            });
            postinstallTask.exec(utils_1.NodePackageUtils.command.exec(this.package.packageManager, ...nxRunManyInstall));
            // Ensure that install-py follows dependency order
            installProjects.forEach((p) => {
                nx_project_1.NxProject.ensure(p).setTarget("install", {
                    dependsOn: ["^install"],
                });
            });
        }
    }
    /**
     * Suppress Node warnings from being presented in the console when running builds.
     */
    disableNodeWarnings() {
        this.tasks.addEnvironment("NODE_NO_WARNINGS", "1");
        this.subprojects.forEach((subProject) => subProject.tasks.addEnvironment("NODE_NO_WARNINGS", "1"));
    }
    /**
     * Resolve all local workspace dependencies to keep bun happy.
     */
    resolveLocalBunDependencies() {
        if (this.package.packageManager !== javascript_1.NodePackageManager.BUN) {
            return;
        }
        const nodeSubProjectNames = new Set(this.subprojects
            .filter((s) => utils_1.NodePackageUtils.isNodeProject(s))
            .map((s) => s.package.packageName));
        this.subprojects.forEach((subProject) => {
            if (utils_1.NodePackageUtils.isNodeProject(subProject)) {
                subProject.deps.all
                    .filter((dep) => nodeSubProjectNames.has(dep.name))
                    .forEach((d) => {
                    switch (d.type) {
                        case projen_1.DependencyType.RUNTIME:
                            subProject.addDeps(`${d.name}@workspace:*`);
                            break;
                        case projen_1.DependencyType.BUILD:
                            subProject.addDevDeps(`${d.name}@workspace:*`);
                            break;
                        case projen_1.DependencyType.PEER:
                            subProject.addPeerDeps(`${d.name}@workspace:*`);
                            break;
                        default:
                            console.warn(`Cannot update local dependency due to unsupported type: ${d.type}`);
                    }
                });
            }
        });
    }
}
exports.MonorepoTsProject = MonorepoTsProject;
_a = JSII_RTTI_SYMBOL_1;
MonorepoTsProject[_a] = { fqn: "@aws/pdk.monorepo.MonorepoTsProject", version: "0.23.9" };
/**
 * Gets bundled dependencies for a given project
 */
function getBundledDeps(project) {
    return project.deps.all.filter((dep) => dep.type === projen_1.DependencyType.BUNDLED);
}
//# sourceMappingURL=data:application/json;base64,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