"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.createLockfileObject = exports.readWantedLockfile = exports.readWantedLockfileAndAutofixConflicts = exports.readCurrentLockfile = void 0;
const fs_1 = require("fs");
const path_1 = __importDefault(require("path"));
const constants_1 = require("@pnpm/constants");
const error_1 = require("@pnpm/error");
const merge_lockfile_changes_1 = require("@pnpm/merge-lockfile-changes");
const types_1 = require("@pnpm/types");
const comver_to_semver_1 = __importDefault(require("comver-to-semver"));
const js_yaml_1 = __importDefault(require("js-yaml"));
const semver_1 = __importDefault(require("semver"));
const strip_bom_1 = __importDefault(require("strip-bom"));
const errors_1 = require("./errors");
const gitMergeFile_1 = require("./gitMergeFile");
const logger_1 = require("./logger");
const lockfileName_1 = require("./lockfileName");
const gitBranchLockfile_1 = require("./gitBranchLockfile");
const inlineSpecifiersLockfileConverters_1 = require("./experiments/inlineSpecifiersLockfileConverters");
async function readCurrentLockfile(virtualStoreDir, opts) {
    const lockfilePath = path_1.default.join(virtualStoreDir, 'lock.yaml');
    return (await _read(lockfilePath, virtualStoreDir, opts)).lockfile;
}
exports.readCurrentLockfile = readCurrentLockfile;
async function readWantedLockfileAndAutofixConflicts(pkgPath, opts) {
    return _readWantedLockfile(pkgPath, {
        ...opts,
        autofixMergeConflicts: true,
    });
}
exports.readWantedLockfileAndAutofixConflicts = readWantedLockfileAndAutofixConflicts;
async function readWantedLockfile(pkgPath, opts) {
    return (await _readWantedLockfile(pkgPath, opts)).lockfile;
}
exports.readWantedLockfile = readWantedLockfile;
async function _read(lockfilePath, prefix, // only for logging
opts) {
    let lockfileRawContent;
    try {
        lockfileRawContent = (0, strip_bom_1.default)(await fs_1.promises.readFile(lockfilePath, 'utf8'));
    }
    catch (err) { // eslint-disable-line
        if (err.code !== 'ENOENT') {
            throw err;
        }
        return {
            lockfile: null,
            hadConflicts: false,
        };
    }
    let lockfile;
    let hadConflicts;
    try {
        lockfile = (0, inlineSpecifiersLockfileConverters_1.revertFromInlineSpecifiersFormatIfNecessary)(convertFromLockfileFileMutable(js_yaml_1.default.load(lockfileRawContent)));
        hadConflicts = false;
    }
    catch (err) { // eslint-disable-line
        if (!opts.autofixMergeConflicts || !(0, gitMergeFile_1.isDiff)(lockfileRawContent)) {
            throw new error_1.PnpmError('BROKEN_LOCKFILE', `The lockfile at "${lockfilePath}" is broken: ${err.message}`);
        }
        hadConflicts = true;
        lockfile = convertFromLockfileFileMutable((0, gitMergeFile_1.autofixMergeConflicts)(lockfileRawContent));
        logger_1.lockfileLogger.info({
            message: `Merge conflict detected in ${constants_1.WANTED_LOCKFILE} and successfully merged`,
            prefix,
        });
    }
    if (lockfile) {
        const lockfileSemver = (0, comver_to_semver_1.default)((lockfile.lockfileVersion ?? 0).toString());
        /* eslint-enable @typescript-eslint/dot-notation */
        if (!opts.wantedVersions ||
            opts.wantedVersions.length === 0 ||
            opts.wantedVersions.some((wantedVersion) => {
                if (semver_1.default.major(lockfileSemver) !== semver_1.default.major((0, comver_to_semver_1.default)(wantedVersion)))
                    return false;
                if (lockfile.lockfileVersion !== '6.1' && semver_1.default.gt(lockfileSemver, (0, comver_to_semver_1.default)(wantedVersion))) {
                    logger_1.lockfileLogger.warn({
                        message: `Your ${constants_1.WANTED_LOCKFILE} was generated by a newer version of pnpm. ` +
                            `It is a compatible version but it might get downgraded to version ${wantedVersion}`,
                        prefix,
                    });
                }
                return true;
            })) {
            return { lockfile, hadConflicts };
        }
    }
    if (opts.ignoreIncompatible) {
        logger_1.lockfileLogger.warn({
            message: `Ignoring not compatible lockfile at ${lockfilePath}`,
            prefix,
        });
        return { lockfile: null, hadConflicts: false };
    }
    throw new errors_1.LockfileBreakingChangeError(lockfilePath);
}
function createLockfileObject(importerIds, opts) {
    const importers = importerIds.reduce((acc, importerId) => {
        acc[importerId] = {
            dependencies: {},
            specifiers: {},
        };
        return acc;
    }, {});
    return {
        importers,
        lockfileVersion: opts.lockfileVersion || constants_1.LOCKFILE_VERSION,
        settings: {
            autoInstallPeers: opts.autoInstallPeers,
            excludeLinksFromLockfile: opts.excludeLinksFromLockfile,
        },
    };
}
exports.createLockfileObject = createLockfileObject;
async function _readWantedLockfile(pkgPath, opts) {
    const lockfileNames = [constants_1.WANTED_LOCKFILE];
    if (opts.useGitBranchLockfile) {
        const gitBranchLockfileName = await (0, lockfileName_1.getWantedLockfileName)(opts);
        if (gitBranchLockfileName !== constants_1.WANTED_LOCKFILE) {
            lockfileNames.unshift(gitBranchLockfileName);
        }
    }
    let result = { lockfile: null, hadConflicts: false };
    /* eslint-disable no-await-in-loop */
    for (const lockfileName of lockfileNames) {
        result = await _read(path_1.default.join(pkgPath, lockfileName), pkgPath, { ...opts, autofixMergeConflicts: true });
        if (result.lockfile) {
            if (opts.mergeGitBranchLockfiles) {
                result.lockfile = await _mergeGitBranchLockfiles(result.lockfile, pkgPath, pkgPath, opts);
            }
            break;
        }
    }
    /* eslint-enable no-await-in-loop */
    return result;
}
async function _mergeGitBranchLockfiles(lockfile, lockfileDir, prefix, opts) {
    if (!lockfile) {
        return lockfile;
    }
    const gitBranchLockfiles = (await _readGitBranchLockfiles(lockfileDir, prefix, opts)).map(({ lockfile }) => lockfile);
    let mergedLockfile = lockfile;
    for (const gitBranchLockfile of gitBranchLockfiles) {
        if (!gitBranchLockfile) {
            continue;
        }
        mergedLockfile = (0, merge_lockfile_changes_1.mergeLockfileChanges)(mergedLockfile, gitBranchLockfile);
    }
    return mergedLockfile;
}
async function _readGitBranchLockfiles(lockfileDir, prefix, opts) {
    const files = await (0, gitBranchLockfile_1.getGitBranchLockfileNames)(lockfileDir);
    return Promise.all(files.map((file) => _read(path_1.default.join(lockfileDir, file), prefix, opts)));
}
/**
 * Reverts changes from the "forceSharedFormat" write option if necessary.
 */
function convertFromLockfileFileMutable(lockfileFile) {
    if (typeof lockfileFile?.['importers'] === 'undefined') {
        lockfileFile.importers = {
            '.': {
                specifiers: lockfileFile['specifiers'] ?? {},
                dependenciesMeta: lockfileFile['dependenciesMeta'],
                publishDirectory: lockfileFile['publishDirectory'],
            },
        };
        delete lockfileFile.specifiers;
        for (const depType of types_1.DEPENDENCIES_FIELDS) {
            if (lockfileFile[depType] != null) {
                lockfileFile.importers['.'][depType] = lockfileFile[depType];
                delete lockfileFile[depType];
            }
        }
    }
    return lockfileFile;
}
//# sourceMappingURL=read.js.map