/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
import { App, AppProps, Stack, StageSynthesisOptions } from "aws-cdk-lib";
import { CloudAssembly } from "aws-cdk-lib/cx-api";
import { NagPack, NagPackSuppression } from "cdk-nag";
import { ExtendedNagResult } from "./loggers/types";
/**
 * Message instance.
 */
export interface Message {
    /**
     * Message description.
     */
    readonly messageDescription: string;
    /**
     * Message type as returned from cdk-nag.
     */
    readonly messageType: string;
}
/**
 * Nag result.
 */
export interface NagResult {
    /**
     * Resource which triggered the message.
     */
    readonly resource: string;
    /**
     * List of messages.
     */
    readonly messages: Message[];
}
/**
 * @inheritDoc
 */
export interface PDKNagAppProps extends AppProps {
    /**
     * Determines whether any errors encountered should trigger a test failure.
     *
     * @default false
     */
    readonly failOnError?: boolean;
    /**
     * Determines whether any warnings encountered should trigger a test failure.
     *
     * @default false
     */
    readonly failOnWarning?: boolean;
    /**
     * Custom nag packs to execute.
     *
     * @default DEFAULT_NAG_PACKS
     */
    readonly nagPacks?: NagPack[];
}
/**
 * @inheritDoc
 */
export declare class PDKNagApp extends App {
    private readonly _nagResults;
    private readonly _extendedNagResults;
    private readonly failOnError;
    private readonly failOnWarning;
    readonly nagPacks: NagPack[];
    constructor(props?: PDKNagAppProps);
    synth(options?: StageSynthesisOptions): CloudAssembly;
    addNagResult(result: NagResult): void;
    /**
     * Returns a list of NagResult.
     *
     * Note: app.synth() must be called before this to retrieve results.
     */
    nagResults(): NagResult[];
    addExtendedNagResults(...results: ExtendedNagResult[]): void;
    /**
     * Returns a list of ExtendedNagResult.
     *
     * Note: app.synth() must be called before this to retrieve results.
     */
    extendedNagResults(): ExtendedNagResult[];
}
/**
 * Helper for create a Nag Enabled App.
 */
export declare class PDKNag {
    /**
     * Returns an instance of an App with Nag enabled.
     *
     * @param props props to initialize the app with.
     */
    static app(props?: PDKNagAppProps): PDKNagApp;
    /**
     * Wrapper around NagSuppressions which does not throw.
     *
     * @param stack stack instance
     * @param path resource path
     * @param suppressions list of suppressions to apply.
     * @param applyToChildren whether to apply to children.
     */
    static addResourceSuppressionsByPathNoThrow(stack: Stack, path: string, suppressions: NagPackSuppression[], applyToChildren?: boolean): void;
    /**
     * Returns a prefix comprising of a delimited set of Stack Ids.
     *
     * For example: StackA/NestedStackB/
     *
     * @param stack stack instance.
     */
    static getStackPrefix(stack: Stack): string;
    /**
     * Returns a stack partition regex.
     *
     * @param stack stack instance.
     */
    static getStackPartitionRegex(stack: Stack): string;
    /**
     * Returns a stack region regex.
     *
     * @param stack stack instance.
     */
    static getStackRegionRegex(stack: Stack): string;
    /**
     * Returns a stack account regex.
     *
     * @param stack stack instance.
     */
    static getStackAccountRegex(stack: Stack): string;
}
