"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.PDKNag = exports.PDKNagApp = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const aws_cdk_lib_1 = require("aws-cdk-lib");
const cdk_nag_1 = require("cdk-nag");
const memory_logger_1 = require("./loggers/memory-logger");
const CDK_NAG_MESSAGE_TYPES = {
    ERROR: "aws:cdk:error",
    WARNING: "aws:cdk:warning",
};
const CDK_NAG_MESSAGE_TYPES_SET = new Set(Object.values(CDK_NAG_MESSAGE_TYPES));
const DEFAULT_NAG_PACKS = [
    new cdk_nag_1.AwsSolutionsChecks({
        verbose: true,
        reports: true,
    }),
];
/**
 * @inheritDoc
 */
class PDKNagApp extends aws_cdk_lib_1.App {
    constructor(props) {
        super(props);
        this._nagResults = [];
        this._extendedNagResults = [];
        this.failOnError = props?.failOnError ?? false;
        this.failOnWarning = props?.failOnWarning ?? false;
        this.nagPacks = props?.nagPacks ?? DEFAULT_NAG_PACKS;
        aws_cdk_lib_1.Aspects.of(this).add(new PDKNagAspect(this));
    }
    synth(options) {
        const assembly = super.synth(options);
        const typesToFail = new Set([
            this.failOnError && CDK_NAG_MESSAGE_TYPES.ERROR,
            this.failOnWarning && CDK_NAG_MESSAGE_TYPES.WARNING,
        ].filter((t) => t));
        if (this._nagResults.find((r) => r.messages.find((m) => typesToFail.has(m.messageType)))) {
            throw new Error(JSON.stringify(this._nagResults, undefined, 2));
        }
        return assembly;
    }
    addNagResult(result) {
        this._nagResults.push(result);
    }
    /**
     * Returns a list of NagResult.
     *
     * Note: app.synth() must be called before this to retrieve results.
     */
    nagResults() {
        return this._nagResults;
    }
    addExtendedNagResults(...results) {
        this._extendedNagResults.push(...results);
    }
    /**
     * Returns a list of ExtendedNagResult.
     *
     * Note: app.synth() must be called before this to retrieve results.
     */
    extendedNagResults() {
        return this._extendedNagResults;
    }
}
exports.PDKNagApp = PDKNagApp;
_a = JSII_RTTI_SYMBOL_1;
PDKNagApp[_a] = { fqn: "@aws/pdk.pdk_nag.PDKNagApp", version: "0.23.9" };
class PDKNagAspect {
    constructor(app) {
        this.app = app;
    }
    visit(node) {
        const memoryLogger = new memory_logger_1.MemoryLogger();
        this.app.nagPacks.forEach((nagPack) => {
            // @ts-ignore loggers is private, but since we haven't called "visit" yet it's safe to add another
            nagPack.loggers.push(memoryLogger);
            nagPack.visit(node);
            // @ts-ignore loggers is private, but remove the memory logger to clean up
            nagPack.loggers.pop();
        });
        this.app.addExtendedNagResults(...memoryLogger.results);
        const results = node.node.metadata.filter((m) => CDK_NAG_MESSAGE_TYPES_SET.has(m.type));
        results.length > 0 &&
            this.app.addNagResult({
                resource: node.node.path,
                messages: results.map((m) => ({
                    messageDescription: m.data,
                    messageType: m.type,
                })),
            });
    }
}
/**
 * Helper for create a Nag Enabled App.
 */
class PDKNag {
    /**
     * Returns an instance of an App with Nag enabled.
     *
     * @param props props to initialize the app with.
     */
    static app(props) {
        return new PDKNagApp(props);
    }
    /**
     * Wrapper around NagSuppressions which does not throw.
     *
     * @param stack stack instance
     * @param path resource path
     * @param suppressions list of suppressions to apply.
     * @param applyToChildren whether to apply to children.
     */
    static addResourceSuppressionsByPathNoThrow(stack, path, suppressions, applyToChildren = false) {
        try {
            cdk_nag_1.NagSuppressions.addResourceSuppressionsByPath(stack, path, suppressions, applyToChildren);
        }
        catch (e) {
            // Do Nothing
        }
    }
    /**
     * Returns a prefix comprising of a delimited set of Stack Ids.
     *
     * For example: StackA/NestedStackB/
     *
     * @param stack stack instance.
     */
    static getStackPrefix(stack) {
        if (stack.nested) {
            return `${PDKNag.getStackPrefix(stack.nestedStackParent)}${stack.node.id}/`;
        }
        else {
            const stageName = aws_cdk_lib_1.Stage.of(stack)?.stageName;
            const stagePrefix = stageName && `${stageName}-`;
            let stackName = stack.stackName;
            stackName =
                stagePrefix && stackName.startsWith(stagePrefix)
                    ? `${stageName}/${stackName.slice(stagePrefix.length)}`
                    : stackName;
            return `${stackName}/`;
        }
    }
    /**
     * Returns a stack partition regex.
     *
     * @param stack stack instance.
     */
    static getStackPartitionRegex(stack) {
        if (stack.nested) {
            return PDKNag.getStackPartitionRegex(stack.nestedStackParent);
        }
        else {
            return stack.partition.startsWith("${Token")
                ? "<AWS::Partition>"
                : `(<AWS::Partition>|${stack.partition})`;
        }
    }
    /**
     * Returns a stack region regex.
     *
     * @param stack stack instance.
     */
    static getStackRegionRegex(stack) {
        if (stack.nested) {
            return PDKNag.getStackRegionRegex(stack.nestedStackParent);
        }
        else {
            return stack.region.startsWith("${Token")
                ? "<AWS::Region>"
                : `(<AWS::Region>|${stack.region})`;
        }
    }
    /**
     * Returns a stack account regex.
     *
     * @param stack stack instance.
     */
    static getStackAccountRegex(stack) {
        if (stack.nested) {
            return PDKNag.getStackAccountRegex(stack.nestedStackParent);
        }
        else {
            return stack.account.startsWith("${Token")
                ? "<AWS::AccountId>"
                : `(<AWS::AccountId>|${stack.account})`;
        }
    }
}
exports.PDKNag = PDKNag;
_b = JSII_RTTI_SYMBOL_1;
PDKNag[_b] = { fqn: "@aws/pdk.pdk_nag.PDKNag", version: "0.23.9" };
//# sourceMappingURL=data:application/json;base64,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