"use strict";
var __createBinding = (this && this.__createBinding) || (Object.create ? (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    var desc = Object.getOwnPropertyDescriptor(m, k);
    if (!desc || ("get" in desc ? !m.__esModule : desc.writable || desc.configurable)) {
      desc = { enumerable: true, get: function() { return m[k]; } };
    }
    Object.defineProperty(o, k2, desc);
}) : (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    o[k2] = m[k];
}));
var __exportStar = (this && this.__exportStar) || function(m, exports) {
    for (var p in m) if (p !== "default" && !Object.prototype.hasOwnProperty.call(exports, p)) __createBinding(exports, m, p);
};
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.PDKPipeline = exports.DEFAULT_BRANCH_NAME = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_codecommit_1 = require("aws-cdk-lib/aws-codecommit");
const aws_codepipeline_1 = require("aws-cdk-lib/aws-codepipeline");
const aws_kms_1 = require("aws-cdk-lib/aws-kms");
const aws_s3_1 = require("aws-cdk-lib/aws-s3");
const pipelines_1 = require("aws-cdk-lib/pipelines");
const cdk_nag_1 = require("cdk-nag");
const constructs_1 = require("constructs");
const sonar_code_scanner_1 = require("./code_scanner/sonar-code-scanner");
const feature_branches_1 = require("./feature-branches");
__exportStar(require("./code_scanner/sonar-code-scanner"), exports);
exports.DEFAULT_BRANCH_NAME = "mainline";
/**
 * An extension to CodePipeline which configures sane defaults for a NX Monorepo
 * codebase. In addition to this, it also creates a CodeCommit repository with
 * automated PR builds and approvals.
 */
class PDKPipeline extends constructs_1.Construct {
    /**
     * A helper function to normalize the branch name with only alphanumeric characters and hypens ('-').
     * @param branchName The name of the branch to normalize.
     * @returns The normalized branch name.
     */
    static normalizeBranchName(branchName) {
        return branchName.replace(/[^a-zA-Z0-9-]/g, "-");
    }
    /**
     * A helper function to determine if the current branch is the default branch.
     *
     * If there is no BRANCH environment variable, then assume this is the default
     * branch. Otherwise, check that BRANCH matches the default branch name.
     *
     * The default branch name is determined in the following priority:
     *
     * 1. defaultBranchName property
     * 2. defaultBranchName context
     * 3. PDKPipeline.defaultBranchName constant
     *
     * @param props? {
     *    defaultBranchName? Specify the default branch name without context.
     *    node? The current app to fetch defaultBranchName from context.
     * }
     * @returns True if the current branch is the default branch.
     */
    static isDefaultBranch(props = {
        defaultBranchName: undefined,
        node: undefined,
    }) {
        if (!process.env.BRANCH) {
            return true;
        }
        const defaultBranchName = props.defaultBranchName ||
            (props.node && props.node.tryGetContext("defaultBranchName")) ||
            PDKPipeline.defaultBranchName;
        return defaultBranchName === process.env.BRANCH;
    }
    /**
     * A helper function to create a branch prefix. The prefix is empty on the default branch.
     * @param props? {
     *    defaultBranchName? Specify the default branch name without context.
     *    node? The current app to fetch defaultBranchName from context.
     * }
     * @returns The branch prefix.
     */
    static getBranchPrefix(props = {
        defaultBranchName: undefined,
        node: undefined,
    }) {
        return PDKPipeline.isDefaultBranch(props)
            ? ""
            : PDKPipeline.normalizeBranchName(process.env.BRANCH) + "-";
    }
    constructor(scope, id, props) {
        super(scope, id);
        this.node.setContext("@aws-cdk/aws-s3:serverAccessLogsUseBucketPolicy", true);
        let codeRepository;
        // process.env.BRANCH is set only in CodeBuild builds
        if (PDKPipeline.isDefaultBranch({
            node: this.node,
            defaultBranchName: props.defaultBranchName,
        })) {
            // In the default branch, create a CodeCommit repository
            codeRepository = new aws_codecommit_1.Repository(this, "CodeRepository", {
                repositoryName: props.repositoryName,
            });
            codeRepository.applyRemovalPolicy(props.codeCommitRemovalPolicy ?? aws_cdk_lib_1.RemovalPolicy.RETAIN);
        }
        else {
            // In a non-default branch, use an existing CodeCommit repository
            codeRepository = aws_codecommit_1.Repository.fromRepositoryName(scope, "CodeRepository", props.repositoryName);
        }
        const accessLogsBucket = new aws_s3_1.Bucket(this, "AccessLogsBucket", {
            versioned: false,
            enforceSSL: true,
            autoDeleteObjects: true,
            removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
            encryption: aws_s3_1.BucketEncryption.S3_MANAGED,
            objectOwnership: aws_s3_1.ObjectOwnership.OBJECT_WRITER,
            publicReadAccess: false,
            blockPublicAccess: aws_s3_1.BlockPublicAccess.BLOCK_ALL,
        });
        const artifactBucket = new aws_s3_1.Bucket(this, "ArtifactsBucket", {
            enforceSSL: true,
            autoDeleteObjects: true,
            removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
            encryption: props.crossAccountKeys
                ? aws_s3_1.BucketEncryption.KMS
                : aws_s3_1.BucketEncryption.S3_MANAGED,
            encryptionKey: props.crossAccountKeys
                ? new aws_kms_1.Key(this, "ArtifactKey", {
                    enableKeyRotation: true,
                    removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
                })
                : undefined,
            objectOwnership: aws_s3_1.ObjectOwnership.BUCKET_OWNER_ENFORCED,
            publicReadAccess: false,
            blockPublicAccess: aws_s3_1.BlockPublicAccess.BLOCK_ALL,
            serverAccessLogsPrefix: "access-logs",
            serverAccessLogsBucket: accessLogsBucket,
        });
        const codePipeline = new aws_codepipeline_1.Pipeline(this, "CodePipeline", {
            enableKeyRotation: props.crossAccountKeys,
            restartExecutionOnUpdate: true,
            crossAccountKeys: props.crossAccountKeys,
            artifactBucket,
        });
        // ignore input and primaryOutputDirectory
        const { input, primaryOutputDirectory, commands, ...synthShellStepPartialProps } = props.synthShellStepPartialProps || {};
        const branch = process.env.BRANCH || props.defaultBranchName || exports.DEFAULT_BRANCH_NAME;
        const synthShellStep = new pipelines_1.ShellStep("Synth", {
            input: pipelines_1.CodePipelineSource.codeCommit(codeRepository, branch),
            env: props.branchNamePrefixes && props.branchNamePrefixes.length > 0
                ? {
                    BRANCH: branch,
                }
                : undefined,
            installCommands: ["npm install -g aws-cdk pnpm", "npx projen install"],
            commands: commands && commands.length > 0 ? commands : ["npx projen build"],
            primaryOutputDirectory: props.primarySynthDirectory,
            ...(synthShellStepPartialProps || {}),
        });
        synthShellStep.addOutputDirectory(".");
        const codePipelineProps = {
            codePipeline,
            ...props,
            crossAccountKeys: undefined,
            synth: synthShellStep,
        };
        this.codePipeline = new pipelines_1.CodePipeline(this, id, codePipelineProps);
        this.codeRepository = codeRepository;
        this.sonarCodeScannerConfig = props.sonarCodeScannerConfig
            ? {
                cdkOutDir: props.primarySynthDirectory,
                ...props.sonarCodeScannerConfig,
            }
            : undefined;
        this.branchNamePrefixes = props.branchNamePrefixes;
        this.defaultBranchName = props.defaultBranchName;
        this.repositoryName = props.repositoryName;
        if (props.branchNamePrefixes &&
            PDKPipeline.isDefaultBranch({
                node: this.node,
                defaultBranchName: props.defaultBranchName,
            })) {
            new feature_branches_1.FeatureBranches(this, "FeatureBranchPipelines", {
                codeRepository: this.codeRepository,
                cdkSrcDir: props.cdkSrcDir || path.dirname(props.primarySynthDirectory),
                synthShellStepPartialProps: props.synthShellStepPartialProps,
                cdkCommand: props.cdkCommand,
                branchNamePrefixes: props.branchNamePrefixes,
                defaultBranchName: props.defaultBranchName || exports.DEFAULT_BRANCH_NAME,
                codeBuildDefaults: props.codeBuildDefaults,
                dockerEnabledForSynth: props.dockerEnabledForSynth,
            });
        }
        else if (props.branchNamePrefixes) {
            aws_cdk_lib_1.Tags.of(aws_cdk_lib_1.Stack.of(this)).add("FeatureBranch", branch);
            aws_cdk_lib_1.Tags.of(aws_cdk_lib_1.Stack.of(this)).add("RepoName", this.repositoryName);
        }
        new aws_cdk_lib_1.CfnOutput(this, "CodeRepositoryGRCUrl", {
            value: this.codeRepository.repositoryCloneUrlGrc,
        });
    }
    /**
     * @inheritDoc
     */
    addStage(stage, options) {
        if (this.branchNamePrefixes &&
            !PDKPipeline.isDefaultBranch({
                node: stage.node,
                defaultBranchName: this.defaultBranchName,
            })) {
            aws_cdk_lib_1.Tags.of(stage).add("FeatureBranch", process.env.BRANCH);
            aws_cdk_lib_1.Tags.of(stage).add("RepoName", this.repositoryName);
        }
        // Add any root Aspects to the stage level as currently this doesn't happen automatically
        aws_cdk_lib_1.Aspects.of(stage.node.root).all.forEach((aspect) => aws_cdk_lib_1.Aspects.of(stage).add(aspect));
        return this.codePipeline.addStage(stage, options);
    }
    buildPipeline() {
        this.codePipeline.buildPipeline();
        this.sonarCodeScannerConfig &&
            new sonar_code_scanner_1.SonarCodeScanner(this, "SonarCodeScanner", {
                artifactBucketArn: this.codePipeline.pipeline.artifactBucket.bucketArn,
                artifactBucketKeyArn: this.codePipeline.pipeline.artifactBucket.encryptionKey?.keyArn,
                synthBuildArn: this.codePipeline.synthProject.projectArn,
                ...this.sonarCodeScannerConfig,
            });
        this.suppressCDKViolations();
    }
    suppressCDKViolations() {
        this.suppressRules(["AwsSolutions-IAM5", "AwsPrototyping-IAMNoWildcardPermissions"], "Wildcards are needed for dynamically created resources.");
        this.suppressRules([
            "AwsSolutions-CB4",
            "AwsPrototyping-CodeBuildProjectKMSEncryptedArtifacts",
        ], "Encryption of Codebuild is not required.");
        this.suppressRules(["AwsSolutions-S1", "AwsPrototyping-S3BucketLoggingEnabled"], "Access Log buckets should not have s3 bucket logging");
    }
    suppressRules(rules, reason) {
        cdk_nag_1.NagSuppressions.addResourceSuppressions(this, rules.map((r) => ({
            id: r,
            reason,
        })), true);
    }
}
exports.PDKPipeline = PDKPipeline;
_a = JSII_RTTI_SYMBOL_1;
PDKPipeline[_a] = { fqn: "@aws/pdk.pipeline.PDKPipeline", version: "0.23.9" };
PDKPipeline.ALL_BRANCHES = [""];
PDKPipeline.defaultBranchName = exports.DEFAULT_BRANCH_NAME;
//# sourceMappingURL=data:application/json;base64,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