"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const client_wafv2_1 = require("@aws-sdk/client-wafv2"); // eslint-disable-line
const DELIMITER = ":";
const SCOPE = "CLOUDFRONT";
const client = new client_wafv2_1.WAFV2({
    region: "us-east-1",
    customUserAgent: "aws-pdk/static-website/waf",
});
const MAX_CREATE_RETRY = 10;
const RETRY_INTERVAL = 200;
/**
 * Handler for creating a WAF V2 ACL in US-EAST-1.
 */
exports.onEvent = async (event) => {
    const { ID, MANAGED_RULES, CIDR_ALLOW_LIST } = event.ResourceProperties;
    const [WEB_ACL_ID, IP_SET_ID] = event.PhysicalResourceId
        ? event.PhysicalResourceId.split(DELIMITER)
        : [];
    let response = {};
    switch (event.RequestType) {
        case "Create":
            response = await createWaf(ID, MANAGED_RULES, CIDR_ALLOW_LIST);
            break;
        case "Update":
            response = await updateWaf(WEB_ACL_ID, IP_SET_ID, ID, getIpSetName(ID), MANAGED_RULES, CIDR_ALLOW_LIST);
            break;
        case "Delete":
            response = await deleteWaf(WEB_ACL_ID, IP_SET_ID, ID, getIpSetName(ID));
            break;
        default:
            throw new Error(`Invalid RequestType: ${event.RequestType}`);
    }
    return response;
};
/**
 * Generates the name of the IP Set.
 *
 * @param id param passed in.
 * @returns name of IP Set.
 */
const getIpSetName = (id) => `${id}-IPSet`;
/**
 * Returns a set of rules to apply.
 *
 * @param ipSetArn ip set arn
 * @param ipSetName  ip set name
 * @param managedRules  managed rules
 * @param cidrAllowList cidr allow list
 * @returns set of rules to apply.
 */
const getWafRules = (ipSetArn, ipSetName, managedRules, cidrAllowList) => {
    const rules = [];
    if (cidrAllowList) {
        rules.push({
            Name: ipSetName,
            Priority: 1,
            VisibilityConfig: {
                MetricName: ipSetName,
                CloudWatchMetricsEnabled: true,
                SampledRequestsEnabled: true,
            },
            Action: {
                Block: {},
            },
            Statement: {
                NotStatement: {
                    Statement: {
                        IPSetReferenceStatement: {
                            ARN: ipSetArn,
                        },
                    },
                },
            },
        });
    }
    if (managedRules) {
        rules.push(...managedRules
            .map((r) => ({ VendorName: r.vendor, Name: r.name }))
            .map((rule, Priority) => ({
            Name: `${rule.VendorName}-${rule.Name}`,
            Priority,
            Statement: { ManagedRuleGroupStatement: rule },
            OverrideAction: { None: {} },
            VisibilityConfig: {
                MetricName: `${rule.VendorName}-${rule.Name}`,
                CloudWatchMetricsEnabled: true,
                SampledRequestsEnabled: true,
            },
        })));
    }
    return rules;
};
const createWaf = async (id, managedRules, cidrAllowList) => {
    const ipSetName = getIpSetName(id);
    const createIpSetResponse = await client.createIPSet({
        Name: ipSetName,
        Scope: SCOPE,
        Addresses: cidrAllowList?.cidrRanges ?? [],
        IPAddressVersion: cidrAllowList?.cidrType ?? "IPV4",
    });
    const createWebAclResponse = await createWafAcl(id, ipSetName, createIpSetResponse, managedRules, cidrAllowList);
    return {
        PhysicalResourceId: `${createWebAclResponse.Summary?.Id}${DELIMITER}${createIpSetResponse.Summary?.Id}`,
        Data: {
            WebAclArn: createWebAclResponse.Summary?.ARN,
            WebAclId: createWebAclResponse.Summary?.Id,
            IPSetArn: createIpSetResponse.Summary?.ARN,
            IPSetId: createIpSetResponse.Summary?.Id,
        },
    };
};
const createWafAcl = async (id, ipSetName, createIpSetResponse, managedRules, cidrAllowList) => {
    let counter = 0;
    while (true) {
        try {
            const createWebAclResponse = await client.createWebACL({
                Name: id,
                DefaultAction: { Allow: {} },
                Scope: SCOPE,
                VisibilityConfig: {
                    CloudWatchMetricsEnabled: true,
                    MetricName: id,
                    SampledRequestsEnabled: true,
                },
                Rules: getWafRules(createIpSetResponse.Summary.ARN, ipSetName, managedRules, cidrAllowList),
            });
            return createWebAclResponse;
        }
        catch (e) {
            if (e instanceof client_wafv2_1.WAFUnavailableEntityException &&
                counter < MAX_CREATE_RETRY) {
                counter++;
                console.log(`Received error: ${e.message}; Waiting for retrying ${counter}`);
                await sleep(RETRY_INTERVAL);
                continue;
            }
            throw e;
        }
    }
};
const updateWaf = async (webAclId, ipSetId, id, ipSetName, managedRules, cidrAllowList) => {
    const getIpSetResponse = await client.getIPSet({
        Id: ipSetId,
        Name: ipSetName,
        Scope: SCOPE,
    });
    await client.updateIPSet({
        Id: ipSetId,
        Name: ipSetName,
        Addresses: cidrAllowList?.cidrRanges ?? [],
        Scope: SCOPE,
        LockToken: getIpSetResponse.LockToken,
    });
    const getWebAclResponse = await client.getWebACL({
        Id: webAclId,
        Name: id,
        Scope: SCOPE,
    });
    await client.updateWebACL({
        Name: id,
        DefaultAction: { Allow: {} },
        Scope: SCOPE,
        VisibilityConfig: {
            CloudWatchMetricsEnabled: true,
            MetricName: id,
            SampledRequestsEnabled: true,
        },
        Rules: getWafRules(getIpSetResponse.IPSet?.ARN, ipSetName, managedRules, cidrAllowList),
        Id: getWebAclResponse.WebACL?.Id,
        LockToken: getWebAclResponse.LockToken,
    });
    return {
        Data: {
            WebAclArn: getWebAclResponse.WebACL?.ARN,
            WebAclId: getWebAclResponse.WebACL?.Id,
            IPSetArn: getIpSetResponse.IPSet?.ARN,
            IPSetId: getIpSetResponse.IPSet?.Id,
        },
    };
};
const deleteWaf = async (webAclId, ipSetId, id, ipSetName) => {
    const getWebAclResponse = await client.getWebACL({
        Id: webAclId,
        Name: id,
        Scope: SCOPE,
    });
    await client.deleteWebACL({
        Id: webAclId,
        Name: id,
        Scope: SCOPE,
        LockToken: getWebAclResponse.LockToken,
    });
    const getIpSetResponse = await client.getIPSet({
        Id: ipSetId,
        Name: ipSetName,
        Scope: SCOPE,
    });
    await client.deleteIPSet({
        Id: ipSetId,
        Name: ipSetName,
        Scope: SCOPE,
        LockToken: getIpSetResponse.LockToken,
    });
    return {
        Data: {
            WebAclArn: getWebAclResponse.WebACL?.ARN,
            WebAclId: getWebAclResponse.WebACL?.Id,
            IPSetArn: getIpSetResponse.IPSet?.ARN,
            IPSetId: getIpSetResponse.IPSet?.Id,
        },
    };
};
const sleep = async (duration) => {
    return new Promise((resolve) => setTimeout(resolve, duration));
};
//# sourceMappingURL=data:application/json;base64,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