"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.S3Integration = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const integration_1 = require("./integration");
const integration_response_sets_1 = require("./integration-response-sets");
const utils_1 = require("../spec/utils");
/**
 * An S3 integration
 * @see https://docs.aws.amazon.com/apigateway/latest/developerguide/integrating-api-with-aws-services-s3.html
 */
class S3Integration extends integration_1.Integration {
    constructor(props) {
        super();
        this.executionRoleId = "S3IntegrationsExecutionRole";
        this.bucket = props.bucket;
        this.method = props.method;
        this.path = props.path;
        this.integrationResponseSet = props.integrationResponseSet;
    }
    isRole(construct) {
        return "roleArn" in construct && "grantPrincipal" in construct;
    }
    executionRole(scope) {
        // Retrieve or create the shared S3 execution role
        const existingExecutionRole = scope.node.tryFindChild(this.executionRoleId);
        if (existingExecutionRole) {
            if (this.isRole(existingExecutionRole)) {
                return existingExecutionRole;
            }
            throw new Error(`Found construct with ID ${this.executionRoleId} in API scope which was not a role`);
        }
        return new aws_iam_1.Role(scope, this.executionRoleId, {
            assumedBy: new aws_iam_1.ServicePrincipal("apigateway.amazonaws.com"),
        });
    }
    /**
     * Render the S3 integration as a snippet of OpenAPI
     */
    render(props) {
        return {
            type: "AWS",
            httpMethod: (this.method ?? props.method).toUpperCase(),
            uri: (0, utils_1.bucketInvocationUri)(this.bucket, this.path ?? props.path),
            credentials: this.executionRole(props.scope).roleArn,
            requestParameters: {
                // Add every path parameter to the integration request
                ...Object.fromEntries([...props.path.matchAll(/\{([^\}]*)\}/g)]
                    .map((m) => m[1])
                    .map((param) => [
                    `integration.request.path.${param}`,
                    `method.request.path.${param}`,
                ])),
            },
            responses: {
                ...(this.integrationResponseSet ??
                    integration_response_sets_1.IntegrationResponseSets.composite(integration_response_sets_1.IntegrationResponseSets.defaultPassthrough(), integration_response_sets_1.IntegrationResponseSets.s3JsonErrorMessage())).render(props),
            },
        };
    }
    /**
     * Grant API Gateway permissions to invoke the S3 bucket
     */
    grant({ scope, method, path }) {
        const executionRole = this.executionRole(scope);
        // Replace path parameters with * to grant access to arbitrary values for path parameters
        const permissionPath = (this.path ?? path).replace(/{[^\}]*\}/g, "*");
        // Grant read access for GET/HEAD/OPTIONS/TRACE, otherwise write
        if (["get", "head", "options", "trace"].includes((this.method ?? method).toLowerCase())) {
            this.bucket.grantRead(executionRole, permissionPath);
        }
        else {
            this.bucket.grantWrite(executionRole, permissionPath);
        }
    }
}
exports.S3Integration = S3Integration;
_a = JSII_RTTI_SYMBOL_1;
S3Integration[_a] = { fqn: "@aws/pdk.type_safe_api.S3Integration", version: "0.23.9" };
//# sourceMappingURL=data:application/json;base64,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