"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.prepareSecuritySchemes = exports.getAllAuthorizers = exports.serializeAsAuthorizerReference = void 0;
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const aws_apigateway_1 = require("aws-cdk-lib/aws-apigateway");
const utils_1 = require("./utils");
const authorizers_1 = require("../authorizers");
const predicates_1 = require("../authorizers/predicates");
const constants_1 = require("../prepare-spec-event-handler/constants");
// Regex to match against a single header identity source
const SINGLE_HEADER_IDENTITY_SOURCE_REGEX = /^method.request.header.(?<header>[^\.\s,]+)$/;
/**
 * Build a serialized reference to an authorizer for use in an api method
 * @param authorizer the author to serialize
 */
const serializeAsAuthorizerReference = (authorizer) => authorizer
    ? {
        authorizerId: authorizer.authorizerId,
        authorizationScopes: authorizer.authorizationScopes,
    }
    : undefined;
exports.serializeAsAuthorizerReference = serializeAsAuthorizerReference;
/**
 * Create an OpenAPI security scheme definition for an iam authorizer
 * @see https://docs.amazonaws.cn/en_us/apigateway/latest/developerguide/api-gateway-swagger-extensions-authtype.html
 */
const iamSecurityScheme = () => ({
    type: "apiKey",
    name: "Authorization",
    in: "header",
    "x-amazon-apigateway-authtype": "awsSigv4",
});
/**
 * Create an API key security scheme
 * @see https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-api-key-source.html
 */
const apiKeySecurityScheme = () => ({
    type: "apiKey",
    // Header must be x-api-key for API Gateway
    // See: https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-api-key-source.html
    name: "x-api-key",
    in: "header",
});
/**
 * Create an OpenAPI security scheme definition for a cognito authorizer
 * @param authorizer cognito authorizer
 */
const cognitoSecurityScheme = (authorizer) => ({
    type: "apiKey",
    name: "Authorization",
    in: "header",
    "x-amazon-apigateway-authtype": authorizer.authorizationType,
    "x-amazon-apigateway-authorizer": {
        type: aws_apigateway_1.AuthorizationType.COGNITO,
        providerARNs: authorizer.userPools.map((pool) => pool.userPoolArn),
    },
});
/**
 * Create an OpenAPI security scheme definition for a custom authorizer
 * @param scope the scope in which the api construct is defined
 * @param authorizer custom authorizer
 */
const customSecurityScheme = (_scope, authorizer) => {
    const singleHeaderMatch = authorizer.identitySource.match(SINGLE_HEADER_IDENTITY_SOURCE_REGEX);
    const singleHeader = singleHeaderMatch
        ? singleHeaderMatch.groups.header
        : undefined;
    // Token type must use a single header
    if (!singleHeader && authorizer.type === authorizers_1.CustomAuthorizerType.TOKEN) {
        throw new Error(`identitySource must be a single header for a ${authorizers_1.CustomAuthorizerType.TOKEN} authorizer, eg method.request.header.Authorization`);
    }
    return {
        type: "apiKey",
        in: "header",
        // When the identity source is not a single header, the value must be "Unused"
        // https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-swagger-extensions-authorizer.html
        name: singleHeader ? singleHeader : "Unused",
        "x-amazon-apigateway-authtype": authorizer.authorizationType,
        "x-amazon-apigateway-authorizer": {
            type: authorizer.type,
            authorizerUri: (0, utils_1.functionInvocationUri)(authorizer.function),
            authorizerResultTtlInSeconds: authorizer.authorizerResultTtlInSeconds,
            identitySource: authorizer.identitySource,
        },
    };
};
/**
 * Return a list of all unique authorizers used in the api
 */
const getAllAuthorizers = (integrations, defaultAuthorizer) => Object.values(Object.fromEntries([
    ...(defaultAuthorizer ? [defaultAuthorizer] : []),
    ...Object.values(integrations).flatMap(({ authorizer }) => authorizer ? [authorizer] : []),
].map((a) => [a.authorizerId, a])));
exports.getAllAuthorizers = getAllAuthorizers;
/**
 * Generate the security schemes section of an OpenAPI specification
 */
const prepareSecuritySchemes = (scope, integrations, defaultAuthorizer, apiKeyOptions) => {
    // All the defined authorizers
    const allAuthorizers = (0, exports.getAllAuthorizers)(integrations, defaultAuthorizer);
    // Cognito, IAM and custom authorizers must be declared in security schemes
    return {
        ...Object.fromEntries(allAuthorizers
            .filter((authorizer) => (0, predicates_1.isCognitoAuthorizer)(authorizer))
            .map((authorizer) => [
            authorizer.authorizerId,
            cognitoSecurityScheme(authorizer),
        ])),
        ...Object.fromEntries(allAuthorizers
            .filter((authorizer) => (0, predicates_1.isCustomAuthorizer)(authorizer))
            .map((authorizer) => [
            authorizer.authorizerId,
            customSecurityScheme(scope, authorizer),
        ])),
        ...Object.fromEntries(allAuthorizers
            .filter((authorizer) => (0, predicates_1.isIamAuthorizer)(authorizer))
            .map((authorizer) => [authorizer.authorizerId, iamSecurityScheme()])),
        // A security scheme must be defined if the api key is in the header
        ...(apiKeyOptions?.source === aws_apigateway_1.ApiKeySourceType.HEADER
            ? {
                [constants_1.DefaultAuthorizerIds.API_KEY]: apiKeySecurityScheme(),
            }
            : {}),
    };
};
exports.prepareSecuritySchemes = prepareSecuritySchemes;
//# sourceMappingURL=data:application/json;base64,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