"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.TypeSafeRestApi = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const fs = require("fs");
const path = require("path");
const pdk_nag_1 = require("../../pdk-nag");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_apigateway_1 = require("aws-cdk-lib/aws-apigateway");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_lambda_1 = require("aws-cdk-lib/aws-lambda");
const aws_logs_1 = require("aws-cdk-lib/aws-logs");
const aws_s3_assets_1 = require("aws-cdk-lib/aws-s3-assets");
const custom_resources_1 = require("aws-cdk-lib/custom-resources");
const cdk_nag_1 = require("cdk-nag");
const constructs_1 = require("constructs");
const prepare_spec_1 = require("./prepare-spec-event-handler/prepare-spec");
const api_gateway_auth_1 = require("./spec/api-gateway-auth");
const api_gateway_integrations_1 = require("./spec/api-gateway-integrations");
const open_api_gateway_web_acl_1 = require("./waf/open-api-gateway-web-acl");
/**
 * A construct for creating an api gateway rest api based on the definition in the OpenAPI spec.
 */
class TypeSafeRestApi extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const { integrations, specPath, operationLookup, defaultAuthorizer, corsOptions, ...options } = props;
        // Upload the spec to s3 as an asset
        const inputSpecAsset = new aws_s3_assets_1.Asset(this, "InputSpec", {
            path: specPath,
        });
        // We'll output the prepared spec in the same asset bucket
        const preparedSpecOutputKeyPrefix = `${inputSpecAsset.s3ObjectKey}-prepared`;
        const stack = aws_cdk_lib_1.Stack.of(this);
        // Lambda name prefix is truncated to 48 characters (16 below the max of 64)
        const lambdaNamePrefix = `${pdk_nag_1.PDKNag.getStackPrefix(stack)
            .split("/")
            .join("-")
            .slice(0, 40)}${this.node.addr.slice(-8).toUpperCase()}`;
        const prepareSpecLambdaName = `${lambdaNamePrefix}PrepSpec`;
        const prepareSpecRole = new aws_iam_1.Role(this, "PrepareSpecRole", {
            assumedBy: new aws_iam_1.ServicePrincipal("lambda.amazonaws.com"),
            inlinePolicies: {
                logs: new aws_iam_1.PolicyDocument({
                    statements: [
                        new aws_iam_1.PolicyStatement({
                            effect: aws_iam_1.Effect.ALLOW,
                            actions: [
                                "logs:CreateLogGroup",
                                "logs:CreateLogStream",
                                "logs:PutLogEvents",
                            ],
                            resources: [
                                `arn:aws:logs:${stack.region}:${stack.account}:log-group:/aws/lambda/${prepareSpecLambdaName}`,
                                `arn:aws:logs:${stack.region}:${stack.account}:log-group:/aws/lambda/${prepareSpecLambdaName}:*`,
                            ],
                        }),
                    ],
                }),
                s3: new aws_iam_1.PolicyDocument({
                    statements: [
                        new aws_iam_1.PolicyStatement({
                            effect: aws_iam_1.Effect.ALLOW,
                            actions: ["s3:getObject"],
                            resources: [
                                inputSpecAsset.bucket.arnForObjects(inputSpecAsset.s3ObjectKey),
                            ],
                        }),
                        new aws_iam_1.PolicyStatement({
                            effect: aws_iam_1.Effect.ALLOW,
                            actions: ["s3:putObject"],
                            resources: [
                                // The output file will include a hash of the prepared spec, which is not known until deploy time since
                                // tokens must be resolved
                                inputSpecAsset.bucket.arnForObjects(`${preparedSpecOutputKeyPrefix}/*`),
                            ],
                        }),
                    ],
                }),
            },
        });
        ["AwsSolutions-IAM5", "AwsPrototyping-IAMNoWildcardPermissions"].forEach((RuleId) => {
            cdk_nag_1.NagSuppressions.addResourceSuppressions(prepareSpecRole, [
                {
                    id: RuleId,
                    reason: "Cloudwatch resources have been scoped down to the LogGroup level, however * is still needed as stream names are created just in time.",
                    appliesTo: [
                        {
                            regex: `/^Resource::arn:aws:logs:${pdk_nag_1.PDKNag.getStackRegionRegex(stack)}:${pdk_nag_1.PDKNag.getStackAccountRegex(stack)}:log-group:/aws/lambda/${prepareSpecLambdaName}:\*/g`,
                        },
                    ],
                },
                {
                    id: RuleId,
                    reason: "S3 resources have been scoped down to the appropriate prefix in the CDK asset bucket, however * is still needed as since the prepared spec hash is not known until deploy time.",
                    appliesTo: [
                        {
                            regex: `/^Resource::arn:${pdk_nag_1.PDKNag.getStackPartitionRegex(stack)}:s3:.*/${preparedSpecOutputKeyPrefix}/\*/g`,
                        },
                    ],
                },
            ], true);
        });
        // Create a custom resource for preparing the spec for deployment (adding integrations, authorizers, etc)
        const prepareSpec = new aws_lambda_1.Function(this, "PrepareSpecHandler", {
            handler: "index.handler",
            runtime: aws_lambda_1.Runtime.NODEJS_18_X,
            code: aws_lambda_1.Code.fromAsset(path.join(__dirname, "./prepare-spec-event-handler")),
            timeout: aws_cdk_lib_1.Duration.seconds(30),
            role: prepareSpecRole,
            functionName: prepareSpecLambdaName,
        });
        const providerFunctionName = `${lambdaNamePrefix}PrepSpecProvider`;
        const providerRole = new aws_iam_1.Role(this, "PrepareSpecProviderRole", {
            assumedBy: new aws_iam_1.ServicePrincipal("lambda.amazonaws.com"),
            inlinePolicies: {
                logs: new aws_iam_1.PolicyDocument({
                    statements: [
                        new aws_iam_1.PolicyStatement({
                            effect: aws_iam_1.Effect.ALLOW,
                            actions: [
                                "logs:CreateLogGroup",
                                "logs:CreateLogStream",
                                "logs:PutLogEvents",
                            ],
                            resources: [
                                `arn:aws:logs:${stack.region}:${stack.account}:log-group:/aws/lambda/${providerFunctionName}`,
                                `arn:aws:logs:${stack.region}:${stack.account}:log-group:/aws/lambda/${providerFunctionName}:*`,
                            ],
                        }),
                    ],
                }),
            },
        });
        const provider = new custom_resources_1.Provider(this, "PrepareSpecProvider", {
            onEventHandler: prepareSpec,
            role: providerRole,
            providerFunctionName,
        });
        ["AwsSolutions-IAM5", "AwsPrototyping-IAMNoWildcardPermissions"].forEach((RuleId) => {
            cdk_nag_1.NagSuppressions.addResourceSuppressions(providerRole, [
                {
                    id: RuleId,
                    reason: "Cloudwatch resources have been scoped down to the LogGroup level, however * is still needed as stream names are created just in time.",
                },
            ], true);
        });
        ["AwsSolutions-L1", "AwsPrototyping-LambdaLatestVersion"].forEach((RuleId) => {
            cdk_nag_1.NagSuppressions.addResourceSuppressions(provider, [
                {
                    id: RuleId,
                    reason: "Latest runtime cannot be configured. CDK will need to upgrade the Provider construct accordingly.",
                },
            ], true);
        });
        const serializedCorsOptions = corsOptions && {
            allowHeaders: corsOptions.allowHeaders || [
                ...aws_apigateway_1.Cors.DEFAULT_HEADERS,
                "x-amz-content-sha256",
            ],
            allowMethods: corsOptions.allowMethods || aws_apigateway_1.Cors.ALL_METHODS,
            allowOrigins: corsOptions.allowOrigins,
            statusCode: corsOptions.statusCode || 204,
        };
        const prepareSpecOptions = {
            defaultAuthorizerReference: (0, api_gateway_auth_1.serializeAsAuthorizerReference)(defaultAuthorizer),
            integrations: Object.fromEntries(Object.entries(integrations).map(([operationId, integration]) => [
                operationId,
                {
                    integration: integration.integration.render({
                        operationId,
                        scope: this,
                        ...operationLookup[operationId],
                        corsOptions: serializedCorsOptions,
                        operationLookup,
                    }),
                    methodAuthorizer: (0, api_gateway_auth_1.serializeAsAuthorizerReference)(integration.authorizer),
                    options: integration.options,
                },
            ])),
            securitySchemes: (0, api_gateway_auth_1.prepareSecuritySchemes)(this, integrations, defaultAuthorizer, options.apiKeyOptions),
            corsOptions: serializedCorsOptions,
            operationLookup,
            apiKeyOptions: options.apiKeyOptions,
        };
        // Spec preparation will happen in a custom resource lambda so that references to lambda integrations etc can be
        // resolved. However, we also prepare inline to perform some additional validation at synth time.
        const spec = JSON.parse(fs.readFileSync(specPath, "utf-8"));
        this.extendedApiSpecification = (0, prepare_spec_1.prepareApiSpec)(spec, prepareSpecOptions);
        const prepareApiSpecCustomResourceProperties = {
            inputSpecLocation: {
                bucket: inputSpecAsset.bucket.bucketName,
                key: inputSpecAsset.s3ObjectKey,
            },
            outputSpecLocation: {
                bucket: inputSpecAsset.bucket.bucketName,
                key: preparedSpecOutputKeyPrefix,
            },
            ...prepareSpecOptions,
        };
        const prepareSpecCustomResource = new aws_cdk_lib_1.CustomResource(this, "PrepareSpecCustomResource", {
            serviceToken: provider.serviceToken,
            properties: prepareApiSpecCustomResourceProperties,
        });
        // Create the api gateway resources from the spec, augmenting the spec with the properties specific to api gateway
        // such as integrations or auth types
        this.api = new aws_apigateway_1.SpecRestApi(this, id, {
            apiDefinition: this.node.tryGetContext("type-safe-api-local")
                ? aws_apigateway_1.ApiDefinition.fromInline(this.extendedApiSpecification)
                : aws_apigateway_1.ApiDefinition.fromBucket(inputSpecAsset.bucket, prepareSpecCustomResource.getAttString("outputSpecKey")),
            deployOptions: {
                accessLogDestination: new aws_apigateway_1.LogGroupLogDestination(new aws_logs_1.LogGroup(this, `AccessLogs`)),
                accessLogFormat: aws_apigateway_1.AccessLogFormat.clf(),
                loggingLevel: aws_apigateway_1.MethodLoggingLevel.INFO,
            },
            ...options,
        });
        this.api.node.addDependency(prepareSpecCustomResource);
        // While the api will be updated when the output path from the custom resource changes, CDK still needs to know when
        // to redeploy the api. This is achieved by including a hash of the spec in the logical id (internalised in the
        // addToLogicalId method since this is how changes of individual resources/methods etc trigger redeployments in CDK)
        this.api.latestDeployment?.addToLogicalId(this.extendedApiSpecification);
        // Grant API Gateway permission to invoke the integrations
        Object.keys(integrations).forEach((operationId) => {
            integrations[operationId].integration.grant({
                operationId,
                scope: this,
                api: this.api,
                ...operationLookup[operationId],
                operationLookup,
            });
        });
        // Grant API Gateway permission to invoke each custom authorizer lambda (if any)
        (0, api_gateway_integrations_1.getAuthorizerFunctions)(props).forEach(({ label, function: lambda }) => {
            new aws_lambda_1.CfnPermission(this, `LambdaPermission-${label}`, {
                action: "lambda:InvokeFunction",
                principal: "apigateway.amazonaws.com",
                functionName: lambda.functionArn,
                sourceArn: stack.formatArn({
                    service: "execute-api",
                    resource: this.api.restApiId,
                    resourceName: "*/*",
                }),
            });
        });
        // Create and associate the web acl if not disabled
        if (!props.webAclOptions?.disable) {
            const acl = new open_api_gateway_web_acl_1.OpenApiGatewayWebAcl(this, `${id}-Acl`, {
                ...props.webAclOptions,
                apiDeploymentStageArn: this.api.deploymentStage.stageArn,
            });
            this.webAcl = acl.webAcl;
            this.ipSet = acl.ipSet;
            this.webAclAssociation = acl.webAclAssociation;
        }
        ["AwsSolutions-IAM4", "AwsPrototyping-IAMNoManagedPolicies"].forEach((RuleId) => {
            cdk_nag_1.NagSuppressions.addResourceSuppressions(this, [
                {
                    id: RuleId,
                    reason: "Cloudwatch Role requires access to create/read groups at the root level.",
                    appliesTo: [
                        {
                            regex: `/^Policy::arn:${pdk_nag_1.PDKNag.getStackPartitionRegex(stack)}:iam::aws:policy/service-role/AmazonAPIGatewayPushToCloudWatchLogs$/g`,
                        },
                    ],
                },
            ], true);
        });
        ["AwsSolutions-APIG2", "AwsPrototyping-APIGWRequestValidation"].forEach((RuleId) => {
            cdk_nag_1.NagSuppressions.addResourceSuppressions(this, [
                {
                    id: RuleId,
                    reason: "This construct implements fine grained validation via OpenApi.",
                },
            ], true);
        });
    }
}
exports.TypeSafeRestApi = TypeSafeRestApi;
_a = JSII_RTTI_SYMBOL_1;
TypeSafeRestApi[_a] = { fqn: "@aws/pdk.type_safe_api.TypeSafeRestApi", version: "0.23.9" };
//# sourceMappingURL=data:application/json;base64,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