"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getHandlersProjectVendorExtensions = exports.buildInvokeMockDataGeneratorCommand = exports.buildCleanOpenApiGeneratedCodeCommand = exports.buildInvokeOpenApiGeneratorCommandArgs = exports.buildTypeSafeApiExecCommand = exports.getTypeSafeApiTaskEnvironment = exports.TypeSafeApiScript = exports.OtherGenerators = void 0;
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const path = require("path");
const readPkg = require("read-pkg-up");
const runtime_version_utils_1 = require("../runtime-version-utils");
/**
 * Enum for generator directories for non-runtime generators
 */
var OtherGenerators;
(function (OtherGenerators) {
    OtherGenerators["DOCS"] = "docs";
    // Infrastructure
    OtherGenerators["TYPESCRIPT_CDK_INFRASTRUCTURE"] = "typescript-cdk-infrastructure";
    OtherGenerators["PYTHON_CDK_INFRASTRUCTURE"] = "python-cdk-infrastructure";
    OtherGenerators["JAVA_CDK_INFRASTRUCTURE"] = "java-cdk-infrastructure";
    // Handlers
    OtherGenerators["TYPESCRIPT_LAMBDA_HANDLERS"] = "typescript-lambda-handlers";
    OtherGenerators["PYTHON_LAMBDA_HANDLERS"] = "python-lambda-handlers";
    OtherGenerators["JAVA_LAMBDA_HANDLERS"] = "java-lambda-handlers";
})(OtherGenerators || (exports.OtherGenerators = OtherGenerators = {}));
var TypeSafeApiScript;
(function (TypeSafeApiScript) {
    TypeSafeApiScript["PARSE_OPENAPI_SPEC"] = "type-safe-api.parse-openapi-spec";
    TypeSafeApiScript["GENERATE"] = "type-safe-api.generate";
    TypeSafeApiScript["GENERATE_MOCK_DATA"] = "type-safe-api.generate-mock-data";
    TypeSafeApiScript["GENERATE_HTML_REDOC_DOCS"] = "type-safe-api.generate-html-redoc-docs";
    TypeSafeApiScript["CLEAN_OPENAPI_GENERATED_CODE"] = "type-safe-api.clean-openapi-generated-code";
    TypeSafeApiScript["COPY_GRADLE_WRAPPER"] = "type-safe-api.copy-gradle-wrapper";
})(TypeSafeApiScript || (exports.TypeSafeApiScript = TypeSafeApiScript = {}));
/**
 * Get the current package version
 */
const getPackageVersion = () => {
    const { packageJson } = readPkg.sync({
        cwd: path.resolve(__dirname),
    });
    return packageJson.version;
};
/**
 * Return the environment that should be used for executing type safe api commands
 */
const getTypeSafeApiTaskEnvironment = () => ({
    AWS_PDK_VERSION: getPackageVersion(),
});
exports.getTypeSafeApiTaskEnvironment = getTypeSafeApiTaskEnvironment;
/**
 * Build a command for running a script from this project's bin
 */
const buildTypeSafeApiExecCommand = (script, args) => {
    return `npx --yes -p @aws/pdk@$AWS_PDK_VERSION ${script}${args ? ` ${args}` : ""}`;
};
exports.buildTypeSafeApiExecCommand = buildTypeSafeApiExecCommand;
const serializeProperties = (properties) => Object.entries(properties)
    .map(([key, value]) => `${key}=${value}`)
    .join(",");
/**
 * Generate code or docs by invoking the root generate script
 */
const buildInvokeOpenApiGeneratorCommandArgs = (options) => {
    const srcDir = options.srcDir ?? "src";
    const tstDir = options.tstDir ?? "test";
    const additionalProperties = options.additionalProperties
        ? ` --additional-properties "${serializeProperties(options.additionalProperties)}"`
        : "";
    const normalizers = options.normalizers
        ? ` --openapi-normalizer "${serializeProperties(Object.fromEntries(Object.entries(options.normalizers).map(([k, v]) => [k, `${v}`])))}"`
        : "";
    const extensions = options.extraVendorExtensions
        ? ` --extra-vendor-extensions '${JSON.stringify(options.extraVendorExtensions)}'`
        : "";
    const generateAliasAsModel = options.generateAliasAsModel ?? true ? " --generate-alias-as-model" : "";
    const specPath = options.specPath;
    const outputPath = ".";
    return `--generator ${options.generator} --spec-path ${specPath} --output-path ${outputPath} --generator-dir ${options.generatorDirectory} --src-dir ${srcDir} --tst-dir ${tstDir}${additionalProperties}${normalizers}${extensions}${generateAliasAsModel}`;
};
exports.buildInvokeOpenApiGeneratorCommandArgs = buildInvokeOpenApiGeneratorCommandArgs;
/**
 * Builds a command to clean up files which were previously generated by openapi generator
 */
const buildCleanOpenApiGeneratedCodeCommand = () => {
    return (0, exports.buildTypeSafeApiExecCommand)(TypeSafeApiScript.CLEAN_OPENAPI_GENERATED_CODE, `--code-path .`);
};
exports.buildCleanOpenApiGeneratedCodeCommand = buildCleanOpenApiGeneratedCodeCommand;
/**
 * Invoke the mock data generator script
 */
const buildInvokeMockDataGeneratorCommand = (options) => {
    const outputPath = options.outputSubDir ?? ".";
    const locale = options.locale ? ` --locale ${options.locale}` : "";
    const maxArrayLength = options.maxArrayLength !== undefined
        ? ` --max-array-length ${options.maxArrayLength}`
        : "";
    const seed = options.seed !== undefined ? ` --seed ${options.seed}` : "";
    return (0, exports.buildTypeSafeApiExecCommand)(TypeSafeApiScript.GENERATE_MOCK_DATA, `--spec-path ${options.specPath} --output-path ${outputPath}${locale}${maxArrayLength}${seed}`);
};
exports.buildInvokeMockDataGeneratorCommand = buildInvokeMockDataGeneratorCommand;
/**
 * Return vendor extensions containing details about the handler projects
 */
const getHandlersProjectVendorExtensions = (targetProject, { java, python, typescript }) => ({
    "x-handlers-python-module": python?.moduleName ?? "",
    "x-handlers-java-package": java?.packageName ?? "",
    "x-handlers-typescript-asset-path": typescript
        ? path.join(path.relative(targetProject.outdir, typescript.outdir), "dist", "lambda")
        : "",
    "x-handlers-python-asset-path": python
        ? path.join(path.relative(targetProject.outdir, python.outdir), "dist", "lambda")
        : "",
    "x-handlers-java-asset-path": java
        ? path.join(path.relative(targetProject.outdir, java.outdir), java.distdir, ...java.pom.groupId.split("."), java.pom.artifactId, java.pom.version, `${java.pom.artifactId}-${java.pom.version}.jar`)
        : "",
    "x-handlers-node-lambda-runtime-version": typescript
        ? runtime_version_utils_1.RuntimeVersionUtils.NODE.getLambdaRuntime(typescript.runtimeVersion)
        : "",
    "x-handlers-python-lambda-runtime-version": python
        ? runtime_version_utils_1.RuntimeVersionUtils.PYTHON.getLambdaRuntime(python.runtimeVersion)
        : "",
    "x-handlers-java-lambda-runtime-version": java
        ? runtime_version_utils_1.RuntimeVersionUtils.JAVA.getLambdaRuntime(java.runtimeVersion)
        : "",
});
exports.getHandlersProjectVendorExtensions = getHandlersProjectVendorExtensions;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidXRpbHMuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJ1dGlscy50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7QUFBQTtzQ0FDc0M7QUFDdEMsNkJBQTZCO0FBRTdCLHVDQUF1QztBQUl2QyxvRUFBK0Q7QUFFL0Q7O0dBRUc7QUFDSCxJQUFZLGVBVVg7QUFWRCxXQUFZLGVBQWU7SUFDekIsZ0NBQWEsQ0FBQTtJQUNiLGlCQUFpQjtJQUNqQixrRkFBK0QsQ0FBQTtJQUMvRCwwRUFBdUQsQ0FBQTtJQUN2RCxzRUFBbUQsQ0FBQTtJQUNuRCxXQUFXO0lBQ1gsNEVBQXlELENBQUE7SUFDekQsb0VBQWlELENBQUE7SUFDakQsZ0VBQTZDLENBQUE7QUFDL0MsQ0FBQyxFQVZXLGVBQWUsK0JBQWYsZUFBZSxRQVUxQjtBQUVELElBQVksaUJBT1g7QUFQRCxXQUFZLGlCQUFpQjtJQUMzQiw0RUFBdUQsQ0FBQTtJQUN2RCx3REFBbUMsQ0FBQTtJQUNuQyw0RUFBdUQsQ0FBQTtJQUN2RCx3RkFBbUUsQ0FBQTtJQUNuRSxnR0FBMkUsQ0FBQTtJQUMzRSw4RUFBeUQsQ0FBQTtBQUMzRCxDQUFDLEVBUFcsaUJBQWlCLGlDQUFqQixpQkFBaUIsUUFPNUI7QUFnRUQ7O0dBRUc7QUFDSCxNQUFNLGlCQUFpQixHQUFHLEdBQVcsRUFBRTtJQUNyQyxNQUFNLEVBQUUsV0FBVyxFQUFFLEdBQUcsT0FBTyxDQUFDLElBQUksQ0FBQztRQUNuQyxHQUFHLEVBQUUsSUFBSSxDQUFDLE9BQU8sQ0FBQyxTQUFTLENBQUM7S0FDN0IsQ0FBRSxDQUFDO0lBQ0osT0FBTyxXQUFXLENBQUMsT0FBTyxDQUFDO0FBQzdCLENBQUMsQ0FBQztBQUVGOztHQUVHO0FBQ0ksTUFBTSw2QkFBNkIsR0FBRyxHQUE4QixFQUFFLENBQUMsQ0FBQztJQUM3RSxlQUFlLEVBQUUsaUJBQWlCLEVBQUU7Q0FDckMsQ0FBQyxDQUFDO0FBRlUsUUFBQSw2QkFBNkIsaUNBRXZDO0FBRUg7O0dBRUc7QUFDSSxNQUFNLDJCQUEyQixHQUFHLENBQ3pDLE1BQXlCLEVBQ3pCLElBQWEsRUFDTCxFQUFFO0lBQ1YsT0FBTywwQ0FBMEMsTUFBTSxHQUNyRCxJQUFJLENBQUMsQ0FBQyxDQUFDLElBQUksSUFBSSxFQUFFLENBQUMsQ0FBQyxDQUFDLEVBQ3RCLEVBQUUsQ0FBQztBQUNMLENBQUMsQ0FBQztBQVBXLFFBQUEsMkJBQTJCLCtCQU90QztBQUVGLE1BQU0sbUJBQW1CLEdBQUcsQ0FBQyxVQUFxQyxFQUFFLEVBQUUsQ0FDcEUsTUFBTSxDQUFDLE9BQU8sQ0FBQyxVQUFVLENBQUM7S0FDdkIsR0FBRyxDQUFDLENBQUMsQ0FBQyxHQUFHLEVBQUUsS0FBSyxDQUFDLEVBQUUsRUFBRSxDQUFDLEdBQUcsR0FBRyxJQUFJLEtBQUssRUFBRSxDQUFDO0tBQ3hDLElBQUksQ0FBQyxHQUFHLENBQUMsQ0FBQztBQUVmOztHQUVHO0FBQ0ksTUFBTSxzQ0FBc0MsR0FBRyxDQUNwRCxPQUEwQixFQUNsQixFQUFFO0lBQ1YsTUFBTSxNQUFNLEdBQUcsT0FBTyxDQUFDLE1BQU0sSUFBSSxLQUFLLENBQUM7SUFDdkMsTUFBTSxNQUFNLEdBQUcsT0FBTyxDQUFDLE1BQU0sSUFBSSxNQUFNLENBQUM7SUFFeEMsTUFBTSxvQkFBb0IsR0FBRyxPQUFPLENBQUMsb0JBQW9CO1FBQ3ZELENBQUMsQ0FBQyw2QkFBNkIsbUJBQW1CLENBQzlDLE9BQU8sQ0FBQyxvQkFBb0IsQ0FDN0IsR0FBRztRQUNOLENBQUMsQ0FBQyxFQUFFLENBQUM7SUFFUCxNQUFNLFdBQVcsR0FBRyxPQUFPLENBQUMsV0FBVztRQUNyQyxDQUFDLENBQUMsMEJBQTBCLG1CQUFtQixDQUMzQyxNQUFNLENBQUMsV0FBVyxDQUNoQixNQUFNLENBQUMsT0FBTyxDQUFDLE9BQU8sQ0FBQyxXQUFXLENBQUMsQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUMsRUFBRSxFQUFFLENBQUMsQ0FBQyxDQUFDLEVBQUUsR0FBRyxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQ2pFLENBQ0YsR0FBRztRQUNOLENBQUMsQ0FBQyxFQUFFLENBQUM7SUFFUCxNQUFNLFVBQVUsR0FBRyxPQUFPLENBQUMscUJBQXFCO1FBQzlDLENBQUMsQ0FBQywrQkFBK0IsSUFBSSxDQUFDLFNBQVMsQ0FDM0MsT0FBTyxDQUFDLHFCQUFxQixDQUM5QixHQUFHO1FBQ04sQ0FBQyxDQUFDLEVBQUUsQ0FBQztJQUVQLE1BQU0sb0JBQW9CLEdBQ3hCLE9BQU8sQ0FBQyxvQkFBb0IsSUFBSSxJQUFJLENBQUMsQ0FBQyxDQUFDLDRCQUE0QixDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUM7SUFFM0UsTUFBTSxRQUFRLEdBQUcsT0FBTyxDQUFDLFFBQVEsQ0FBQztJQUNsQyxNQUFNLFVBQVUsR0FBRyxHQUFHLENBQUM7SUFFdkIsT0FBTyxlQUFlLE9BQU8sQ0FBQyxTQUFTLGdCQUFnQixRQUFRLGtCQUFrQixVQUFVLG9CQUFvQixPQUFPLENBQUMsa0JBQWtCLGNBQWMsTUFBTSxjQUFjLE1BQU0sR0FBRyxvQkFBb0IsR0FBRyxXQUFXLEdBQUcsVUFBVSxHQUFHLG9CQUFvQixFQUFFLENBQUM7QUFDL1AsQ0FBQyxDQUFDO0FBakNXLFFBQUEsc0NBQXNDLDBDQWlDakQ7QUFFRjs7R0FFRztBQUNJLE1BQU0scUNBQXFDLEdBQUcsR0FBVyxFQUFFO0lBQ2hFLE9BQU8sSUFBQSxtQ0FBMkIsRUFDaEMsaUJBQWlCLENBQUMsNEJBQTRCLEVBQzlDLGVBQWUsQ0FDaEIsQ0FBQztBQUNKLENBQUMsQ0FBQztBQUxXLFFBQUEscUNBQXFDLHlDQUtoRDtBQW1CRjs7R0FFRztBQUNJLE1BQU0sbUNBQW1DLEdBQUcsQ0FDakQsT0FBa0MsRUFDMUIsRUFBRTtJQUNWLE1BQU0sVUFBVSxHQUFHLE9BQU8sQ0FBQyxZQUFZLElBQUksR0FBRyxDQUFDO0lBQy9DLE1BQU0sTUFBTSxHQUFHLE9BQU8sQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDLGFBQWEsT0FBTyxDQUFDLE1BQU0sRUFBRSxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUM7SUFDbkUsTUFBTSxjQUFjLEdBQ2xCLE9BQU8sQ0FBQyxjQUFjLEtBQUssU0FBUztRQUNsQyxDQUFDLENBQUMsdUJBQXVCLE9BQU8sQ0FBQyxjQUFjLEVBQUU7UUFDakQsQ0FBQyxDQUFDLEVBQUUsQ0FBQztJQUNULE1BQU0sSUFBSSxHQUFHLE9BQU8sQ0FBQyxJQUFJLEtBQUssU0FBUyxDQUFDLENBQUMsQ0FBQyxXQUFXLE9BQU8sQ0FBQyxJQUFJLEVBQUUsQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDO0lBQ3pFLE9BQU8sSUFBQSxtQ0FBMkIsRUFDaEMsaUJBQWlCLENBQUMsa0JBQWtCLEVBQ3BDLGVBQWUsT0FBTyxDQUFDLFFBQVEsa0JBQWtCLFVBQVUsR0FBRyxNQUFNLEdBQUcsY0FBYyxHQUFHLElBQUksRUFBRSxDQUMvRixDQUFDO0FBQ0osQ0FBQyxDQUFDO0FBZFcsUUFBQSxtQ0FBbUMsdUNBYzlDO0FBRUY7O0dBRUc7QUFDSSxNQUFNLGtDQUFrQyxHQUFHLENBQ2hELGFBQXNCLEVBQ3RCLEVBQUUsSUFBSSxFQUFFLE1BQU0sRUFBRSxVQUFVLEVBQTZCLEVBQ3JCLEVBQUUsQ0FBQyxDQUFDO0lBQ3RDLDBCQUEwQixFQUFFLE1BQU0sRUFBRSxVQUFVLElBQUksRUFBRTtJQUNwRCx5QkFBeUIsRUFBRSxJQUFJLEVBQUUsV0FBVyxJQUFJLEVBQUU7SUFDbEQsa0NBQWtDLEVBQUUsVUFBVTtRQUM1QyxDQUFDLENBQUMsSUFBSSxDQUFDLElBQUksQ0FDUCxJQUFJLENBQUMsUUFBUSxDQUFDLGFBQWEsQ0FBQyxNQUFNLEVBQUUsVUFBVSxDQUFDLE1BQU0sQ0FBQyxFQUN0RCxNQUFNLEVBQ04sUUFBUSxDQUNUO1FBQ0gsQ0FBQyxDQUFDLEVBQUU7SUFDTiw4QkFBOEIsRUFBRSxNQUFNO1FBQ3BDLENBQUMsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUNQLElBQUksQ0FBQyxRQUFRLENBQUMsYUFBYSxDQUFDLE1BQU0sRUFBRSxNQUFNLENBQUMsTUFBTSxDQUFDLEVBQ2xELE1BQU0sRUFDTixRQUFRLENBQ1Q7UUFDSCxDQUFDLENBQUMsRUFBRTtJQUNOLDRCQUE0QixFQUFFLElBQUk7UUFDaEMsQ0FBQyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQ1AsSUFBSSxDQUFDLFFBQVEsQ0FBQyxhQUFhLENBQUMsTUFBTSxFQUFFLElBQUksQ0FBQyxNQUFNLENBQUMsRUFDaEQsSUFBSSxDQUFDLE9BQU8sRUFDWixHQUFHLElBQUksQ0FBQyxHQUFHLENBQUMsT0FBTyxDQUFDLEtBQUssQ0FBQyxHQUFHLENBQUMsRUFDOUIsSUFBSSxDQUFDLEdBQUcsQ0FBQyxVQUFVLEVBQ25CLElBQUksQ0FBQyxHQUFHLENBQUMsT0FBTyxFQUNoQixHQUFHLElBQUksQ0FBQyxHQUFHLENBQUMsVUFBVSxJQUFJLElBQUksQ0FBQyxHQUFHLENBQUMsT0FBTyxNQUFNLENBQ2pEO1FBQ0gsQ0FBQyxDQUFDLEVBQUU7SUFDTix3Q0FBd0MsRUFBRSxVQUFVO1FBQ2xELENBQUMsQ0FBQywyQ0FBbUIsQ0FBQyxJQUFJLENBQUMsZ0JBQWdCLENBQUMsVUFBVSxDQUFDLGNBQWMsQ0FBQztRQUN0RSxDQUFDLENBQUMsRUFBRTtJQUNOLDBDQUEwQyxFQUFFLE1BQU07UUFDaEQsQ0FBQyxDQUFDLDJDQUFtQixDQUFDLE1BQU0sQ0FBQyxnQkFBZ0IsQ0FBQyxNQUFNLENBQUMsY0FBYyxDQUFDO1FBQ3BFLENBQUMsQ0FBQyxFQUFFO0lBQ04sd0NBQXdDLEVBQUUsSUFBSTtRQUM1QyxDQUFDLENBQUMsMkNBQW1CLENBQUMsSUFBSSxDQUFDLGdCQUFnQixDQUFDLElBQUksQ0FBQyxjQUFjLENBQUM7UUFDaEUsQ0FBQyxDQUFDLEVBQUU7Q0FDUCxDQUFDLENBQUM7QUF2Q1UsUUFBQSxrQ0FBa0Msc0NBdUM1QyIsInNvdXJjZXNDb250ZW50IjpbIi8qISBDb3B5cmlnaHQgW0FtYXpvbi5jb21dKGh0dHA6Ly9hbWF6b24uY29tLyksIEluYy4gb3IgaXRzIGFmZmlsaWF0ZXMuIEFsbCBSaWdodHMgUmVzZXJ2ZWQuXG5TUERYLUxpY2Vuc2UtSWRlbnRpZmllcjogQXBhY2hlLTIuMCAqL1xuaW1wb3J0ICogYXMgcGF0aCBmcm9tIFwicGF0aFwiO1xuaW1wb3J0IHsgUHJvamVjdCB9IGZyb20gXCJwcm9qZW5cIjtcbmltcG9ydCAqIGFzIHJlYWRQa2cgZnJvbSBcInJlYWQtcGtnLXVwXCI7XG5pbXBvcnQgeyBMYW5ndWFnZSwgTGlicmFyeSB9IGZyb20gXCIuLi8uLi9sYW5ndWFnZXNcIjtcbmltcG9ydCB7IE1vY2tSZXNwb25zZURhdGFHZW5lcmF0aW9uT3B0aW9ucyB9IGZyb20gXCIuLi8uLi90eXBlc1wiO1xuaW1wb3J0IHsgR2VuZXJhdGVkSGFuZGxlcnNQcm9qZWN0cyB9IGZyb20gXCIuLi9nZW5lcmF0ZVwiO1xuaW1wb3J0IHsgUnVudGltZVZlcnNpb25VdGlscyB9IGZyb20gXCIuLi9ydW50aW1lLXZlcnNpb24tdXRpbHNcIjtcblxuLyoqXG4gKiBFbnVtIGZvciBnZW5lcmF0b3IgZGlyZWN0b3JpZXMgZm9yIG5vbi1ydW50aW1lIGdlbmVyYXRvcnNcbiAqL1xuZXhwb3J0IGVudW0gT3RoZXJHZW5lcmF0b3JzIHtcbiAgRE9DUyA9IFwiZG9jc1wiLFxuICAvLyBJbmZyYXN0cnVjdHVyZVxuICBUWVBFU0NSSVBUX0NES19JTkZSQVNUUlVDVFVSRSA9IFwidHlwZXNjcmlwdC1jZGstaW5mcmFzdHJ1Y3R1cmVcIixcbiAgUFlUSE9OX0NES19JTkZSQVNUUlVDVFVSRSA9IFwicHl0aG9uLWNkay1pbmZyYXN0cnVjdHVyZVwiLFxuICBKQVZBX0NES19JTkZSQVNUUlVDVFVSRSA9IFwiamF2YS1jZGstaW5mcmFzdHJ1Y3R1cmVcIixcbiAgLy8gSGFuZGxlcnNcbiAgVFlQRVNDUklQVF9MQU1CREFfSEFORExFUlMgPSBcInR5cGVzY3JpcHQtbGFtYmRhLWhhbmRsZXJzXCIsXG4gIFBZVEhPTl9MQU1CREFfSEFORExFUlMgPSBcInB5dGhvbi1sYW1iZGEtaGFuZGxlcnNcIixcbiAgSkFWQV9MQU1CREFfSEFORExFUlMgPSBcImphdmEtbGFtYmRhLWhhbmRsZXJzXCIsXG59XG5cbmV4cG9ydCBlbnVtIFR5cGVTYWZlQXBpU2NyaXB0IHtcbiAgUEFSU0VfT1BFTkFQSV9TUEVDID0gXCJ0eXBlLXNhZmUtYXBpLnBhcnNlLW9wZW5hcGktc3BlY1wiLFxuICBHRU5FUkFURSA9IFwidHlwZS1zYWZlLWFwaS5nZW5lcmF0ZVwiLFxuICBHRU5FUkFURV9NT0NLX0RBVEEgPSBcInR5cGUtc2FmZS1hcGkuZ2VuZXJhdGUtbW9jay1kYXRhXCIsXG4gIEdFTkVSQVRFX0hUTUxfUkVET0NfRE9DUyA9IFwidHlwZS1zYWZlLWFwaS5nZW5lcmF0ZS1odG1sLXJlZG9jLWRvY3NcIixcbiAgQ0xFQU5fT1BFTkFQSV9HRU5FUkFURURfQ09ERSA9IFwidHlwZS1zYWZlLWFwaS5jbGVhbi1vcGVuYXBpLWdlbmVyYXRlZC1jb2RlXCIsXG4gIENPUFlfR1JBRExFX1dSQVBQRVIgPSBcInR5cGUtc2FmZS1hcGkuY29weS1ncmFkbGUtd3JhcHBlclwiLFxufVxuXG4vKipcbiAqIEdlbmVyYXRvciBkaXJlY3RvcnkgZm9yIG9wZW5hcGkgZ2VuZXJhdGlvbiBjb250YWluaW5nIHRlbXBsYXRlcywgY29uZmlnIGV0Yy5cbiAqL1xuZXhwb3J0IHR5cGUgR2VuZXJhdG9yRGlyZWN0b3J5ID0gTGFuZ3VhZ2UgfCBMaWJyYXJ5IHwgT3RoZXJHZW5lcmF0b3JzO1xuXG4vKipcbiAqIFR5cGVzIG9mIG5vcm1hbGl6ZXJzIHN1cHBvcnRlZCBieSBvcGVuYXBpLWdlbmVyYXRvclxuICogQHNlZSBodHRwczovL29wZW5hcGktZ2VuZXJhdG9yLnRlY2gvZG9jcy9jdXN0b21pemF0aW9uLyNvcGVuYXBpLW5vcm1hbGl6ZXJcbiAqL1xuZXhwb3J0IHR5cGUgT3BlbkFwaU5vcm1hbGl6ZXIgPSBcIktFRVBfT05MWV9GSVJTVF9UQUdfSU5fT1BFUkFUSU9OXCI7XG5cbi8qKlxuICogT3B0aW9ucyBmb3IgZ2VuZXJhdGluZyBjb2RlIG9yIGRvY3MgdXNpbmcgT3BlbkFQSSBHZW5lcmF0b3IgQ0xJXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgR2VuZXJhdGlvbk9wdGlvbnMge1xuICAvKipcbiAgICogVGhlIE9wZW5BUEkgZ2VuZXJhdG9yIHRvIHVzZSB0byBnZW5lcmF0ZSB0aGUgY29kZS9kb2NzXG4gICAqL1xuICByZWFkb25seSBnZW5lcmF0b3I6IHN0cmluZztcbiAgLyoqXG4gICAqIFRoZSBkaXJlY3RvcnkgdG8gdXNlIGZvciBPcGVuQVBJIGdlbmVyYXRpb25cbiAgICovXG4gIHJlYWRvbmx5IGdlbmVyYXRvckRpcmVjdG9yeTogR2VuZXJhdG9yRGlyZWN0b3J5O1xuICAvKipcbiAgICogVGhlIHBhdGggb2YgdGhlIE9wZW5BUEkgc3BlYyB0byBnZW5lcmF0ZSBjb2RlIGZyb21cbiAgICovXG4gIHJlYWRvbmx5IHNwZWNQYXRoOiBzdHJpbmc7XG4gIC8qKlxuICAgKiBBZGRpdGlvbmFsIHByb3BlcnRpZXMgdG8gcGFzcyB0byB0aGUgZ2VuZXJhdGUgY2xpXG4gICAqL1xuICByZWFkb25seSBhZGRpdGlvbmFsUHJvcGVydGllcz86IHtcbiAgICBba2V5OiBzdHJpbmddOiBzdHJpbmc7XG4gIH07XG4gIC8qKlxuICAgKiBTdXBwbHkgdGhlIHJlbGF0aXZlIHBhdGggZnJvbSB0aGUgY29kZSBwcm9qZWN0IHJvb3QgdG8gdGhlIHNvdXJjZSBjb2RlIGRpcmVjdG9yeSBpbiB3aGljaCBjdXN0b20gZ2VuZXJhdGVkIGZpbGVzXG4gICAqIChlZy4gb3BlcmF0aW9uIGNvbmZpZykgc2hvdWxkIGJlIHBsYWNlZC5cbiAgICovXG4gIHJlYWRvbmx5IHNyY0Rpcj86IHN0cmluZztcbiAgLyoqXG4gICAqIFN1cHBseSB0aGUgcmVsYXRpdmUgcGF0aCBmcm9tIHRoZSBjb2RlIHByb2plY3Qgcm9vdCB0byB0aGUgdGVzdCBkaXJlY3RvcnkgaW4gd2hpY2ggY3VzdG9tIGdlbmVyYXRlZCB0ZXN0IGZpbGVzXG4gICAqIHNob3VsZCBiZSBnZW5lcmF0ZWRcbiAgICovXG4gIHJlYWRvbmx5IHRzdERpcj86IHN0cmluZztcbiAgLyoqXG4gICAqIE5vcm1hbGl6ZXJzIHRvIGFwcGx5IHRvIHRoZSBzcGVjIHByaW9yIHRvIGdlbmVyYXRpb24sIGlmIGFueVxuICAgKiBAc2VlIGh0dHBzOi8vb3BlbmFwaS1nZW5lcmF0b3IudGVjaC9kb2NzL2N1c3RvbWl6YXRpb24vI29wZW5hcGktbm9ybWFsaXplclxuICAgKi9cbiAgcmVhZG9ubHkgbm9ybWFsaXplcnM/OiBQYXJ0aWFsPFJlY29yZDxPcGVuQXBpTm9ybWFsaXplciwgYm9vbGVhbj4+O1xuICAvKipcbiAgICogVmVuZG9yIGV4dGVuc2lvbnMgdG8gYWRkIGZvciBjb2RlIGdlbmVyYXRpb24sIGFsbG93aW5nIGN1c3RvbSBwcm9wZXJ0aWVzIHRvIGJlIHBhc3NlZCB0byB0aGUgZ2VuZXJhdG9yIHRlbXBsYXRlc1xuICAgKiBLZXlzIHNob3VsZCBiZWdpbiB3aXRoIFwieC1cIlxuICAgKi9cbiAgcmVhZG9ubHkgZXh0cmFWZW5kb3JFeHRlbnNpb25zPzogUmVjb3JkPHN0cmluZywgc3RyaW5nIHwgYm9vbGVhbj47XG4gIC8qKlxuICAgKiBHZW5lcmF0ZSBhbGlhcyBhcyBtb2RlbC5cbiAgICogRGVmaW5lcyB3aGV0aGVyIFwicHJpbWl0aXZlIHR5cGVzXCIgZGVmaW5lZCBhdCB0aGUgbW9kZWwvc2NoZW1hIGxldmVsIHdpbGwgYmUgd3JhcHBlZCBpbiBhIG1vZGVsIChpZSBtYXBzL2xpc3RzKVxuICAgKiBAc2VlIGh0dHBzOi8vb3BlbmFwaS1nZW5lcmF0b3IudGVjaC9kb2NzL2dsb2JhbHMvXG4gICAqIEBkZWZhdWx0IHRydWVcbiAgICovXG4gIHJlYWRvbmx5IGdlbmVyYXRlQWxpYXNBc01vZGVsPzogYm9vbGVhbjtcbn1cblxuLyoqXG4gKiBHZXQgdGhlIGN1cnJlbnQgcGFja2FnZSB2ZXJzaW9uXG4gKi9cbmNvbnN0IGdldFBhY2thZ2VWZXJzaW9uID0gKCk6IHN0cmluZyA9PiB7XG4gIGNvbnN0IHsgcGFja2FnZUpzb24gfSA9IHJlYWRQa2cuc3luYyh7XG4gICAgY3dkOiBwYXRoLnJlc29sdmUoX19kaXJuYW1lKSxcbiAgfSkhO1xuICByZXR1cm4gcGFja2FnZUpzb24udmVyc2lvbjtcbn07XG5cbi8qKlxuICogUmV0dXJuIHRoZSBlbnZpcm9ubWVudCB0aGF0IHNob3VsZCBiZSB1c2VkIGZvciBleGVjdXRpbmcgdHlwZSBzYWZlIGFwaSBjb21tYW5kc1xuICovXG5leHBvcnQgY29uc3QgZ2V0VHlwZVNhZmVBcGlUYXNrRW52aXJvbm1lbnQgPSAoKTogeyBba2V5OiBzdHJpbmddOiBzdHJpbmcgfSA9PiAoe1xuICBBV1NfUERLX1ZFUlNJT046IGdldFBhY2thZ2VWZXJzaW9uKCksXG59KTtcblxuLyoqXG4gKiBCdWlsZCBhIGNvbW1hbmQgZm9yIHJ1bm5pbmcgYSBzY3JpcHQgZnJvbSB0aGlzIHByb2plY3QncyBiaW5cbiAqL1xuZXhwb3J0IGNvbnN0IGJ1aWxkVHlwZVNhZmVBcGlFeGVjQ29tbWFuZCA9IChcbiAgc2NyaXB0OiBUeXBlU2FmZUFwaVNjcmlwdCxcbiAgYXJncz86IHN0cmluZ1xuKTogc3RyaW5nID0+IHtcbiAgcmV0dXJuIGBucHggLS15ZXMgLXAgQGF3cy9wZGtAJEFXU19QREtfVkVSU0lPTiAke3NjcmlwdH0ke1xuICAgIGFyZ3MgPyBgICR7YXJnc31gIDogXCJcIlxuICB9YDtcbn07XG5cbmNvbnN0IHNlcmlhbGl6ZVByb3BlcnRpZXMgPSAocHJvcGVydGllczogeyBba2V5OiBzdHJpbmddOiBzdHJpbmcgfSkgPT5cbiAgT2JqZWN0LmVudHJpZXMocHJvcGVydGllcylcbiAgICAubWFwKChba2V5LCB2YWx1ZV0pID0+IGAke2tleX09JHt2YWx1ZX1gKVxuICAgIC5qb2luKFwiLFwiKTtcblxuLyoqXG4gKiBHZW5lcmF0ZSBjb2RlIG9yIGRvY3MgYnkgaW52b2tpbmcgdGhlIHJvb3QgZ2VuZXJhdGUgc2NyaXB0XG4gKi9cbmV4cG9ydCBjb25zdCBidWlsZEludm9rZU9wZW5BcGlHZW5lcmF0b3JDb21tYW5kQXJncyA9IChcbiAgb3B0aW9uczogR2VuZXJhdGlvbk9wdGlvbnNcbik6IHN0cmluZyA9PiB7XG4gIGNvbnN0IHNyY0RpciA9IG9wdGlvbnMuc3JjRGlyID8/IFwic3JjXCI7XG4gIGNvbnN0IHRzdERpciA9IG9wdGlvbnMudHN0RGlyID8/IFwidGVzdFwiO1xuXG4gIGNvbnN0IGFkZGl0aW9uYWxQcm9wZXJ0aWVzID0gb3B0aW9ucy5hZGRpdGlvbmFsUHJvcGVydGllc1xuICAgID8gYCAtLWFkZGl0aW9uYWwtcHJvcGVydGllcyBcIiR7c2VyaWFsaXplUHJvcGVydGllcyhcbiAgICAgICAgb3B0aW9ucy5hZGRpdGlvbmFsUHJvcGVydGllc1xuICAgICAgKX1cImBcbiAgICA6IFwiXCI7XG5cbiAgY29uc3Qgbm9ybWFsaXplcnMgPSBvcHRpb25zLm5vcm1hbGl6ZXJzXG4gICAgPyBgIC0tb3BlbmFwaS1ub3JtYWxpemVyIFwiJHtzZXJpYWxpemVQcm9wZXJ0aWVzKFxuICAgICAgICBPYmplY3QuZnJvbUVudHJpZXMoXG4gICAgICAgICAgT2JqZWN0LmVudHJpZXMob3B0aW9ucy5ub3JtYWxpemVycykubWFwKChbaywgdl0pID0+IFtrLCBgJHt2fWBdKVxuICAgICAgICApXG4gICAgICApfVwiYFxuICAgIDogXCJcIjtcblxuICBjb25zdCBleHRlbnNpb25zID0gb3B0aW9ucy5leHRyYVZlbmRvckV4dGVuc2lvbnNcbiAgICA/IGAgLS1leHRyYS12ZW5kb3ItZXh0ZW5zaW9ucyAnJHtKU09OLnN0cmluZ2lmeShcbiAgICAgICAgb3B0aW9ucy5leHRyYVZlbmRvckV4dGVuc2lvbnNcbiAgICAgICl9J2BcbiAgICA6IFwiXCI7XG5cbiAgY29uc3QgZ2VuZXJhdGVBbGlhc0FzTW9kZWwgPVxuICAgIG9wdGlvbnMuZ2VuZXJhdGVBbGlhc0FzTW9kZWwgPz8gdHJ1ZSA/IFwiIC0tZ2VuZXJhdGUtYWxpYXMtYXMtbW9kZWxcIiA6IFwiXCI7XG5cbiAgY29uc3Qgc3BlY1BhdGggPSBvcHRpb25zLnNwZWNQYXRoO1xuICBjb25zdCBvdXRwdXRQYXRoID0gXCIuXCI7XG5cbiAgcmV0dXJuIGAtLWdlbmVyYXRvciAke29wdGlvbnMuZ2VuZXJhdG9yfSAtLXNwZWMtcGF0aCAke3NwZWNQYXRofSAtLW91dHB1dC1wYXRoICR7b3V0cHV0UGF0aH0gLS1nZW5lcmF0b3ItZGlyICR7b3B0aW9ucy5nZW5lcmF0b3JEaXJlY3Rvcnl9IC0tc3JjLWRpciAke3NyY0Rpcn0gLS10c3QtZGlyICR7dHN0RGlyfSR7YWRkaXRpb25hbFByb3BlcnRpZXN9JHtub3JtYWxpemVyc30ke2V4dGVuc2lvbnN9JHtnZW5lcmF0ZUFsaWFzQXNNb2RlbH1gO1xufTtcblxuLyoqXG4gKiBCdWlsZHMgYSBjb21tYW5kIHRvIGNsZWFuIHVwIGZpbGVzIHdoaWNoIHdlcmUgcHJldmlvdXNseSBnZW5lcmF0ZWQgYnkgb3BlbmFwaSBnZW5lcmF0b3JcbiAqL1xuZXhwb3J0IGNvbnN0IGJ1aWxkQ2xlYW5PcGVuQXBpR2VuZXJhdGVkQ29kZUNvbW1hbmQgPSAoKTogc3RyaW5nID0+IHtcbiAgcmV0dXJuIGJ1aWxkVHlwZVNhZmVBcGlFeGVjQ29tbWFuZChcbiAgICBUeXBlU2FmZUFwaVNjcmlwdC5DTEVBTl9PUEVOQVBJX0dFTkVSQVRFRF9DT0RFLFxuICAgIGAtLWNvZGUtcGF0aCAuYFxuICApO1xufTtcblxuLyoqXG4gKiBPcHRpb25zIGZvciBnZW5lcmF0aW5nIG1vY2sgZGF0YSBqc29uIGZpbGVzXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgTW9ja0RhdGFHZW5lcmF0aW9uT3B0aW9uc1xuICBleHRlbmRzIE1vY2tSZXNwb25zZURhdGFHZW5lcmF0aW9uT3B0aW9ucyB7XG4gIC8qKlxuICAgKiBUaGUgcGF0aCBvZiB0aGUgT3BlbkFQSSBzcGVjIHRvIGdlbmVyYXRlIGRhdGEgZm9yXG4gICAqL1xuICByZWFkb25seSBzcGVjUGF0aDogc3RyaW5nO1xuICAvKipcbiAgICogT3V0cHV0IHN1YiBkaXJlY3RvcnkgcmVsYXRpdmUgdG8gdGhlIG91dGRpciBpbiB3aGljaCB0byBnZW5lcmF0ZSBtb2NrIGRhdGFcbiAgICogTW9jayBkYXRhIHdpbGwgYmUgd3JpdHRlbiB0byBhIGRpcmVjdG9yeSBuYW1lZCAnbW9ja3MnIHdpdGhpbiB0aGUgc3ViIGRpcmVjdG9yeVxuICAgKiBAZGVmYXVsdCAuXG4gICAqL1xuICByZWFkb25seSBvdXRwdXRTdWJEaXI/OiBzdHJpbmc7XG59XG5cbi8qKlxuICogSW52b2tlIHRoZSBtb2NrIGRhdGEgZ2VuZXJhdG9yIHNjcmlwdFxuICovXG5leHBvcnQgY29uc3QgYnVpbGRJbnZva2VNb2NrRGF0YUdlbmVyYXRvckNvbW1hbmQgPSAoXG4gIG9wdGlvbnM6IE1vY2tEYXRhR2VuZXJhdGlvbk9wdGlvbnNcbik6IHN0cmluZyA9PiB7XG4gIGNvbnN0IG91dHB1dFBhdGggPSBvcHRpb25zLm91dHB1dFN1YkRpciA/PyBcIi5cIjtcbiAgY29uc3QgbG9jYWxlID0gb3B0aW9ucy5sb2NhbGUgPyBgIC0tbG9jYWxlICR7b3B0aW9ucy5sb2NhbGV9YCA6IFwiXCI7XG4gIGNvbnN0IG1heEFycmF5TGVuZ3RoID1cbiAgICBvcHRpb25zLm1heEFycmF5TGVuZ3RoICE9PSB1bmRlZmluZWRcbiAgICAgID8gYCAtLW1heC1hcnJheS1sZW5ndGggJHtvcHRpb25zLm1heEFycmF5TGVuZ3RofWBcbiAgICAgIDogXCJcIjtcbiAgY29uc3Qgc2VlZCA9IG9wdGlvbnMuc2VlZCAhPT0gdW5kZWZpbmVkID8gYCAtLXNlZWQgJHtvcHRpb25zLnNlZWR9YCA6IFwiXCI7XG4gIHJldHVybiBidWlsZFR5cGVTYWZlQXBpRXhlY0NvbW1hbmQoXG4gICAgVHlwZVNhZmVBcGlTY3JpcHQuR0VORVJBVEVfTU9DS19EQVRBLFxuICAgIGAtLXNwZWMtcGF0aCAke29wdGlvbnMuc3BlY1BhdGh9IC0tb3V0cHV0LXBhdGggJHtvdXRwdXRQYXRofSR7bG9jYWxlfSR7bWF4QXJyYXlMZW5ndGh9JHtzZWVkfWBcbiAgKTtcbn07XG5cbi8qKlxuICogUmV0dXJuIHZlbmRvciBleHRlbnNpb25zIGNvbnRhaW5pbmcgZGV0YWlscyBhYm91dCB0aGUgaGFuZGxlciBwcm9qZWN0c1xuICovXG5leHBvcnQgY29uc3QgZ2V0SGFuZGxlcnNQcm9qZWN0VmVuZG9yRXh0ZW5zaW9ucyA9IChcbiAgdGFyZ2V0UHJvamVjdDogUHJvamVjdCxcbiAgeyBqYXZhLCBweXRob24sIHR5cGVzY3JpcHQgfTogR2VuZXJhdGVkSGFuZGxlcnNQcm9qZWN0c1xuKTogUmVjb3JkPHN0cmluZywgc3RyaW5nIHwgYm9vbGVhbj4gPT4gKHtcbiAgXCJ4LWhhbmRsZXJzLXB5dGhvbi1tb2R1bGVcIjogcHl0aG9uPy5tb2R1bGVOYW1lID8/IFwiXCIsXG4gIFwieC1oYW5kbGVycy1qYXZhLXBhY2thZ2VcIjogamF2YT8ucGFja2FnZU5hbWUgPz8gXCJcIixcbiAgXCJ4LWhhbmRsZXJzLXR5cGVzY3JpcHQtYXNzZXQtcGF0aFwiOiB0eXBlc2NyaXB0XG4gICAgPyBwYXRoLmpvaW4oXG4gICAgICAgIHBhdGgucmVsYXRpdmUodGFyZ2V0UHJvamVjdC5vdXRkaXIsIHR5cGVzY3JpcHQub3V0ZGlyKSxcbiAgICAgICAgXCJkaXN0XCIsXG4gICAgICAgIFwibGFtYmRhXCJcbiAgICAgIClcbiAgICA6IFwiXCIsXG4gIFwieC1oYW5kbGVycy1weXRob24tYXNzZXQtcGF0aFwiOiBweXRob25cbiAgICA/IHBhdGguam9pbihcbiAgICAgICAgcGF0aC5yZWxhdGl2ZSh0YXJnZXRQcm9qZWN0Lm91dGRpciwgcHl0aG9uLm91dGRpciksXG4gICAgICAgIFwiZGlzdFwiLFxuICAgICAgICBcImxhbWJkYVwiXG4gICAgICApXG4gICAgOiBcIlwiLFxuICBcIngtaGFuZGxlcnMtamF2YS1hc3NldC1wYXRoXCI6IGphdmFcbiAgICA/IHBhdGguam9pbihcbiAgICAgICAgcGF0aC5yZWxhdGl2ZSh0YXJnZXRQcm9qZWN0Lm91dGRpciwgamF2YS5vdXRkaXIpLFxuICAgICAgICBqYXZhLmRpc3RkaXIsXG4gICAgICAgIC4uLmphdmEucG9tLmdyb3VwSWQuc3BsaXQoXCIuXCIpLFxuICAgICAgICBqYXZhLnBvbS5hcnRpZmFjdElkLFxuICAgICAgICBqYXZhLnBvbS52ZXJzaW9uLFxuICAgICAgICBgJHtqYXZhLnBvbS5hcnRpZmFjdElkfS0ke2phdmEucG9tLnZlcnNpb259LmphcmBcbiAgICAgIClcbiAgICA6IFwiXCIsXG4gIFwieC1oYW5kbGVycy1ub2RlLWxhbWJkYS1ydW50aW1lLXZlcnNpb25cIjogdHlwZXNjcmlwdFxuICAgID8gUnVudGltZVZlcnNpb25VdGlscy5OT0RFLmdldExhbWJkYVJ1bnRpbWUodHlwZXNjcmlwdC5ydW50aW1lVmVyc2lvbilcbiAgICA6IFwiXCIsXG4gIFwieC1oYW5kbGVycy1weXRob24tbGFtYmRhLXJ1bnRpbWUtdmVyc2lvblwiOiBweXRob25cbiAgICA/IFJ1bnRpbWVWZXJzaW9uVXRpbHMuUFlUSE9OLmdldExhbWJkYVJ1bnRpbWUocHl0aG9uLnJ1bnRpbWVWZXJzaW9uKVxuICAgIDogXCJcIixcbiAgXCJ4LWhhbmRsZXJzLWphdmEtbGFtYmRhLXJ1bnRpbWUtdmVyc2lvblwiOiBqYXZhXG4gICAgPyBSdW50aW1lVmVyc2lvblV0aWxzLkpBVkEuZ2V0TGFtYmRhUnVudGltZShqYXZhLnJ1bnRpbWVWZXJzaW9uKVxuICAgIDogXCJcIixcbn0pO1xuIl19