import { Project } from "projen";
import { GeneratedJavaHandlersProject } from "./handlers/generated-java-handlers-project";
import { GeneratedPythonHandlersProject } from "./handlers/generated-python-handlers-project";
import { GeneratedTypescriptHandlersProject } from "./handlers/generated-typescript-handlers-project";
import { GeneratedJavaRuntimeProject, GeneratedJavaTypesProjectOptions } from "./runtime/generated-java-runtime-project";
import { GeneratedPythonRuntimeProject, GeneratedPythonTypesProjectOptions } from "./runtime/generated-python-runtime-project";
import { GeneratedTypescriptRuntimeProject, GeneratedTypescriptTypesProjectOptions } from "./runtime/generated-typescript-runtime-project";
import { DocumentationFormat, Language, Library } from "../languages";
import { GeneratedDocumentationOptions } from "../types";
type CommonProjectOptions = "artifactId" | "generateClient" | "groupId" | "moduleName" | "name" | "outdir" | "specPath";
export interface GenerateProjectsOptions {
    /**
     * The parent project for the generated types
     */
    readonly parent: Project;
    /**
     * Whether the infrastructure and types projects are parented by an monorepo or not
     */
    readonly isWithinMonorepo?: boolean;
    /**
     * The name of the api package, used to infer package names unless overrides are specified
     */
    readonly parentPackageName: string;
    /**
     * The directory in which to generate code for all packages
     */
    readonly generatedCodeDir: string;
    /**
     * Path to the parsed spec file
     * We use the parsed spec such that refs are resolved to support multi-file specs
     */
    readonly parsedSpecPath: string;
}
/**
 * Options for generating types
 */
export interface GenerateRuntimeProjectsOptions extends GenerateProjectsOptions {
    /**
     * Options for the typescript project.
     * These will override any inferred properties (such as the package name).
     */
    readonly typescriptOptions: Omit<GeneratedTypescriptTypesProjectOptions, CommonProjectOptions>;
    /**
     * Options for the python project
     * These will override any inferred properties (such as the package name).
     */
    readonly pythonOptions: Omit<GeneratedPythonTypesProjectOptions, CommonProjectOptions>;
    /**
     * Options for the java project
     * These will override any inferred properties (such as the package name).
     */
    readonly javaOptions: Omit<GeneratedJavaTypesProjectOptions, CommonProjectOptions>;
}
/**
 * Options for generating libraries
 */
export interface GenerateLibraryProjectsOptions extends GenerateProjectsOptions {
    /**
     * Options for the react query hooks project
     * These will override any inferred properties (such as the package name)
     */
    readonly typescriptReactQueryHooksOptions: Omit<GeneratedTypescriptTypesProjectOptions, CommonProjectOptions>;
}
/**
 * Generated handlers projects
 */
export interface GeneratedHandlersProjects {
    /**
     * Java handlers project
     */
    readonly java?: GeneratedJavaHandlersProject;
    /**
     * Python handlers project
     */
    readonly python?: GeneratedPythonHandlersProject;
    /**
     * TypeScript handlers project
     */
    readonly typescript?: GeneratedTypescriptHandlersProject;
}
export interface GenerateInfraProjectOptions extends GenerateRuntimeProjectsOptions {
    /**
     * Generated runtime projects
     */
    readonly generatedRuntimes: {
        readonly java?: GeneratedJavaRuntimeProject;
        readonly python?: GeneratedPythonRuntimeProject;
        readonly typescript?: GeneratedTypescriptRuntimeProject;
    };
    readonly generatedHandlers: GeneratedHandlersProjects;
}
/**
 * Returns a generated infrastructure project for the given language
 */
export declare const generateInfraProject: (language: Language, options: GenerateInfraProjectOptions) => Project;
export interface GenerateHandlersProjectOptions extends GenerateRuntimeProjectsOptions {
    /**
     * Generated runtime projects
     */
    readonly generatedRuntimes: {
        readonly java?: GeneratedJavaRuntimeProject;
        readonly python?: GeneratedPythonRuntimeProject;
        readonly typescript?: GeneratedTypescriptRuntimeProject;
    };
}
/**
 * Create handlers projects in the given languages
 * @param languages the languages to generate for
 * @param options options for the projects to be created
 */
export declare const generateHandlersProjects: (languages: Language[], options: GenerateHandlersProjectOptions) => {
    [language: string]: Project;
};
/**
 * Create runtime projects in the given languages
 * @param languages the languages to generate for
 * @param options options for the projects to be created
 */
export declare const generateRuntimeProjects: (languages: Language[], options: GenerateRuntimeProjectsOptions) => {
    [language: string]: Project;
};
/**
 * Create library projects
 * @param libraries the libraries to generate for
 * @param options options for the projects to be created
 */
export declare const generateLibraryProjects: (libraries: Library[], options: GenerateLibraryProjectsOptions) => {
    [library: string]: Project;
};
export interface GenerateDocsProjectsOptions {
    /**
     * The parent project for the generated clients
     */
    readonly parent: Project;
    /**
     * The name of the api package, used to infer doc package names
     */
    readonly parentPackageName: string;
    /**
     * The directory in which to generate docs for all formats
     */
    readonly generatedDocsDir: string;
    /**
     * Path to the parsed spec file
     * We use the parsed spec such that refs are resolved to support multi-file specs
     */
    readonly parsedSpecPath: string;
    /**
     * User-specified options for generated documentation
     */
    readonly documentationOptions?: GeneratedDocumentationOptions;
}
export declare const generateDocsProjects: (formats: DocumentationFormat[], options: GenerateDocsProjectsOptions) => {
    [language: string]: Project;
};
export {};
