"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.generateDocsProjects = exports.generateLibraryProjects = exports.generateRuntimeProjects = exports.generateHandlersProjects = exports.generateInfraProject = void 0;
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const path = require("path");
const log4js_1 = require("log4js");
const projen_1 = require("projen");
const generated_html_redoc_documentation_project_1 = require("./documentation/generated-html-redoc-documentation-project");
const generated_html2_documentation_project_1 = require("./documentation/generated-html2-documentation-project");
const generated_markdown_documentation_project_1 = require("./documentation/generated-markdown-documentation-project");
const generated_plantuml_documentation_project_1 = require("./documentation/generated-plantuml-documentation-project");
const generated_java_handlers_project_1 = require("./handlers/generated-java-handlers-project");
const generated_python_handlers_project_1 = require("./handlers/generated-python-handlers-project");
const generated_typescript_handlers_project_1 = require("./handlers/generated-typescript-handlers-project");
const generated_java_cdk_infrastructure_project_1 = require("./infrastructure/cdk/generated-java-cdk-infrastructure-project");
const generated_python_cdk_infrastructure_project_1 = require("./infrastructure/cdk/generated-python-cdk-infrastructure-project");
const generated_typescript_cdk_infrastructure_project_1 = require("./infrastructure/cdk/generated-typescript-cdk-infrastructure-project");
const typescript_react_query_hooks_library_1 = require("./library/typescript-react-query-hooks-library");
const generated_java_runtime_project_1 = require("./runtime/generated-java-runtime-project");
const generated_python_runtime_project_1 = require("./runtime/generated-python-runtime-project");
const generated_typescript_runtime_project_1 = require("./runtime/generated-typescript-runtime-project");
const languages_1 = require("../languages");
const logger = (0, log4js_1.getLogger)();
// No dashes or underscores since this is used in the package name in imports
const sanitiseJavaProjectName = (name) => name.replace(/@/g, "").replace(/[\-/_]/g, "");
// kebab-case for java artifact ids
const sanitiseJavaArtifactId = (name) => name.replace(/@/g, "").replace(/[/_]/g, "-");
// kebab-case for typescript packages
const sanitiseTypescriptPackageName = (name) => name.replace(/_/g, "-");
// snake_case for python modules
const sanitisePythonModuleName = (name) => name.replace(/@/g, "").replace(/[\-/]/g, "_");
// Use dashes in project name since distributable's PKG-INFO always converts _ to -
// https://stackoverflow.com/questions/36300788/python-package-wheel-pkg-info-name
const sanitisePythonPackageName = (name) => name.replace(/@/g, "").replace(/[_/]/g, "-");
/**
 * Returns a generated client project for the given language
 */
const generateRuntimeProject = (language, options) => {
    const packageName = `${options.parentPackageName}-${language}-runtime`;
    const commonOptions = {
        outdir: path.join(options.generatedCodeDir, language),
        specPath: options.parsedSpecPath,
        parent: options.parent,
    };
    switch (language) {
        case languages_1.Language.TYPESCRIPT: {
            logger.trace("Attempting to generate TYPESCRIPT runtime project.");
            return new generated_typescript_runtime_project_1.GeneratedTypescriptRuntimeProject({
                ...commonOptions,
                name: sanitiseTypescriptPackageName(packageName),
                ...options.typescriptOptions,
                isWithinMonorepo: options.isWithinMonorepo,
            });
        }
        case languages_1.Language.PYTHON: {
            logger.trace("Attempting to generate PYTHON runtime project.");
            return new generated_python_runtime_project_1.GeneratedPythonRuntimeProject({
                ...commonOptions,
                name: sanitisePythonPackageName(packageName),
                moduleName: sanitisePythonModuleName(packageName),
                ...options.pythonOptions,
            });
        }
        case languages_1.Language.JAVA: {
            logger.trace("Attempting to generate JAVA runtime project.");
            return new generated_java_runtime_project_1.GeneratedJavaRuntimeProject({
                ...commonOptions,
                name: sanitiseJavaProjectName(packageName),
                artifactId: sanitiseJavaArtifactId(packageName),
                groupId: "com.generated.api",
                ...options.javaOptions,
            });
        }
        default:
            throw new Error(`Unknown runtime language ${language}`);
    }
};
/**
 * Returns a generated infrastructure project for the given language
 */
const generateInfraProject = (language, options) => {
    new projen_1.TextFile(options.parent, path.join(options.generatedCodeDir, "README.md"), {
        lines: [
            "## Generated Infrastructure",
            "",
            "This directory contains a generated type-safe CDK construct which will can the API gateway infrastructure for an API based on your model.",
        ],
        readonly: true,
    });
    const infraName = `${options.parentPackageName}-${language}-infra`;
    const commonOptions = {
        outdir: path.join(options.generatedCodeDir, language),
        specPath: options.parsedSpecPath,
        parent: options.parent,
        generatedHandlers: options.generatedHandlers,
    };
    switch (language) {
        case languages_1.Language.TYPESCRIPT: {
            logger.trace("Attempting to generate TYPESCRIPT infra project.");
            if (!options.generatedRuntimes.typescript) {
                throw new Error("A typescript types project must be created for typescript infrastructure");
            }
            return new generated_typescript_cdk_infrastructure_project_1.GeneratedTypescriptCdkInfrastructureProject({
                ...commonOptions,
                name: sanitiseTypescriptPackageName(infraName),
                generatedTypescriptTypes: options.generatedRuntimes.typescript,
                ...options.typescriptOptions,
                isWithinMonorepo: options.isWithinMonorepo,
            });
        }
        case languages_1.Language.PYTHON: {
            logger.trace("Attempting to generate PYTHON infra project.");
            if (!options.generatedRuntimes.python) {
                throw new Error("A python types project must be created for python infrastructure");
            }
            return new generated_python_cdk_infrastructure_project_1.GeneratedPythonCdkInfrastructureProject({
                ...commonOptions,
                name: sanitisePythonPackageName(infraName),
                moduleName: sanitisePythonModuleName(infraName),
                generatedPythonTypes: options.generatedRuntimes.python,
                ...options.pythonOptions,
            });
        }
        case languages_1.Language.JAVA: {
            logger.trace("Attempting to generate JAVA infra project.");
            if (!options.generatedRuntimes.java) {
                throw new Error("A java types project must be created for java infrastructure");
            }
            return new generated_java_cdk_infrastructure_project_1.GeneratedJavaCdkInfrastructureProject({
                ...commonOptions,
                name: sanitiseJavaProjectName(infraName),
                artifactId: sanitiseJavaArtifactId(infraName),
                groupId: "com.generated.api",
                generatedJavaTypes: options.generatedRuntimes.java,
                ...options.javaOptions,
            });
        }
        default:
            throw new Error(`Unknown infrastructure language ${language}`);
    }
};
exports.generateInfraProject = generateInfraProject;
/**
 * Returns a generated handlers project for the given language
 */
const generateHandlersProject = (language, options) => {
    const handlersName = `${options.parentPackageName}-${language}-handlers`;
    const commonOptions = {
        outdir: path.join(options.generatedCodeDir, language),
        specPath: options.parsedSpecPath,
        parent: options.parent,
    };
    switch (language) {
        case languages_1.Language.TYPESCRIPT: {
            logger.trace("Attempting to generate TYPESCRIPT handlers project.");
            if (!options.generatedRuntimes.typescript) {
                throw new Error("A typescript runtime project must be created for typescript handlers");
            }
            return new generated_typescript_handlers_project_1.GeneratedTypescriptHandlersProject({
                ...commonOptions,
                name: sanitiseTypescriptPackageName(handlersName),
                generatedTypescriptTypes: options.generatedRuntimes.typescript,
                ...options.typescriptOptions,
                isWithinMonorepo: options.isWithinMonorepo,
            });
        }
        case languages_1.Language.PYTHON: {
            logger.trace("Attempting to generate PYTHON handlers project.");
            if (!options.generatedRuntimes.python) {
                throw new Error("A python runtime project must be created for python handlers");
            }
            return new generated_python_handlers_project_1.GeneratedPythonHandlersProject({
                ...commonOptions,
                name: sanitisePythonPackageName(handlersName),
                moduleName: sanitisePythonModuleName(handlersName),
                generatedPythonTypes: options.generatedRuntimes.python,
                ...options.pythonOptions,
            });
        }
        case languages_1.Language.JAVA: {
            logger.trace("Attempting to generate JAVA handlers project.");
            if (!options.generatedRuntimes.java) {
                throw new Error("A java runtime project must be created for java handlers");
            }
            return new generated_java_handlers_project_1.GeneratedJavaHandlersProject({
                ...commonOptions,
                name: sanitiseJavaProjectName(handlersName),
                artifactId: sanitiseJavaArtifactId(handlersName),
                groupId: "com.generated.api",
                generatedJavaTypes: options.generatedRuntimes.java,
                ...options.javaOptions,
            });
        }
        default:
            throw new Error(`Unknown infrastructure language ${language}`);
    }
};
/**
 * Create handlers projects in the given languages
 * @param languages the languages to generate for
 * @param options options for the projects to be created
 */
const generateHandlersProjects = (languages, options) => {
    if (languages.length > 0) {
        new projen_1.TextFile(options.parent, path.join(options.generatedCodeDir, "README.md"), {
            lines: [
                "## Handlers",
                "",
                "This directory contains lambda handlers for implementing your API.",
                "",
                "Whenever an operation is annotated with the `@handler` trait in Smithy (or the `x-handler` vendor extension in OpenAPI), a stub handler implementation will be generated for you, which you are free to modify.",
            ],
            readonly: true,
        });
    }
    const generatedHandlers = {};
    languages.forEach((language) => {
        const project = generateHandlersProject(language, options);
        if (project != null) {
            generatedHandlers[language] = project;
        }
    });
    return generatedHandlers;
};
exports.generateHandlersProjects = generateHandlersProjects;
/**
 * Create runtime projects in the given languages
 * @param languages the languages to generate for
 * @param options options for the projects to be created
 */
const generateRuntimeProjects = (languages, options) => {
    new projen_1.TextFile(options.parent, path.join(options.generatedCodeDir, "README.md"), {
        lines: [
            "## Generated Runtimes",
            "",
            "This directory contains generated runtime projects based on your API model.",
            "",
            "Each runtime project includes types from your API model, as well as type-safe client and server code.",
        ],
        readonly: true,
    });
    const generatedRuntimes = {};
    languages.forEach((language) => {
        const project = generateRuntimeProject(language, options);
        if (project != null) {
            generatedRuntimes[language] = project;
        }
    });
    return generatedRuntimes;
};
exports.generateRuntimeProjects = generateRuntimeProjects;
/**
 * Returns a generated client project for the given language
 */
const generateLibraryProject = (library, options) => {
    const packageName = `${options.parentPackageName}-${library}`;
    const commonOptions = {
        outdir: path.join(options.generatedCodeDir, library),
        specPath: options.parsedSpecPath,
        parent: options.parent,
    };
    switch (library) {
        case languages_1.Library.TYPESCRIPT_REACT_QUERY_HOOKS: {
            return new typescript_react_query_hooks_library_1.TypescriptReactQueryHooksLibrary({
                ...commonOptions,
                name: sanitiseTypescriptPackageName(packageName),
                ...options.typescriptReactQueryHooksOptions,
                isWithinMonorepo: options.isWithinMonorepo,
            });
        }
        default:
            throw new Error(`Unknown library ${library}`);
    }
};
/**
 * Create library projects
 * @param libraries the libraries to generate for
 * @param options options for the projects to be created
 */
const generateLibraryProjects = (libraries, options) => {
    if (libraries.length > 0) {
        new projen_1.TextFile(options.parent, path.join(options.generatedCodeDir, "README.md"), {
            lines: [
                "## Generated Libraries",
                "",
                "This directory contains generated libraries based on your API model.",
            ],
            readonly: true,
        });
    }
    const generatedLibraries = {};
    libraries.forEach((library) => {
        const project = generateLibraryProject(library, options);
        if (project != null) {
            generatedLibraries[library] = project;
        }
    });
    return generatedLibraries;
};
exports.generateLibraryProjects = generateLibraryProjects;
const generateDocsProject = (format, options) => {
    const commonProps = {
        name: `${options.parentPackageName}-documentation-${format.replace(/_/g, "-")}`,
        parent: options.parent,
        outdir: path.join(options.generatedDocsDir, format),
        specPath: options.parsedSpecPath,
    };
    switch (format) {
        case languages_1.DocumentationFormat.HTML2: {
            return new generated_html2_documentation_project_1.GeneratedHtml2DocumentationProject({
                ...commonProps,
                ...options.documentationOptions?.html2,
            });
        }
        case languages_1.DocumentationFormat.HTML_REDOC: {
            return new generated_html_redoc_documentation_project_1.GeneratedHtmlRedocDocumentationProject({
                ...commonProps,
                ...options.documentationOptions?.htmlRedoc,
            });
        }
        case languages_1.DocumentationFormat.MARKDOWN: {
            return new generated_markdown_documentation_project_1.GeneratedMarkdownDocumentationProject({
                ...commonProps,
                ...options.documentationOptions?.markdown,
            });
        }
        case languages_1.DocumentationFormat.PLANTUML: {
            return new generated_plantuml_documentation_project_1.GeneratedPlantumlDocumentationProject({
                ...commonProps,
                ...options.documentationOptions?.plantuml,
            });
        }
        default:
            throw new Error(`Unknown documentation format ${format}`);
    }
};
const generateDocsProjects = (formats, options) => {
    if (formats.length > 0) {
        new projen_1.TextFile(options.parent, path.join(options.generatedDocsDir, "README.md"), {
            lines: [
                "## Generated Documentation",
                "",
                "This directory contains generated documentation based on your API model.",
            ],
            readonly: true,
        });
    }
    const generatedClients = {};
    formats.forEach((format) => {
        const project = generateDocsProject(format, options);
        if (project != null) {
            generatedClients[format] = project;
        }
    });
    return generatedClients;
};
exports.generateDocsProjects = generateDocsProjects;
//# sourceMappingURL=data:application/json;base64,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