"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.GeneratedJavaHandlersProject = void 0;
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const path = require("path");
const projen_1 = require("projen");
const java_1 = require("projen/lib/java");
const languages_1 = require("../../languages");
const open_api_generator_handlebars_ignore_file_1 = require("../components/open-api-generator-handlebars-ignore-file");
const open_api_generator_ignore_file_1 = require("../components/open-api-generator-ignore-file");
const open_api_tools_json_file_1 = require("../components/open-api-tools-json-file");
const type_safe_api_command_environment_1 = require("../components/type-safe-api-command-environment");
const utils_1 = require("../components/utils");
const runtime_version_utils_1 = require("../runtime-version-utils");
class GeneratedJavaHandlersProject extends java_1.JavaProject {
    constructor(options) {
        super({
            sample: false,
            junit: false,
            compileOptions: runtime_version_utils_1.RuntimeVersionUtils.JAVA.getMavenCompileOptions(options.runtimeVersion),
            ...options,
        });
        this.buildGenerateCommandArgs = () => {
            return (0, utils_1.buildInvokeOpenApiGeneratorCommandArgs)({
                generator: "java",
                specPath: this.options.specPath,
                generatorDirectory: utils_1.OtherGenerators.JAVA_LAMBDA_HANDLERS,
                srcDir: this.srcDir,
                tstDir: this.tstDir,
                normalizers: {
                    KEEP_ONLY_FIRST_TAG_IN_OPERATION: true,
                },
                extraVendorExtensions: {
                    "x-handlers-package": this.packageName,
                    "x-runtime-package": this.options.generatedJavaTypes.packageName,
                },
                // Do not generate map/list types. Generator will use built in HashMap, ArrayList instead
                generateAliasAsModel: false,
            });
        };
        type_safe_api_command_environment_1.TypeSafeApiCommandEnvironment.ensure(this);
        this.options = options;
        this.runtimeVersion = options.runtimeVersion ?? languages_1.JavaVersion.JAVA_17;
        this.packageName = `${this.pom.groupId}.${this.name}.handlers`;
        this.srcDir = path.join("src", "main", "java", ...this.packageName.split("."));
        this.tstDir = path.join("src", "test", "java", ...this.packageName.split("."));
        [
            `${options.generatedJavaTypes.pom.groupId}/${options.generatedJavaTypes.pom.artifactId}@${options.generatedJavaTypes.pom.version}`,
        ]
            .filter((dep) => !this.deps.tryGetDependency(dep.split("@")[0], projen_1.DependencyType.RUNTIME))
            .forEach((dep) => this.addDependency(dep));
        // Dependency on junit for tests
        this.deps.addDependency("org.junit.jupiter/junit-jupiter-engine@^5.10.0", projen_1.DependencyType.TEST);
        // Remove the projen test dependency since otherwise it takes precedence, causing projen to be unavailable at synth time
        this.deps.removeDependency("io.github.cdklabs/projen", projen_1.DependencyType.TEST);
        // Add a dependency on the generated java types repository
        this.pom.addRepository({
            url: `file://${path.relative(this.outdir, options.generatedJavaTypes.outdir)}/dist/java`,
            id: `${options.generatedJavaTypes.pom.groupId}-${options.generatedJavaTypes.pom.artifactId}-repo`,
        });
        // Ignore everything for the first mustache pass
        const openapiGeneratorIgnore = new open_api_generator_ignore_file_1.OpenApiGeneratorIgnoreFile(this);
        openapiGeneratorIgnore.addPatterns("/*", "**/*", "*");
        // Ignore everything but the handler files for the handlebars pass
        const openapiGeneratorHandlebarsIgnore = new open_api_generator_handlebars_ignore_file_1.OpenApiGeneratorHandlebarsIgnoreFile(this);
        openapiGeneratorHandlebarsIgnore.addPatterns("/*", "**/*", "*", 
        // This will be split into a file per targeted handler
        `!${this.srcDir}/__all_handlers.java`, `!${this.tstDir}/__all_tests.java`);
        // Add OpenAPI Generator cli configuration
        open_api_tools_json_file_1.OpenApiToolsJsonFile.ensure(this).addOpenApiGeneratorCliConfig(options.openApiGeneratorCliConfig);
        const generateTask = this.addTask("generate");
        generateTask.exec((0, utils_1.buildCleanOpenApiGeneratedCodeCommand)());
        generateTask.exec((0, utils_1.buildTypeSafeApiExecCommand)(utils_1.TypeSafeApiScript.GENERATE, this.buildGenerateCommandArgs()));
        this.preCompileTask.spawn(generateTask);
        // Ignore the openapi generator metadata files
        this.gitignore.addPatterns(".openapi-generator");
        // Use the maven shade plugin to build a "super jar" which we can deploy to AWS Lambda
        this.pom.addPlugin("org.apache.maven.plugins/maven-shade-plugin@3.3.0", {
            configuration: {
                createDependencyReducedPom: false,
                transformers: [
                    {
                        // Transformer required for merging log4j2 plugins cache file
                        // https://docs.aws.amazon.com/lambda/latest/dg/java-logging.html#java-logging-cdk
                        transformer: {
                            "@implementation": "com.github.edwgiz.maven_shade_plugin.log4j2_cache_transformer.PluginsCacheFileTransformer",
                        },
                    },
                ],
            },
            executions: [
                {
                    id: "shade-task",
                    phase: "package",
                    goals: ["shade"],
                },
            ],
            dependencies: [
                "com.github.edwgiz/maven-shade-plugin.log4j2-cachefile-transformer@2.15.0",
            ],
        });
        // Use the maven surefire plugin to run junit tests
        this.pom.addPlugin("org.apache.maven.plugins/maven-surefire-plugin@3.2.1");
        // Log4j2 configuration for powertools logger
        new projen_1.SampleDir(this, "src/main/resources", {
            files: {
                "log4j2.xml": `<?xml version="1.0" encoding="UTF-8"?>
<Configuration>
    <Appenders>
        <Console name="JsonAppender" target="SYSTEM_OUT">
            <JsonTemplateLayout eventTemplateUri="classpath:LambdaJsonLayout.json" />
        </Console>
    </Appenders>
    <Loggers>
        <Logger name="JsonLogger" level="INFO" additivity="false">
            <AppenderRef ref="JsonAppender"/>
        </Logger>
        <Root level="info">
            <AppenderRef ref="JsonAppender"/>
        </Root>
    </Loggers>
</Configuration>`,
            },
        });
    }
}
exports.GeneratedJavaHandlersProject = GeneratedJavaHandlersProject;
//# sourceMappingURL=data:application/json;base64,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