"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.GeneratedPythonHandlersProject = void 0;
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const path = require("path");
const projen_1 = require("projen");
const python_1 = require("projen/lib/python");
const languages_1 = require("../../languages");
const types_1 = require("../../types");
const open_api_generator_handlebars_ignore_file_1 = require("../components/open-api-generator-handlebars-ignore-file");
const open_api_generator_ignore_file_1 = require("../components/open-api-generator-ignore-file");
const open_api_tools_json_file_1 = require("../components/open-api-tools-json-file");
const type_safe_api_command_environment_1 = require("../components/type-safe-api-command-environment");
const utils_1 = require("../components/utils");
const runtime_version_utils_1 = require("../runtime-version-utils");
class GeneratedPythonHandlersProject extends python_1.PythonProject {
    constructor(options) {
        super({
            pytest: true,
            poetryOptions: {
                packages: [{ include: options.moduleName }],
                // Module must be explicitly added to include since poetry excludes everything in .gitignore by default
                include: [options.moduleName, `${options.moduleName}/**/*.py`],
            },
            ...options,
            sample: false,
            poetry: true,
        });
        this.buildGenerateCommandArgs = () => {
            return (0, utils_1.buildInvokeOpenApiGeneratorCommandArgs)({
                generator: "python-nextgen",
                specPath: this.options.specPath,
                generatorDirectory: utils_1.OtherGenerators.PYTHON_LAMBDA_HANDLERS,
                // Tell the generator where python source files live
                srcDir: this.moduleName,
                tstDir: this.tstDir,
                additionalProperties: {
                    packageName: this.moduleName,
                    projectName: this.name,
                },
                normalizers: {
                    KEEP_ONLY_FIRST_TAG_IN_OPERATION: true,
                },
                extraVendorExtensions: {
                    "x-runtime-module-name": this.options.generatedPythonTypes.moduleName,
                },
            });
        };
        type_safe_api_command_environment_1.TypeSafeApiCommandEnvironment.ensure(this);
        this.options = options;
        this.runtimeVersion = options.runtimeVersion ?? languages_1.PythonVersion.PYTHON_3_11;
        this.tstDir = "test";
        if (options.pytest ?? true) {
            // Pytest fails with exit code 5 when there are no tests.
            // We want to allow users to delete all their tests, or to upgrade an existing project without breaking their build
            // See: https://github.com/pytest-dev/pytest/issues/2393
            this.testTask.reset(`pytest ${this.tstDir}/ || ([ $? = 5 ] && exit 0 || exit $?)`);
        }
        [
            runtime_version_utils_1.RuntimeVersionUtils.PYTHON.getPythonDependencyVersion(this.runtimeVersion),
            `${options.generatedPythonTypes.name}@{path="${path.relative(this.outdir, options.generatedPythonTypes.outdir)}", develop=true}`,
        ]
            .filter((dep) => !this.deps.tryGetDependency(dep, projen_1.DependencyType.RUNTIME))
            .forEach((dep) => this.addDependency(dep));
        // Ignore everything for the first mustache pass
        const openapiGeneratorIgnore = new open_api_generator_ignore_file_1.OpenApiGeneratorIgnoreFile(this);
        openapiGeneratorIgnore.addPatterns("/*", "**/*", "*");
        // Ignore everything but the handler files for the handlebars pass
        const openapiGeneratorHandlebarsIgnore = new open_api_generator_handlebars_ignore_file_1.OpenApiGeneratorHandlebarsIgnoreFile(this);
        openapiGeneratorHandlebarsIgnore.addPatterns("/*", "**/*", "*", 
        // This will be split into a file per targeted handler
        `!${this.moduleName}/__all_handlers.py`, `!${this.tstDir}/__all_tests.py`);
        // Add OpenAPI Generator cli configuration
        open_api_tools_json_file_1.OpenApiToolsJsonFile.ensure(this).addOpenApiGeneratorCliConfig({
            version: "6.6.0",
            ...options.openApiGeneratorCliConfig,
        });
        const generateTask = this.addTask("generate");
        generateTask.exec((0, utils_1.buildCleanOpenApiGeneratedCodeCommand)());
        generateTask.exec((0, utils_1.buildTypeSafeApiExecCommand)(utils_1.TypeSafeApiScript.GENERATE, this.buildGenerateCommandArgs()));
        this.preCompileTask.spawn(generateTask);
        // Ignore the generated code
        this.gitignore.addPatterns(".openapi-generator");
        // Write __init__.py as sample code
        new projen_1.SampleFile(this, path.join(this.moduleName, "__init__.py"), {
            contents: "#",
        });
        // Package task to build a distributable which can be deployed to lambda
        this.packageTask.exec("mkdir -p dist/lambda && rm -rf dist/lambda/*");
        this.packageTask.exec(`cp -r ${this.moduleName} dist/lambda/${this.moduleName}`);
        this.packageTask.exec("poetry export --without-hashes --format=requirements.txt > dist/lambda/requirements.txt");
        // Select the platform based on the specified architecture, defaulting to x86_64
        // See: https://docs.aws.amazon.com/lambda/latest/dg/python-package.html#python-package-native-libraries
        const platform = options.architecture === types_1.Architecture.ARM_64
            ? "manylinux2014_aarch64"
            : "manylinux2014_x86_64";
        this.packageTask.exec(`pip install -r dist/lambda/requirements.txt --target dist/lambda --upgrade --platform ${platform} --only-binary :all: --python-version ${runtime_version_utils_1.RuntimeVersionUtils.PYTHON.getPipPackagingPythonVersion(this.runtimeVersion)}`);
    }
}
exports.GeneratedPythonHandlersProject = GeneratedPythonHandlersProject;
//# sourceMappingURL=data:application/json;base64,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