"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.GeneratedTypescriptHandlersProject = void 0;
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const path = require("path");
const projen_1 = require("projen");
const javascript_1 = require("projen/lib/javascript");
const typescript_1 = require("projen/lib/typescript");
const languages_1 = require("../../languages");
const open_api_generator_handlebars_ignore_file_1 = require("../components/open-api-generator-handlebars-ignore-file");
const open_api_generator_ignore_file_1 = require("../components/open-api-generator-ignore-file");
const open_api_tools_json_file_1 = require("../components/open-api-tools-json-file");
const type_safe_api_command_environment_1 = require("../components/type-safe-api-command-environment");
const utils_1 = require("../components/utils");
const runtime_version_utils_1 = require("../runtime-version-utils");
class GeneratedTypescriptHandlersProject extends typescript_1.TypeScriptProject {
    constructor(options) {
        super({
            ...options,
            sampleCode: false,
            tsconfig: {
                compilerOptions: {
                    lib: ["dom", "es2019"],
                    // Generated code imports all models, and may not reference them all
                    noUnusedLocals: false,
                    noUnusedParameters: false,
                    skipLibCheck: true,
                    ...options?.tsconfig?.compilerOptions,
                },
            },
            npmignoreEnabled: false,
        });
        this.buildGenerateCommandArgs = () => {
            return (0, utils_1.buildInvokeOpenApiGeneratorCommandArgs)({
                generator: "typescript-fetch",
                specPath: this.options.specPath,
                generatorDirectory: utils_1.OtherGenerators.TYPESCRIPT_LAMBDA_HANDLERS,
                srcDir: this.srcdir,
                tstDir: this.testdir,
                normalizers: {
                    KEEP_ONLY_FIRST_TAG_IN_OPERATION: true,
                },
                extraVendorExtensions: {
                    "x-runtime-package-name": this.options.generatedTypescriptTypes.package.packageName,
                },
            });
        };
        this.options = options;
        this.runtimeVersion = options.runtimeVersion ?? languages_1.NodeVersion.NODE_18;
        type_safe_api_command_environment_1.TypeSafeApiCommandEnvironment.ensure(this);
        this.addDeps(...[
            // If within a monorepo, add a regular dependency. Otherwise, use a file dependency to ensure the runtime can be
            // resolved
            options.isWithinMonorepo
                ? `${options.generatedTypescriptTypes.package.packageName}@${options.generatedTypescriptTypes.package.manifest.version}`
                : `${options.generatedTypescriptTypes.package.packageName}@file:${path.relative(this.outdir, options.generatedTypescriptTypes.outdir)}`,
        ].filter((dep) => !this.deps.tryGetDependency(dep, projen_1.DependencyType.RUNTIME)));
        this.addDevDeps("esbuild", "@types/aws-lambda");
        // Minimal .npmignore to avoid impacting OpenAPI Generator
        const npmignore = new projen_1.IgnoreFile(this, ".npmignore");
        npmignore.addPatterns("/.projen/", "/src", "/dist");
        // Ignore everything for the regular open api generator pass
        const openapiGeneratorIgnore = new open_api_generator_ignore_file_1.OpenApiGeneratorIgnoreFile(this);
        openapiGeneratorIgnore.addPatterns("/*", "**/*", "*");
        // Ignore everything but the handler files for the handlebars pass
        const openapiGeneratorHandlebarsIgnore = new open_api_generator_handlebars_ignore_file_1.OpenApiGeneratorHandlebarsIgnoreFile(this);
        openapiGeneratorHandlebarsIgnore.addPatterns("/*", "**/*", "*", 
        // This will be split into a file per targeted handler
        `!${this.srcdir}/__all_handlers.ts`, `!${this.testdir}/__all_tests.ts`);
        // Add OpenAPI Generator cli configuration
        open_api_tools_json_file_1.OpenApiToolsJsonFile.ensure(this).addOpenApiGeneratorCliConfig(options.openApiGeneratorCliConfig);
        const generateTask = this.addTask("generate");
        generateTask.exec((0, utils_1.buildCleanOpenApiGeneratedCodeCommand)());
        generateTask.exec((0, utils_1.buildTypeSafeApiExecCommand)(utils_1.TypeSafeApiScript.GENERATE, this.buildGenerateCommandArgs()));
        this.preCompileTask.spawn(generateTask);
        // Ignore the openapi generator metadata
        this.gitignore.addPatterns(".openapi-generator");
        // Create a separate lambda bundle for each handler as part of the package task.
        // Note that every typescript file directly in src is bundled by default, but users may specify their own
        // entry point globs if they prefer a different directory structure.
        this.packageTask.exec(`mkdir -p dist/lambda && rm -rf dist/lambda/*`);
        this.packageTask.exec(`esbuild --bundle ${(options.handlerEntryPoints ?? [`${this.srcdir}/*.ts`]).join(" ")} --platform=node --outdir=dist/lambda --target=${runtime_version_utils_1.RuntimeVersionUtils.NODE.getEsbuildNodeTarget(this.runtimeVersion)}`);
        // Move each bundled file into a separate directory
        this.packageTask.exec("for f in $(ls dist/lambda); do mkdir dist/lambda/$(basename $f .js) && mv dist/lambda/$f dist/lambda/$(basename $f .js)/index.js; done");
        // Create an empty index.ts sample on synth so that tsc is happy if the handlers project is configured
        // but no operations have @handler(language: "typescript")
        new projen_1.SampleDir(this, this.srcdir, {
            files: {
                "index.ts": "",
            },
        });
        // If we're not in a monorepo, we need to link the generated types such that the local dependency can be resolved
        if (!options.isWithinMonorepo) {
            switch (this.package.packageManager) {
                case javascript_1.NodePackageManager.PNPM:
                    this.tasks
                        .tryFind("install")
                        ?.prependExec(`${this.package.packageManager} link /${path.relative(this.outdir, this.options.generatedTypescriptTypes.outdir)}`);
                    break;
                default:
                    this.tasks
                        .tryFind("install")
                        ?.prependExec(`${this.package.packageManager} link ${this.options.generatedTypescriptTypes.package.packageName}`);
                    break;
            }
        }
    }
}
exports.GeneratedTypescriptHandlersProject = GeneratedTypescriptHandlersProject;
//# sourceMappingURL=data:application/json;base64,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