"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.GeneratedTypescriptCdkInfrastructureProject = void 0;
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const path = require("path");
const projen_1 = require("projen");
const javascript_1 = require("projen/lib/javascript");
const typescript_1 = require("projen/lib/typescript");
const open_api_generator_handlebars_ignore_file_1 = require("../../components/open-api-generator-handlebars-ignore-file");
const open_api_generator_ignore_file_1 = require("../../components/open-api-generator-ignore-file");
const open_api_tools_json_file_1 = require("../../components/open-api-tools-json-file");
const type_safe_api_command_environment_1 = require("../../components/type-safe-api-command-environment");
const utils_1 = require("../../components/utils");
class GeneratedTypescriptCdkInfrastructureProject extends typescript_1.TypeScriptProject {
    constructor(options) {
        super({
            ...options,
            sampleCode: false,
            jest: false,
            eslint: false,
            prettier: false,
            tsconfig: {
                compilerOptions: {
                    lib: ["dom", "es2019"],
                    // Generated code imports all models, and may not reference them all
                    noUnusedLocals: false,
                    noUnusedParameters: false,
                    skipLibCheck: true,
                    ...options?.tsconfig?.compilerOptions,
                },
            },
            npmignoreEnabled: false,
        });
        /**
         * Path to the packaged copy of the openapi specification
         * @private
         */
        this.packagedSpecPath = "assets/api.json";
        this.buildGenerateCommandArgs = () => {
            return (0, utils_1.buildInvokeOpenApiGeneratorCommandArgs)({
                generator: "typescript-fetch",
                specPath: this.options.specPath,
                generatorDirectory: utils_1.OtherGenerators.TYPESCRIPT_CDK_INFRASTRUCTURE,
                srcDir: this.srcdir,
                normalizers: {
                    KEEP_ONLY_FIRST_TAG_IN_OPERATION: true,
                },
                extraVendorExtensions: {
                    "x-runtime-package-name": this.options.generatedTypescriptTypes.package.packageName,
                    // Spec path relative to the source directory
                    "x-relative-spec-path": path.join("..", this.packagedSpecPath),
                    // Enable mock integration generation by default
                    "x-enable-mock-integrations": !this.options.mockDataOptions?.disable,
                    ...(0, utils_1.getHandlersProjectVendorExtensions)(this, this.options.generatedHandlers),
                },
            });
        };
        this.buildGenerateMockDataCommand = () => {
            return (0, utils_1.buildInvokeMockDataGeneratorCommand)({
                specPath: this.options.specPath,
                ...this.options.mockDataOptions,
            });
        };
        type_safe_api_command_environment_1.TypeSafeApiCommandEnvironment.ensure(this);
        this.options = options;
        this.addDevDeps("@types/aws-lambda");
        this.addDeps(...[
            // If within a monorepo, add a regular dependency. Otherwise, use a file dependency to ensure the types can be
            // resolved
            options.isWithinMonorepo
                ? `${options.generatedTypescriptTypes.package.packageName}@${options.generatedTypescriptTypes.package.manifest.version}`
                : `${options.generatedTypescriptTypes.package.packageName}@file:${path.relative(this.outdir, options.generatedTypescriptTypes.outdir)}`,
        ].filter((dep) => !this.deps.tryGetDependency(dep, projen_1.DependencyType.RUNTIME)));
        const devAndPeerDeps = ["@aws/pdk", "constructs", "aws-cdk-lib", "cdk-nag"];
        this.addDevDeps(...devAndPeerDeps.filter((dep) => !this.deps.tryGetDependency(dep, projen_1.DependencyType.BUILD)));
        this.addPeerDeps(...devAndPeerDeps.filter((dep) => !this.deps.tryGetDependency(dep, projen_1.DependencyType.PEER)));
        // Minimal .npmignore to avoid impacting OpenAPI Generator
        const npmignore = new projen_1.IgnoreFile(this, ".npmignore");
        npmignore.addPatterns("/.projen/", "/src", "/dist");
        // Ignore everything but the target files
        const openapiGeneratorIgnore = new open_api_generator_ignore_file_1.OpenApiGeneratorIgnoreFile(this);
        openapiGeneratorIgnore.addPatterns("/*", "**/*", "*", `!${this.srcdir}/index.ts`, `!${this.srcdir}/api.ts`, `!${this.srcdir}/mock-integrations.ts`);
        const openapiGeneratorHandlebarsIgnore = new open_api_generator_handlebars_ignore_file_1.OpenApiGeneratorHandlebarsIgnoreFile(this);
        openapiGeneratorHandlebarsIgnore.addPatterns("/*", "**/*", "*", `!${this.srcdir}/__functions.ts`);
        // Add OpenAPI Generator cli configuration
        open_api_tools_json_file_1.OpenApiToolsJsonFile.ensure(this).addOpenApiGeneratorCliConfig(options.openApiGeneratorCliConfig);
        const generateTask = this.addTask("generate");
        generateTask.exec((0, utils_1.buildCleanOpenApiGeneratedCodeCommand)());
        generateTask.exec((0, utils_1.buildTypeSafeApiExecCommand)(utils_1.TypeSafeApiScript.GENERATE, this.buildGenerateCommandArgs()));
        if (!this.options.mockDataOptions?.disable) {
            generateTask.exec(this.buildGenerateMockDataCommand());
        }
        // Copy the api spec to within the package
        generateTask.exec(`mkdir -p ${path.dirname(this.packagedSpecPath)}`);
        generateTask.exec(`cp -f ${this.options.specPath} ${this.packagedSpecPath}`);
        this.gitignore.addPatterns(`/${this.packagedSpecPath}`);
        this.preCompileTask.spawn(generateTask);
        // Ignore the generated code
        this.gitignore.addPatterns(this.srcdir, ".openapi-generator", "mocks");
        // If we're not in a monorepo, we need to link the generated types such that the local dependency can be resolved
        if (!options.isWithinMonorepo) {
            switch (this.package.packageManager) {
                case javascript_1.NodePackageManager.PNPM:
                    this.tasks
                        .tryFind("install")
                        ?.prependExec(`${this.package.packageManager} link /${path.relative(this.outdir, this.options.generatedTypescriptTypes.outdir)}`);
                    break;
                default:
                    this.tasks
                        .tryFind("install")
                        ?.prependExec(`${this.package.packageManager} link ${this.options.generatedTypescriptTypes.package.packageName}`);
                    break;
            }
        }
    }
}
exports.GeneratedTypescriptCdkInfrastructureProject = GeneratedTypescriptCdkInfrastructureProject;
//# sourceMappingURL=data:application/json;base64,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