"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.TypescriptReactQueryHooksLibrary = void 0;
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const monorepo_1 = require("../../../../monorepo");
const javascript_1 = require("projen/lib/javascript");
const typescript_1 = require("projen/lib/typescript");
const languages_1 = require("../../languages");
const open_api_generator_handlebars_ignore_file_1 = require("../components/open-api-generator-handlebars-ignore-file");
const open_api_generator_ignore_file_1 = require("../components/open-api-generator-ignore-file");
const open_api_tools_json_file_1 = require("../components/open-api-tools-json-file");
const type_safe_api_command_environment_1 = require("../components/type-safe-api-command-environment");
const utils_1 = require("../components/utils");
/**
 * Typescript project containing generated react-query hooks
 */
class TypescriptReactQueryHooksLibrary extends typescript_1.TypeScriptProject {
    constructor(options) {
        super({
            ...options,
            sampleCode: false,
            tsconfig: {
                ...options.tsconfig,
                compilerOptions: {
                    jsx: javascript_1.TypeScriptJsxMode.REACT,
                    lib: ["dom", "es2019"],
                    // Generated code isn't very strict!
                    strict: false,
                    alwaysStrict: false,
                    noImplicitAny: false,
                    noImplicitReturns: false,
                    noImplicitThis: false,
                    noUnusedLocals: false,
                    noUnusedParameters: false,
                    strictNullChecks: false,
                    strictPropertyInitialization: false,
                    skipLibCheck: true,
                    ...options?.tsconfig?.compilerOptions,
                },
            },
            eslint: false,
            // Disable tests unless explicitly enabled
            jest: options.jest ?? false,
            npmignoreEnabled: false,
        });
        this.buildGenerateCommandArgs = () => {
            return (0, utils_1.buildInvokeOpenApiGeneratorCommandArgs)({
                generator: "typescript-fetch",
                specPath: this.options.specPath,
                generatorDirectory: languages_1.Library.TYPESCRIPT_REACT_QUERY_HOOKS,
                additionalProperties: {
                    npmName: this.package.packageName,
                    typescriptThreePlus: "true",
                    useSingleParameter: "true",
                    supportsES6: "true",
                },
                srcDir: this.srcdir,
                normalizers: {
                    KEEP_ONLY_FIRST_TAG_IN_OPERATION: true,
                },
            });
        };
        type_safe_api_command_environment_1.TypeSafeApiCommandEnvironment.ensure(this);
        this.options = options;
        // Disable strict peer dependencies for pnpm as the default typescript project dependencies have type mismatches
        // (ts-jest@27 and @types/jest@28)
        if (this.package.packageManager === javascript_1.NodePackageManager.PNPM) {
            this.npmrc.addConfig("strict-peer-dependencies", "false");
        }
        // Add dependencies on react-query and react
        this.addDeps("@tanstack/react-query@^4"); // Pin at 4 for now - requires generated code updates to upgrade to 5
        this.addDevDeps("react", "@types/react");
        this.addPeerDeps("react");
        // Tell OpenAPI Generator CLI not to generate files that we will generate via this project, or don't need.
        const openapiGeneratorIgnore = new open_api_generator_ignore_file_1.OpenApiGeneratorIgnoreFile(this);
        openapiGeneratorIgnore.addPatterns(...TypescriptReactQueryHooksLibrary.openApiIgnorePatterns);
        // Ignore the hooks, since they're generated with the handlebars templating engine
        const hooksPattern = "**/*Hooks.ts";
        openapiGeneratorIgnore.addPatterns(hooksPattern);
        // The hooks are generated using the handlebars templating engine, so we include a handlebars ignore file
        const handlebarsIgnore = new open_api_generator_handlebars_ignore_file_1.OpenApiGeneratorHandlebarsIgnoreFile(this);
        handlebarsIgnore.addPatterns(`!${hooksPattern}`);
        // Add OpenAPI Generator cli configuration
        open_api_tools_json_file_1.OpenApiToolsJsonFile.ensure(this).addOpenApiGeneratorCliConfig(options.openApiGeneratorCliConfig);
        const generateTask = this.addTask("generate");
        generateTask.exec((0, utils_1.buildCleanOpenApiGeneratedCodeCommand)());
        generateTask.exec((0, utils_1.buildTypeSafeApiExecCommand)(utils_1.TypeSafeApiScript.GENERATE, this.buildGenerateCommandArgs()));
        this.preCompileTask.spawn(generateTask);
        // Ignore all the generated code
        this.gitignore.addPatterns("src", ".npmignore", "README.md", ".openapi-generator");
        // If we're not in a monorepo, we need to link the generated client such that any local dependency on it can be
        // resolved
        if (!options.isWithinMonorepo) {
            switch (this.package.packageManager) {
                case javascript_1.NodePackageManager.PNPM:
                    // Nothing to do for pnpm, since the pnpm link command handles both the dependant and dependee
                    break;
                default:
                    this.tasks
                        .tryFind("install")
                        ?.exec(monorepo_1.NodePackageUtils.command.cmd(this.package.packageManager, "link"));
                    break;
            }
        }
    }
}
exports.TypescriptReactQueryHooksLibrary = TypescriptReactQueryHooksLibrary;
/**
 * Patterns that are excluded from code generation
 */
TypescriptReactQueryHooksLibrary.openApiIgnorePatterns = [
    "package.json",
    "tsconfig.json",
    "tsconfig.esm.json",
    ".npmignore",
];
//# sourceMappingURL=data:application/json;base64,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