"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.GeneratedJavaRuntimeProject = void 0;
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const path = require("path");
const java_1 = require("projen/lib/java");
const languages_1 = require("../../languages");
const open_api_generator_ignore_file_1 = require("../components/open-api-generator-ignore-file");
const open_api_tools_json_file_1 = require("../components/open-api-tools-json-file");
const type_safe_api_command_environment_1 = require("../components/type-safe-api-command-environment");
const utils_1 = require("../components/utils");
const DEPENDENCIES = [
    // Required for open api generated code
    "io.swagger/swagger-annotations@1.6.8",
    "com.google.code.findbugs/jsr305@3.0.2",
    "com.squareup.okhttp3/okhttp@4.10.0",
    "com.squareup.okhttp3/logging-interceptor@4.10.0",
    "com.google.code.gson/gson@2.9.1",
    "io.gsonfire/gson-fire@1.8.5",
    "org.apache.commons/commons-lang3@3.12.0",
    "jakarta.annotation/jakarta.annotation-api@1.3.5",
    "org.openapitools/jackson-databind-nullable@0.2.4",
    "javax.ws.rs/jsr311-api@1.1.1",
    "javax.ws.rs/javax.ws.rs-api@2.1.1",
    // For handler wrappers
    "com.amazonaws/aws-lambda-java-core@1.2.1",
    "com.amazonaws/aws-lambda-java-events@3.11.0",
    // Lombok is used to add the builder pattern to models for neater construction
    "org.projectlombok/lombok@1.18.24",
    // Interceptors
    "software.amazon.lambda/powertools-logging@^1.16.1",
    "software.amazon.lambda/powertools-tracing@^1.16.1",
    "software.amazon.lambda/powertools-metrics@^1.16.1",
    // SnapStart
    "io.github.crac/org-crac@^0.1.3",
];
const TEST_DEPENDENCIES = [
    "org.junit.jupiter/junit-jupiter-api@5.9.1",
    "org.mockito/mockito-core@3.12.4",
];
/**
 * Java project containing types generated using OpenAPI Generator CLI
 */
class GeneratedJavaRuntimeProject extends java_1.JavaProject {
    constructor(options) {
        super({
            ...options,
            sample: false,
            junit: false,
        });
        this.buildGenerateCommandArgs = () => {
            return (0, utils_1.buildInvokeOpenApiGeneratorCommandArgs)({
                generator: "java",
                specPath: this.options.specPath,
                generatorDirectory: languages_1.Language.JAVA,
                additionalProperties: {
                    useSingleRequestParameter: "true",
                    groupId: this.pom.groupId,
                    artifactId: this.pom.artifactId,
                    artifactVersion: this.pom.version,
                    invokerPackage: this.packageName,
                    apiPackage: `${this.packageName}.api`,
                    modelPackage: `${this.packageName}.model`,
                    hideGenerationTimestamp: "true",
                    additionalModelTypeAnnotations: [
                        "@lombok.AllArgsConstructor",
                        // Regular lombok builder is not used since an abstract base schema class is also annotated
                        "@lombok.experimental.SuperBuilder",
                    ].join("\\ "),
                },
                srcDir: path.join("src", "main", "java", ...this.packageName.split(".")),
                normalizers: {
                    KEEP_ONLY_FIRST_TAG_IN_OPERATION: true,
                },
                // Do not generate map/list types. Generator will use built in HashMap, ArrayList instead
                generateAliasAsModel: false,
            });
        };
        type_safe_api_command_environment_1.TypeSafeApiCommandEnvironment.ensure(this);
        this.options = options;
        // Ignore files that we will control via projen
        const ignoreFile = new open_api_generator_ignore_file_1.OpenApiGeneratorIgnoreFile(this);
        ignoreFile.addPatterns(...GeneratedJavaRuntimeProject.openApiIgnorePatterns);
        // Add OpenAPI Generator cli configuration
        open_api_tools_json_file_1.OpenApiToolsJsonFile.ensure(this).addOpenApiGeneratorCliConfig(options.openApiGeneratorCliConfig);
        // Add dependencies
        DEPENDENCIES.forEach((dep) => this.addDependency(dep));
        TEST_DEPENDENCIES.forEach((dep) => this.addTestDependency(dep));
        this.packageName = `${this.pom.groupId}.${this.name}.runtime`;
        // Generate the java code
        const generateTask = this.addTask("generate");
        generateTask.exec((0, utils_1.buildCleanOpenApiGeneratedCodeCommand)());
        generateTask.exec((0, utils_1.buildTypeSafeApiExecCommand)(utils_1.TypeSafeApiScript.GENERATE, this.buildGenerateCommandArgs()));
        this.preCompileTask.spawn(generateTask);
        // Ignore all the generated code
        this.gitignore.addPatterns("src", "docs", "api", "README.md", ".openapi-generator");
    }
}
exports.GeneratedJavaRuntimeProject = GeneratedJavaRuntimeProject;
/**
 * Patterns that are excluded from code generation
 */
GeneratedJavaRuntimeProject.openApiIgnorePatterns = [
    "pom.xml",
    "gradle",
    "gradle/**/*",
    "gradlew",
    "gradle.properties",
    "gradlew.bat",
    "build.gradle",
    "settings.gradle",
    "build.sbt",
    ".travis.yml",
    "git_push.sh",
    "src/test",
    "src/test/**/*",
    "src/main/AndroidManifest.xml",
];
//# sourceMappingURL=data:application/json;base64,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