"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.GeneratedPythonRuntimeProject = void 0;
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const python_1 = require("projen/lib/python");
const languages_1 = require("../../languages");
const open_api_generator_ignore_file_1 = require("../components/open-api-generator-ignore-file");
const open_api_tools_json_file_1 = require("../components/open-api-tools-json-file");
const type_safe_api_command_environment_1 = require("../components/type-safe-api-command-environment");
const utils_1 = require("../components/utils");
/**
 * Python project containing types generated using OpenAPI Generator CLI
 */
class GeneratedPythonRuntimeProject extends python_1.PythonProject {
    constructor(options) {
        super({
            ...options,
            sample: false,
            pytest: false,
            poetry: true,
            poetryOptions: {
                packages: [{ include: options.moduleName }],
                // Module must be explicitly added to include since poetry excludes everything in .gitignore by default
                include: [options.moduleName, `${options.moduleName}/**/*.py`],
            },
        });
        this.buildGenerateCommandArgs = () => {
            return (0, utils_1.buildInvokeOpenApiGeneratorCommandArgs)({
                generator: "python",
                specPath: this.options.specPath,
                generatorDirectory: languages_1.Language.PYTHON,
                additionalProperties: {
                    packageName: this.moduleName,
                    projectName: this.name,
                },
                // Tell the generator where python source files live
                srcDir: this.moduleName,
                normalizers: {
                    KEEP_ONLY_FIRST_TAG_IN_OPERATION: true,
                },
            });
        };
        type_safe_api_command_environment_1.TypeSafeApiCommandEnvironment.ensure(this);
        this.options = options;
        // Add dependencies required by the client
        [
            "python-dateutil@~2.8.2",
            "pydantic@^2.5.2",
            "aenum@^3.1.11",
            "urllib3@~1.26.7",
            `aws-lambda-powertools@{extras=["tracer", "aws-sdk"],version="^2.28.0"}`,
            "python@^3.9",
        ].forEach((dep) => this.addDependency(dep));
        const openapiGeneratorIgnore = new open_api_generator_ignore_file_1.OpenApiGeneratorIgnoreFile(this);
        openapiGeneratorIgnore.addPatterns(...GeneratedPythonRuntimeProject.openApiIgnorePatterns);
        // Add OpenAPI Generator cli configuration
        open_api_tools_json_file_1.OpenApiToolsJsonFile.ensure(this).addOpenApiGeneratorCliConfig({
            version: "7.1.0",
            ...options.openApiGeneratorCliConfig,
        });
        const generateTask = this.addTask("generate");
        generateTask.exec((0, utils_1.buildCleanOpenApiGeneratedCodeCommand)());
        generateTask.exec((0, utils_1.buildTypeSafeApiExecCommand)(utils_1.TypeSafeApiScript.GENERATE, this.buildGenerateCommandArgs()));
        this.preCompileTask.spawn(generateTask);
        // Ignore all the generated code
        this.gitignore.addPatterns(this.moduleName, "docs", "README.md", ".openapi-generator");
        // The poetry install that runs as part of post synthesis expects there to be some code present, but code isn't
        // generated until build time. This means that the first install will fail when either generating the project for
        // the first time or checking out a fresh copy (since generated code is not checked in to version control). We
        // therefore add a blank __init__.py as our first install step to keep poetry happy until the generator overwrites
        // it.
        this.tasks
            .tryFind("install")
            ?.prependExec(`mkdir -p ${this.moduleName} && touch ${this.moduleName}/__init__.py`);
    }
}
exports.GeneratedPythonRuntimeProject = GeneratedPythonRuntimeProject;
/**
 * Patterns that are excluded from code generation
 */
GeneratedPythonRuntimeProject.openApiIgnorePatterns = [
    "test",
    "test/*",
    "test/**/*",
    ".github",
    ".github/*",
    ".github/**/*",
    ".gitlab-ci.yml",
    ".travis.yml",
    "git_push.sh",
    "tox.ini",
    "requirements.txt",
    "test-requirements.txt",
    "setup.py",
    "setup.cfg",
    "pyproject.toml",
];
//# sourceMappingURL=data:application/json;base64,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