"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SmithyAwsPdkPrelude = void 0;
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const path = require("path");
const projen_1 = require("projen");
/**
 * Synthesize the aws-pdk/prelude.smithy file
 */
class SmithyAwsPdkPrelude extends projen_1.FileBase {
    constructor(project, options) {
        super(project, path.join(options.generatedModelDir, "aws-pdk", "prelude.smithy"), {
            marker: true,
            readonly: true,
        });
        this.options = options;
    }
    synthesizeContent(_) {
        const hasAnyHandlerProjects = (this.options.handlerLanguages ?? []).length > 0;
        const traitValidator = hasAnyHandlerProjects
            ? `{
        id: "ConfiguredHandlerProject"
        name: "EmitEachSelector"
        configuration: {
            bindToTrait: ${this.options.serviceNamespace}#handler
            selector: """
                [@trait|${this.options.serviceNamespace}#handler: @{language} = typescript, java, python]
                :not([@trait|${this.options.serviceNamespace}#handler: @{language} = ${this.options.handlerLanguages?.join(", ")}])
            """
            messageTemplate: """
                @@handler language @{trait|${this.options.serviceNamespace}#handler|language} cannot be referenced unless a handler project is configured for this language.
                Configured handler project languages are: ${this.options.handlerLanguages?.join(", ")}.
                You can add this language by configuring TypeSafeApiProject in your .projenrc
            """
        }
    }`
            : `{
        id: "TraitNotPermitted"
        name: "EmitEachSelector"
        configuration: {
            bindToTrait: ${this.options.serviceNamespace}#handler
            selector: """
                *
            """
            messageTemplate: """
                @@handler trait cannot be used unless handler project languages have been configured.
                You can add handler projects by configuring TypeSafeApiProject in your .projenrc
            """
        }
    }`;
        return `// ${this.marker}

$version: "2"

metadata validators = [
    {
        id: "SupportedLanguage"
        name: "EmitEachSelector"
        configuration: {
            bindToTrait: ${this.options.serviceNamespace}#handler
            selector: """
                :not([@trait|${this.options.serviceNamespace}#handler: @{language} = typescript, java, python])
            """
            messageTemplate: """
                @{trait|${this.options.serviceNamespace}#handler|language} is not supported by type-safe-api.
                Supported languages are "typescript", "java" and "python".
            """
        }
    }
    ${traitValidator}
]

namespace ${this.options.serviceNamespace}

/// Add this trait to an operation to generate a lambda handler stub for the operation.
/// ${hasAnyHandlerProjects
            ? `You have configured handler projects for ${this.options.handlerLanguages?.join(", ")}`
            : "You have not configured any handler projects, so you cannot use this trait."}
@trait(selector: "operation")
structure handler {
    /// The language you will implement the lambda in.
    /// Valid values: typescript, java, python
    @required
    language: String
}

`;
    }
}
exports.SmithyAwsPdkPrelude = SmithyAwsPdkPrelude;
//# sourceMappingURL=data:application/json;base64,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