"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SmithyBuildGradleFile = void 0;
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const projen_1 = require("projen");
/**
 * Synthesize a build.gradle file for Smithy projects
 */
class SmithyBuildGradleFile extends projen_1.FileBase {
    constructor(project, options) {
        super(project, "build.gradle", {
            marker: true,
            readonly: true,
        });
        this.renderDependency = (dep) => {
            if (dep.startsWith(SmithyBuildGradleFile.fileDependencyPrefix)) {
                return `files("${dep.substring(SmithyBuildGradleFile.fileDependencyPrefix.length)}")`;
            }
            return `"${dep}"`;
        };
        this.sources = [options.modelDir];
        this.dependencies = options.dependencies;
        this.repositories = options.repositoryUrls?.map((url) => `maven { url "${url}" }`) ?? ["mavenLocal()", "mavenCentral()"];
    }
    /**
     * Add dependencies to the build.gradle
     */
    addDeps(...deps) {
        this.dependencies.push(...deps);
    }
    /**
     * Add source directories to the build.gradle
     */
    addSources(...sources) {
        this.sources.push(...sources);
    }
    synthesizeContent(_) {
        return `// ${this.marker}
plugins {
    id "software.amazon.smithy" version "0.6.0"
}

sourceSets {
    main {
        java {
            srcDirs = [${this.sources.map((s) => `'${s}'`).join(", ")}]
        }
    }
}

// Dependencies can be added by configuring smithyBuildOptions in your .projenrc file
repositories {
${this.repositories.map((repository) => `    ${repository}`).join("\n")}
}

// Dependencies can be added by configuring smithyBuildOptions in your .projenrc file
dependencies {
${this.dependencies
            .map((dep) => `    implementation ${this.renderDependency(dep)}`)
            .join("\n")}
}
`;
    }
}
exports.SmithyBuildGradleFile = SmithyBuildGradleFile;
SmithyBuildGradleFile.fileDependencyPrefix = "file://";
//# sourceMappingURL=data:application/json;base64,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