"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.SmithyDefinition = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const path = require("path");
const projen_1 = require("projen");
const smithy_build_1 = require("projen/lib/smithy/smithy-build");
const smithy_aws_pdk_prelude_1 = require("./components/smithy-aws-pdk-prelude");
const smithy_build_gradle_file_1 = require("./components/smithy-build-gradle-file");
const smithy_settings_gradle_file_1 = require("./components/smithy-settings-gradle-file");
const utils_1 = require("../../codegen/components/utils");
/**
 * Creates a project which transforms a Smithy model to OpenAPI
 */
class SmithyDefinition extends projen_1.Component {
    constructor(project, options) {
        super(project);
        const { smithyOptions, handlerLanguages } = options;
        // Ignore gradle wrapper by default
        if (smithyOptions.ignoreGradleWrapper ?? true) {
            project.gitignore.addPatterns("gradle");
            project.gitignore.addPatterns("gradlew");
            project.gitignore.addPatterns("gradlew.bat");
        }
        // Always ignore the .gradle dir which the wrapper downloads gradle into
        project.gitignore.addPatterns(".gradle");
        this.gradleProjectName = project.name.replace(/[\/\\:<>"?\*|]/g, "-");
        // Add settings.gradle
        new smithy_settings_gradle_file_1.SmithySettingsGradleFile(project, {
            gradleProjectName: this.gradleProjectName,
        });
        const modelDir = "src/main/smithy";
        // Always add the following required dependencies
        const requiredDependencies = [
            "software.amazon.smithy:smithy-cli",
            "software.amazon.smithy:smithy-model",
            "software.amazon.smithy:smithy-openapi",
            "software.amazon.smithy:smithy-aws-traits",
        ];
        const requiredSmithyDependencyVersion = "1.28.0";
        // Ensure dependencies always include the required dependencies, allowing users to customise the version
        const userSpecifiedDependencies = smithyOptions.smithyBuildOptions?.maven?.dependencies ?? [];
        const userSpecifiedDependencySet = new Set(userSpecifiedDependencies.map((dep) => dep.split(":").slice(0, -1).join(":")));
        const dependencies = [
            ...requiredDependencies
                .filter((requiredDep) => !userSpecifiedDependencySet.has(requiredDep))
                .map((dep) => `${dep}:${requiredSmithyDependencyVersion}`),
            ...userSpecifiedDependencies,
        ];
        // Add build.gradle
        this.smithyBuildGradleFile = new smithy_build_gradle_file_1.SmithyBuildGradleFile(project, {
            modelDir,
            dependencies,
            repositoryUrls: smithyOptions.smithyBuildOptions?.maven?.repositoryUrls,
        });
        const { namespace: serviceNamespace, serviceName } = smithyOptions.serviceName;
        const firstHandlerLanguage = options.handlerLanguages?.[0];
        // Create the default smithy model
        new projen_1.SampleDir(project, modelDir, {
            files: {
                "main.smithy": `$version: "2"
namespace ${serviceNamespace}

use aws.protocols#restJson1

/// A sample smithy api
@restJson1
service ${serviceName} {
    version: "1.0"
    operations: [SayHello]
    errors: [
      BadRequestError
      NotAuthorizedError
      InternalFailureError
    ]
}`,
                "operations/say-hello.smithy": `$version: "2"
namespace ${serviceNamespace}

@readonly
@http(method: "GET", uri: "/hello")${firstHandlerLanguage
                    ? `\n@handler(language: "${firstHandlerLanguage}")`
                    : ""}
operation SayHello {
    input := {
        @httpQuery("name")
        @required
        name: String
    }
    output := {
        @required
        message: String
    }
    errors: [NotFoundError]
}
`,
                "types/errors.smithy": `$version: "2"
namespace ${serviceNamespace}

/// An error message
string ErrorMessage

/// An internal failure at the fault of the server
@error("server")
@httpError(500)
structure InternalFailureError {
    /// Message with details about the error
    @required
    message: ErrorMessage
}

/// An error at the fault of the client sending invalid input
@error("client")
@httpError(400)
structure BadRequestError {
    /// Message with details about the error
    @required
    message: ErrorMessage
}

/// An error due to the client attempting to access a missing resource
@error("client")
@httpError(404)
structure NotFoundError {
    /// Message with details about the error
    @required
    message: ErrorMessage
}

/// An error due to the client not being authorized to access the resource
@error("client")
@httpError(403)
structure NotAuthorizedError {
    /// Message with details about the error
    @required
    message: ErrorMessage
}
`,
            },
        });
        // Create the smithy build json file
        this.smithyBuild = new smithy_build_1.SmithyBuild(project, {
            version: "2.0",
            ...smithyOptions.smithyBuildOptions,
            sources: [
                modelDir,
                ...this.asRelativePathsToProject(smithyOptions.smithyBuildOptions?.additionalSources ?? []),
            ],
            projections: {
                ...smithyOptions.smithyBuildOptions?.projections,
                openapi: {
                    plugins: {
                        openapi: {
                            service: `${serviceNamespace}#${serviceName}`,
                            // By default, preserve tags in the generated spec, but allow users to explicitly overwrite this
                            tags: true,
                            // By default, use integer types as this is more intuitive when smithy distinguishes between Integers and Doubles.
                            // Users may also override this.
                            useIntegerType: true,
                            ...smithyOptions.smithyBuildOptions?.projections?.openapi?.plugins
                                ?.openapi,
                        },
                    },
                },
            },
            maven: {
                // Filter out any file dependencies since these aren't supported in smithy-build.json
                dependencies: dependencies.filter((dep) => !dep.startsWith(smithy_build_gradle_file_1.SmithyBuildGradleFile.fileDependencyPrefix)),
                repositories: (smithyOptions.smithyBuildOptions?.maven?.repositoryUrls ?? [
                    "https://repo.maven.apache.org/maven2/",
                    "file://~/.m2/repository",
                ]).map((url) => ({ url })),
            },
        });
        // Add the smithy prelude (managed by aws-pdk)
        const generatedModelDir = path.join("generated", "main", "smithy");
        new smithy_aws_pdk_prelude_1.SmithyAwsPdkPrelude(project, {
            generatedModelDir,
            serviceNamespace,
            handlerLanguages,
        });
        this.addSources(generatedModelDir);
        const projectionOutputPath = path.join("build", "smithyprojections", this.gradleProjectName, "openapi");
        this.openApiSpecificationPath = path.join(projectionOutputPath, "openapi", `${serviceName}.openapi.json`);
        this.smithyJsonModelPath = path.join(projectionOutputPath, "model", "model.json");
        // Copy the gradle files during build if they don't exist. We don't overwrite to allow users to BYO gradle wrapper
        // and set `ignoreGradleWrapper: false`.
        project.generateTask.exec((0, utils_1.buildTypeSafeApiExecCommand)(utils_1.TypeSafeApiScript.COPY_GRADLE_WRAPPER));
        // Build with gradle to generate smithy projections, and any other tasks
        project.generateTask.exec("./gradlew build");
        if (smithyOptions.ignoreSmithyBuildOutput ?? true) {
            // Ignore the build directory, and smithy-output which was the old build directory for the cli-based generation
            project.gitignore.addPatterns("build", "smithy-output");
        }
    }
    /**
     * Add maven-style or local file dependencies to the smithy model project
     * @param deps dependencies to add, eg "software.amazon.smithy:smithy-validation-model:1.27.2" or "file://../some/path/build/lib/my-shapes.jar
     */
    addDeps(...deps) {
        this.smithyBuildGradleFile.addDeps(...deps);
        this.smithyBuild.addMavenDependencies(...deps.filter((dep) => !dep.startsWith(smithy_build_gradle_file_1.SmithyBuildGradleFile.fileDependencyPrefix)));
    }
    /**
     * Add dependencies on other smithy models, such that their shapes can be imported in this project
     * @param deps smithy definitions to depend on
     */
    addSmithyDeps(...deps) {
        this.addDeps(...deps.map((dep) => `${smithy_build_gradle_file_1.SmithyBuildGradleFile.fileDependencyPrefix}${path.join(path.relative(this.project.outdir, dep.project.outdir), "build", "libs", `${dep.gradleProjectName}.jar`)}`));
    }
    /**
     * Add additional paths to model source files or directories.
     * Paths should be relative to the project outdir. Any absolute paths will be
     * resolved as relative paths.
     */
    addSources(...sources) {
        const relativeSources = this.asRelativePathsToProject(sources);
        this.smithyBuild.addSources(...relativeSources);
        this.smithyBuildGradleFile.addSources(...relativeSources);
    }
    /**
     * Convert any given absolute paths to relative paths to the project outdir
     * @private
     */
    asRelativePathsToProject(paths) {
        return paths.map((p) => path.isAbsolute(p) ? path.relative(this.project.outdir, p) : p);
    }
}
exports.SmithyDefinition = SmithyDefinition;
_a = JSII_RTTI_SYMBOL_1;
SmithyDefinition[_a] = { fqn: "@aws/pdk.type_safe_api.SmithyDefinition", version: "0.23.9" };
//# sourceMappingURL=data:application/json;base64,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