"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.TypeSafeApiModelProject = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const path = require("path");
const projen_1 = require("projen");
const open_api_definition_1 = require("./openapi/open-api-definition");
const smithy_definition_1 = require("./smithy/smithy-definition");
const type_safe_api_command_environment_1 = require("../codegen/components/type-safe-api-command-environment");
const utils_1 = require("../codegen/components/utils");
const types_1 = require("../types");
class TypeSafeApiModelProject extends projen_1.Project {
    constructor(options) {
        super(options);
        /**
         * Name of the final bundled OpenAPI specification
         */
        this.parsedSpecFile = ".api.json";
        this.addApiDefinition = ({ modelLanguage, modelOptions, handlerLanguages, }) => {
            if (modelLanguage === types_1.ModelLanguage.SMITHY) {
                if (!modelOptions.smithy) {
                    throw new Error(`modelOptions.smithy is required when selected model language is ${types_1.ModelLanguage.SMITHY}`);
                }
                const smithyOptions = modelOptions.smithy;
                const smithy = new smithy_definition_1.SmithyDefinition(this, {
                    smithyOptions,
                    handlerLanguages,
                });
                return { smithy, specPath: smithy.openApiSpecificationPath };
            }
            else if (modelLanguage === types_1.ModelLanguage.OPENAPI) {
                if (!modelOptions.openapi) {
                    throw new Error(`modelOptions.openapi is required when selected model language is ${types_1.ModelLanguage.OPENAPI}`);
                }
                const openApiOptions = modelOptions.openapi;
                const openapi = new open_api_definition_1.OpenApiDefinition(this, {
                    openApiOptions,
                    handlerLanguages,
                });
                return { openapi, specPath: openapi.openApiSpecificationPath };
            }
            else {
                throw new Error(`Unknown model language ${modelLanguage}`);
            }
        };
        this.addParseAndBundleTask = (openApiSpecificationPath) => {
            this.generateTask.exec((0, utils_1.buildTypeSafeApiExecCommand)(utils_1.TypeSafeApiScript.PARSE_OPENAPI_SPEC, `--spec-path ${openApiSpecificationPath} --output-path ${this.parsedSpecFile}${this.smithy
                ? ` --smithy-json-path ${this.smithy.smithyJsonModelPath}`
                : ""}`));
            this.addGitIgnore(this.parsedSpecFile);
        };
        type_safe_api_command_environment_1.TypeSafeApiCommandEnvironment.ensure(this);
        this.generateTask = this.addTask("generate");
        // Add the API definition
        const { specPath, smithy, openapi } = this.addApiDefinition(options);
        this.smithy = smithy;
        this.openapi = openapi;
        this.apiName =
            options.modelOptions.smithy?.serviceName.serviceName ??
                options.modelOptions.openapi?.title;
        // Parse and bundle the openapi specification
        this.addParseAndBundleTask(specPath);
        // Run the generate task as part of build
        this.compileTask.spawn(this.generateTask);
        // Add the README as a sample file which the user may edit
        new projen_1.SampleFile(this, "README.md", {
            sourcePath: path.resolve(__dirname, "..", "..", "..", "samples", "type-safe-api", "readme", "model", `${options.modelLanguage}.md`),
        });
    }
}
exports.TypeSafeApiModelProject = TypeSafeApiModelProject;
_a = JSII_RTTI_SYMBOL_1;
TypeSafeApiModelProject[_a] = { fqn: "@aws/pdk.type_safe_api.TypeSafeApiModelProject", version: "0.23.9" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidHlwZS1zYWZlLWFwaS1tb2RlbC1wcm9qZWN0LmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsidHlwZS1zYWZlLWFwaS1tb2RlbC1wcm9qZWN0LnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7Ozs7O0FBQUE7c0NBQ3NDO0FBQ3RDLDZCQUE2QjtBQUM3QixtQ0FBbUU7QUFDbkUsdUVBQWtFO0FBQ2xFLGtFQUE4RDtBQUM5RCwrR0FBd0c7QUFDeEcsdURBR3FDO0FBRXJDLG9DQUF1RDtBQWlCdkQsTUFBYSx1QkFBd0IsU0FBUSxnQkFBTztJQXlCbEQsWUFBWSxPQUF1QztRQUNqRCxLQUFLLENBQUMsT0FBTyxDQUFDLENBQUM7UUF6QmpCOztXQUVHO1FBQ2EsbUJBQWMsR0FBVyxXQUFXLENBQUM7UUF5RDdDLHFCQUFnQixHQUFHLENBQUMsRUFDMUIsYUFBYSxFQUNiLFlBQVksRUFDWixnQkFBZ0IsR0FDZSxFQUFFLEVBQUU7WUFDbkMsSUFBSSxhQUFhLEtBQUsscUJBQWEsQ0FBQyxNQUFNLEVBQUU7Z0JBQzFDLElBQUksQ0FBQyxZQUFZLENBQUMsTUFBTSxFQUFFO29CQUN4QixNQUFNLElBQUksS0FBSyxDQUNiLG1FQUFtRSxxQkFBYSxDQUFDLE1BQU0sRUFBRSxDQUMxRixDQUFDO2lCQUNIO2dCQUVELE1BQU0sYUFBYSxHQUFHLFlBQVksQ0FBQyxNQUFNLENBQUM7Z0JBQzFDLE1BQU0sTUFBTSxHQUFHLElBQUksb0NBQWdCLENBQUMsSUFBSSxFQUFFO29CQUN4QyxhQUFhO29CQUNiLGdCQUFnQjtpQkFDakIsQ0FBQyxDQUFDO2dCQUVILE9BQU8sRUFBRSxNQUFNLEVBQUUsUUFBUSxFQUFFLE1BQU0sQ0FBQyx3QkFBd0IsRUFBRSxDQUFDO2FBQzlEO2lCQUFNLElBQUksYUFBYSxLQUFLLHFCQUFhLENBQUMsT0FBTyxFQUFFO2dCQUNsRCxJQUFJLENBQUMsWUFBWSxDQUFDLE9BQU8sRUFBRTtvQkFDekIsTUFBTSxJQUFJLEtBQUssQ0FDYixvRUFBb0UscUJBQWEsQ0FBQyxPQUFPLEVBQUUsQ0FDNUYsQ0FBQztpQkFDSDtnQkFFRCxNQUFNLGNBQWMsR0FBRyxZQUFZLENBQUMsT0FBTyxDQUFDO2dCQUM1QyxNQUFNLE9BQU8sR0FBRyxJQUFJLHVDQUFpQixDQUFDLElBQUksRUFBRTtvQkFDMUMsY0FBYztvQkFDZCxnQkFBZ0I7aUJBQ2pCLENBQUMsQ0FBQztnQkFDSCxPQUFPLEVBQUUsT0FBTyxFQUFFLFFBQVEsRUFBRSxPQUFPLENBQUMsd0JBQXdCLEVBQUUsQ0FBQzthQUNoRTtpQkFBTTtnQkFDTCxNQUFNLElBQUksS0FBSyxDQUFDLDBCQUEwQixhQUFhLEVBQUUsQ0FBQyxDQUFDO2FBQzVEO1FBQ0gsQ0FBQyxDQUFDO1FBRU0sMEJBQXFCLEdBQUcsQ0FBQyx3QkFBZ0MsRUFBRSxFQUFFO1lBQ25FLElBQUksQ0FBQyxZQUFZLENBQUMsSUFBSSxDQUNwQixJQUFBLG1DQUEyQixFQUN6Qix5QkFBaUIsQ0FBQyxrQkFBa0IsRUFDcEMsZUFBZSx3QkFBd0Isa0JBQ3JDLElBQUksQ0FBQyxjQUNQLEdBQ0UsSUFBSSxDQUFDLE1BQU07Z0JBQ1QsQ0FBQyxDQUFDLHVCQUF1QixJQUFJLENBQUMsTUFBTSxDQUFDLG1CQUFtQixFQUFFO2dCQUMxRCxDQUFDLENBQUMsRUFDTixFQUFFLENBQ0gsQ0FDRixDQUFDO1lBRUYsSUFBSSxDQUFDLFlBQVksQ0FBQyxJQUFJLENBQUMsY0FBYyxDQUFDLENBQUM7UUFDekMsQ0FBQyxDQUFDO1FBdEZBLGlFQUE2QixDQUFDLE1BQU0sQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUUzQyxJQUFJLENBQUMsWUFBWSxHQUFHLElBQUksQ0FBQyxPQUFPLENBQUMsVUFBVSxDQUFDLENBQUM7UUFFN0MseUJBQXlCO1FBQ3pCLE1BQU0sRUFBRSxRQUFRLEVBQUUsTUFBTSxFQUFFLE9BQU8sRUFBRSxHQUFHLElBQUksQ0FBQyxnQkFBZ0IsQ0FBQyxPQUFPLENBQUMsQ0FBQztRQUNyRSxJQUFJLENBQUMsTUFBTSxHQUFHLE1BQU0sQ0FBQztRQUNyQixJQUFJLENBQUMsT0FBTyxHQUFHLE9BQU8sQ0FBQztRQUN2QixJQUFJLENBQUMsT0FBTztZQUNWLE9BQU8sQ0FBQyxZQUFZLENBQUMsTUFBTSxFQUFFLFdBQVcsQ0FBQyxXQUFXO2dCQUNwRCxPQUFPLENBQUMsWUFBWSxDQUFDLE9BQU8sRUFBRSxLQUFLLENBQUM7UUFFdEMsNkNBQTZDO1FBQzdDLElBQUksQ0FBQyxxQkFBcUIsQ0FBQyxRQUFRLENBQUMsQ0FBQztRQUVyQyx5Q0FBeUM7UUFDekMsSUFBSSxDQUFDLFdBQVcsQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDLFlBQVksQ0FBQyxDQUFDO1FBRTFDLDBEQUEwRDtRQUMxRCxJQUFJLG1CQUFVLENBQUMsSUFBSSxFQUFFLFdBQVcsRUFBRTtZQUNoQyxVQUFVLEVBQUUsSUFBSSxDQUFDLE9BQU8sQ0FDdEIsU0FBUyxFQUNULElBQUksRUFDSixJQUFJLEVBQ0osSUFBSSxFQUNKLFNBQVMsRUFDVCxlQUFlLEVBQ2YsUUFBUSxFQUNSLE9BQU8sRUFDUCxHQUFHLE9BQU8sQ0FBQyxhQUFhLEtBQUssQ0FDOUI7U0FDRixDQUFDLENBQUM7SUFDTCxDQUFDOztBQTNESCwwREFrSEMiLCJzb3VyY2VzQ29udGVudCI6WyIvKiEgQ29weXJpZ2h0IFtBbWF6b24uY29tXShodHRwOi8vYW1hem9uLmNvbS8pLCBJbmMuIG9yIGl0cyBhZmZpbGlhdGVzLiBBbGwgUmlnaHRzIFJlc2VydmVkLlxuU1BEWC1MaWNlbnNlLUlkZW50aWZpZXI6IEFwYWNoZS0yLjAgKi9cbmltcG9ydCAqIGFzIHBhdGggZnJvbSBcInBhdGhcIjtcbmltcG9ydCB7IFByb2plY3QsIFByb2plY3RPcHRpb25zLCBTYW1wbGVGaWxlLCBUYXNrIH0gZnJvbSBcInByb2plblwiO1xuaW1wb3J0IHsgT3BlbkFwaURlZmluaXRpb24gfSBmcm9tIFwiLi9vcGVuYXBpL29wZW4tYXBpLWRlZmluaXRpb25cIjtcbmltcG9ydCB7IFNtaXRoeURlZmluaXRpb24gfSBmcm9tIFwiLi9zbWl0aHkvc21pdGh5LWRlZmluaXRpb25cIjtcbmltcG9ydCB7IFR5cGVTYWZlQXBpQ29tbWFuZEVudmlyb25tZW50IH0gZnJvbSBcIi4uL2NvZGVnZW4vY29tcG9uZW50cy90eXBlLXNhZmUtYXBpLWNvbW1hbmQtZW52aXJvbm1lbnRcIjtcbmltcG9ydCB7XG4gIGJ1aWxkVHlwZVNhZmVBcGlFeGVjQ29tbWFuZCxcbiAgVHlwZVNhZmVBcGlTY3JpcHQsXG59IGZyb20gXCIuLi9jb2RlZ2VuL2NvbXBvbmVudHMvdXRpbHNcIjtcbmltcG9ydCB7IExhbmd1YWdlIH0gZnJvbSBcIi4uL2xhbmd1YWdlc1wiO1xuaW1wb3J0IHsgTW9kZWxMYW5ndWFnZSwgTW9kZWxPcHRpb25zIH0gZnJvbSBcIi4uL3R5cGVzXCI7XG5cbmV4cG9ydCBpbnRlcmZhY2UgVHlwZVNhZmVBcGlNb2RlbFByb2plY3RPcHRpb25zIGV4dGVuZHMgUHJvamVjdE9wdGlvbnMge1xuICAvKipcbiAgICogTGFuZ3VhZ2UgdGhlIG1vZGVsIGlzIGRlZmluZWQgaW5cbiAgICovXG4gIHJlYWRvbmx5IG1vZGVsTGFuZ3VhZ2U6IE1vZGVsTGFuZ3VhZ2U7XG4gIC8qKlxuICAgKiBPcHRpb25zIGZvciB0aGUgbW9kZWxcbiAgICovXG4gIHJlYWRvbmx5IG1vZGVsT3B0aW9uczogTW9kZWxPcHRpb25zO1xuICAvKipcbiAgICogVGhlIGxhbmd1YWdlcyB1c2VycyBoYXZlIHNwZWNpZmllZCBmb3IgaGFuZGxlciBwcm9qZWN0cyAoaWYgYW55KVxuICAgKi9cbiAgcmVhZG9ubHkgaGFuZGxlckxhbmd1YWdlcz86IExhbmd1YWdlW107XG59XG5cbmV4cG9ydCBjbGFzcyBUeXBlU2FmZUFwaU1vZGVsUHJvamVjdCBleHRlbmRzIFByb2plY3Qge1xuICAvKipcbiAgICogTmFtZSBvZiB0aGUgZmluYWwgYnVuZGxlZCBPcGVuQVBJIHNwZWNpZmljYXRpb25cbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBwYXJzZWRTcGVjRmlsZTogc3RyaW5nID0gXCIuYXBpLmpzb25cIjtcbiAgLyoqXG4gICAqIFJlZmVyZW5jZSB0byB0aGUgdGFzayB1c2VkIGZvciBnZW5lcmF0aW5nIHRoZSBmaW5hbCBidW5kbGVkIE9wZW5BUEkgc3BlY2lmaWNhdGlvblxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IGdlbmVyYXRlVGFzazogVGFzaztcblxuICAvKipcbiAgICogUmVmZXJlbmNlIHRvIHRoZSBTbWl0aHkgZGVmaW5pdGlvbiBjb21wb25lbnQuIFdpbGwgYmUgZGVmaW5lZCBpZiB0aGUgbW9kZWwgbGFuZ3VhZ2UgaXMgU21pdGh5XG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgc21pdGh5PzogU21pdGh5RGVmaW5pdGlvbjtcblxuICAvKipcbiAgICogUmVmZXJlbmNlIHRvIHRoZSBPcGVuQVBJIGRlZmluaXRpb24gY29tcG9uZW50LiBXaWxsIGJlIGRlZmluZWQgaWYgdGhlIG1vZGVsIGxhbmd1YWdlIGlzIE9wZW5BUElcbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBvcGVuYXBpPzogT3BlbkFwaURlZmluaXRpb247XG5cbiAgLyoqXG4gICAqIE5hbWUgb2YgdGhlIEFQSS4gSWYgU21pdGh5LCB3aWxsIHJlc29sdmUgdG8gc2VydmljZU5hbWUgb3RoZXJ3aXNlIGl0IHdpbGwgdXNlIHRpdGxlLlxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IGFwaU5hbWU7XG5cbiAgY29uc3RydWN0b3Iob3B0aW9uczogVHlwZVNhZmVBcGlNb2RlbFByb2plY3RPcHRpb25zKSB7XG4gICAgc3VwZXIob3B0aW9ucyk7XG4gICAgVHlwZVNhZmVBcGlDb21tYW5kRW52aXJvbm1lbnQuZW5zdXJlKHRoaXMpO1xuXG4gICAgdGhpcy5nZW5lcmF0ZVRhc2sgPSB0aGlzLmFkZFRhc2soXCJnZW5lcmF0ZVwiKTtcblxuICAgIC8vIEFkZCB0aGUgQVBJIGRlZmluaXRpb25cbiAgICBjb25zdCB7IHNwZWNQYXRoLCBzbWl0aHksIG9wZW5hcGkgfSA9IHRoaXMuYWRkQXBpRGVmaW5pdGlvbihvcHRpb25zKTtcbiAgICB0aGlzLnNtaXRoeSA9IHNtaXRoeTtcbiAgICB0aGlzLm9wZW5hcGkgPSBvcGVuYXBpO1xuICAgIHRoaXMuYXBpTmFtZSA9XG4gICAgICBvcHRpb25zLm1vZGVsT3B0aW9ucy5zbWl0aHk/LnNlcnZpY2VOYW1lLnNlcnZpY2VOYW1lID8/XG4gICAgICBvcHRpb25zLm1vZGVsT3B0aW9ucy5vcGVuYXBpPy50aXRsZTtcblxuICAgIC8vIFBhcnNlIGFuZCBidW5kbGUgdGhlIG9wZW5hcGkgc3BlY2lmaWNhdGlvblxuICAgIHRoaXMuYWRkUGFyc2VBbmRCdW5kbGVUYXNrKHNwZWNQYXRoKTtcblxuICAgIC8vIFJ1biB0aGUgZ2VuZXJhdGUgdGFzayBhcyBwYXJ0IG9mIGJ1aWxkXG4gICAgdGhpcy5jb21waWxlVGFzay5zcGF3bih0aGlzLmdlbmVyYXRlVGFzayk7XG5cbiAgICAvLyBBZGQgdGhlIFJFQURNRSBhcyBhIHNhbXBsZSBmaWxlIHdoaWNoIHRoZSB1c2VyIG1heSBlZGl0XG4gICAgbmV3IFNhbXBsZUZpbGUodGhpcywgXCJSRUFETUUubWRcIiwge1xuICAgICAgc291cmNlUGF0aDogcGF0aC5yZXNvbHZlKFxuICAgICAgICBfX2Rpcm5hbWUsXG4gICAgICAgIFwiLi5cIixcbiAgICAgICAgXCIuLlwiLFxuICAgICAgICBcIi4uXCIsXG4gICAgICAgIFwic2FtcGxlc1wiLFxuICAgICAgICBcInR5cGUtc2FmZS1hcGlcIixcbiAgICAgICAgXCJyZWFkbWVcIixcbiAgICAgICAgXCJtb2RlbFwiLFxuICAgICAgICBgJHtvcHRpb25zLm1vZGVsTGFuZ3VhZ2V9Lm1kYFxuICAgICAgKSxcbiAgICB9KTtcbiAgfVxuXG4gIHByaXZhdGUgYWRkQXBpRGVmaW5pdGlvbiA9ICh7XG4gICAgbW9kZWxMYW5ndWFnZSxcbiAgICBtb2RlbE9wdGlvbnMsXG4gICAgaGFuZGxlckxhbmd1YWdlcyxcbiAgfTogVHlwZVNhZmVBcGlNb2RlbFByb2plY3RPcHRpb25zKSA9PiB7XG4gICAgaWYgKG1vZGVsTGFuZ3VhZ2UgPT09IE1vZGVsTGFuZ3VhZ2UuU01JVEhZKSB7XG4gICAgICBpZiAoIW1vZGVsT3B0aW9ucy5zbWl0aHkpIHtcbiAgICAgICAgdGhyb3cgbmV3IEVycm9yKFxuICAgICAgICAgIGBtb2RlbE9wdGlvbnMuc21pdGh5IGlzIHJlcXVpcmVkIHdoZW4gc2VsZWN0ZWQgbW9kZWwgbGFuZ3VhZ2UgaXMgJHtNb2RlbExhbmd1YWdlLlNNSVRIWX1gXG4gICAgICAgICk7XG4gICAgICB9XG5cbiAgICAgIGNvbnN0IHNtaXRoeU9wdGlvbnMgPSBtb2RlbE9wdGlvbnMuc21pdGh5O1xuICAgICAgY29uc3Qgc21pdGh5ID0gbmV3IFNtaXRoeURlZmluaXRpb24odGhpcywge1xuICAgICAgICBzbWl0aHlPcHRpb25zLFxuICAgICAgICBoYW5kbGVyTGFuZ3VhZ2VzLFxuICAgICAgfSk7XG5cbiAgICAgIHJldHVybiB7IHNtaXRoeSwgc3BlY1BhdGg6IHNtaXRoeS5vcGVuQXBpU3BlY2lmaWNhdGlvblBhdGggfTtcbiAgICB9IGVsc2UgaWYgKG1vZGVsTGFuZ3VhZ2UgPT09IE1vZGVsTGFuZ3VhZ2UuT1BFTkFQSSkge1xuICAgICAgaWYgKCFtb2RlbE9wdGlvbnMub3BlbmFwaSkge1xuICAgICAgICB0aHJvdyBuZXcgRXJyb3IoXG4gICAgICAgICAgYG1vZGVsT3B0aW9ucy5vcGVuYXBpIGlzIHJlcXVpcmVkIHdoZW4gc2VsZWN0ZWQgbW9kZWwgbGFuZ3VhZ2UgaXMgJHtNb2RlbExhbmd1YWdlLk9QRU5BUEl9YFxuICAgICAgICApO1xuICAgICAgfVxuXG4gICAgICBjb25zdCBvcGVuQXBpT3B0aW9ucyA9IG1vZGVsT3B0aW9ucy5vcGVuYXBpO1xuICAgICAgY29uc3Qgb3BlbmFwaSA9IG5ldyBPcGVuQXBpRGVmaW5pdGlvbih0aGlzLCB7XG4gICAgICAgIG9wZW5BcGlPcHRpb25zLFxuICAgICAgICBoYW5kbGVyTGFuZ3VhZ2VzLFxuICAgICAgfSk7XG4gICAgICByZXR1cm4geyBvcGVuYXBpLCBzcGVjUGF0aDogb3BlbmFwaS5vcGVuQXBpU3BlY2lmaWNhdGlvblBhdGggfTtcbiAgICB9IGVsc2Uge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKGBVbmtub3duIG1vZGVsIGxhbmd1YWdlICR7bW9kZWxMYW5ndWFnZX1gKTtcbiAgICB9XG4gIH07XG5cbiAgcHJpdmF0ZSBhZGRQYXJzZUFuZEJ1bmRsZVRhc2sgPSAob3BlbkFwaVNwZWNpZmljYXRpb25QYXRoOiBzdHJpbmcpID0+IHtcbiAgICB0aGlzLmdlbmVyYXRlVGFzay5leGVjKFxuICAgICAgYnVpbGRUeXBlU2FmZUFwaUV4ZWNDb21tYW5kKFxuICAgICAgICBUeXBlU2FmZUFwaVNjcmlwdC5QQVJTRV9PUEVOQVBJX1NQRUMsXG4gICAgICAgIGAtLXNwZWMtcGF0aCAke29wZW5BcGlTcGVjaWZpY2F0aW9uUGF0aH0gLS1vdXRwdXQtcGF0aCAke1xuICAgICAgICAgIHRoaXMucGFyc2VkU3BlY0ZpbGVcbiAgICAgICAgfSR7XG4gICAgICAgICAgdGhpcy5zbWl0aHlcbiAgICAgICAgICAgID8gYCAtLXNtaXRoeS1qc29uLXBhdGggJHt0aGlzLnNtaXRoeS5zbWl0aHlKc29uTW9kZWxQYXRofWBcbiAgICAgICAgICAgIDogXCJcIlxuICAgICAgICB9YFxuICAgICAgKVxuICAgICk7XG5cbiAgICB0aGlzLmFkZEdpdElnbm9yZSh0aGlzLnBhcnNlZFNwZWNGaWxlKTtcbiAgfTtcbn1cbiJdfQ==