import { Project, ProjectOptions } from "projen";
import { DocumentationFormat, Language, Library } from "./languages";
import { TypeSafeApiModelProject } from "./model/type-safe-api-model-project";
import { GeneratedRuntimeCodeOptions, GeneratedCodeProjects, GeneratedDocumentationOptions, GeneratedDocumentationProjects, GeneratedLibraryOptions, GeneratedLibraryProjects, ModelLanguage, ModelOptions, GeneratedInfrastructureCodeOptions, GeneratedHandlersCodeOptions, ProjectCollections } from "./types";
/**
 * Configuration for modelling the API
 */
export interface ModelConfiguration {
    /**
     * The language the API model is defined in.
     */
    readonly language: ModelLanguage;
    /**
     * Options for the API model.
     */
    readonly options: ModelOptions;
}
/**
 * Configuration for generated runtime projects
 */
export interface RuntimeConfiguration {
    /**
     * The languages that runtime projects will be generated in. These projects can be used to provide type safety for
     * both client and server projects.
     */
    readonly languages: Language[];
    /**
     * Options for the generated runtimes. Note that only options provided for the specified languages will apply.
     */
    readonly options?: GeneratedRuntimeCodeOptions;
}
/**
 * Configuration for generated infrastructure
 */
export interface InfrastructureConfiguration {
    /**
     * The language to generate the type-safe CDK infrastructure in
     */
    readonly language: Language;
    /**
     * Options for the infrastructure package. Note that only those provided for the specified language will apply.
     */
    readonly options?: GeneratedInfrastructureCodeOptions;
}
/**
 * Configuration for generated lambda handlers
 */
export interface HandlersConfiguration {
    /**
     * The languages lambda handlers are written in. Specify multiple languages if you wish to implement different operations
     * as AWS Lambda functions in different languages.
     */
    readonly languages: Language[];
    /**
     * Options for the infrastructure package. Note that only those provided for the specified language will apply.
     */
    readonly options?: GeneratedHandlersCodeOptions;
}
/**
 * Configuration for generated documentation
 */
export interface DocumentationConfiguration {
    /**
     * Formats for generated documentation
     */
    readonly formats: DocumentationFormat[];
    /**
     * Options for the generated documentation projects. Note that only those provided for the specified formats will apply
     */
    readonly options?: GeneratedDocumentationOptions;
}
/**
 * Configuration for generated libraries
 */
export interface LibraryConfiguration {
    /**
     * The library to generate
     */
    readonly libraries: Library[];
    /**
     * Options for the generated library package. Note that only options for the specified libraries will apply
     */
    readonly options?: GeneratedLibraryOptions;
}
/**
 * Options for the TypeSafeApiProject
 */
export interface TypeSafeApiProjectOptions extends ProjectOptions {
    /**
     * Configuration for the API model
     */
    readonly model: ModelConfiguration;
    /**
     * Configuration for generated runtime projects (containing types, clients and server code)
     */
    readonly runtime?: RuntimeConfiguration;
    /**
     * Configuration for generated infrastructure
     */
    readonly infrastructure: InfrastructureConfiguration;
    /**
     * Configuration for lambda handlers for implementing the API
     */
    readonly handlers?: HandlersConfiguration;
    /**
     * Configuration for generated documentation
     */
    readonly documentation?: DocumentationConfiguration;
    /**
     * Configuration for generated libraries. Libraries are projects which are generated from your model, but are not
     * fully-fledged runtimes, for example react hooks or clients in languages that aren't supported as runtimes.
     */
    readonly library?: LibraryConfiguration;
}
/**
 * Project for a type-safe API, defined using Smithy or OpenAPI.
 *
 * Generates a CDK construct to deploy your API, as well as client and server code to help build your API quickly.
 *
 * @pjid type-safe-api
 */
export declare class TypeSafeApiProject extends Project {
    /**
     * Project for the api model
     */
    readonly model: TypeSafeApiModelProject;
    /**
     * Generated runtime projects. When `runtime.languages` includes the corresponding language, the project can be
     * assumed to be defined.
     */
    readonly runtime: GeneratedCodeProjects;
    /**
     * Generated infrastructure projects. Only the property corresponding to `infrastructure.language` will be defined.
     */
    readonly infrastructure: GeneratedCodeProjects;
    /**
     * Lambda handlers projects. Only the properties corresponding to `handlers.languages` will be defined.
     */
    readonly handlers: GeneratedCodeProjects;
    /**
     * Generated library projects. Only the properties corresponding to specified `library.libraries` will be defined.
     */
    readonly library: GeneratedLibraryProjects;
    /**
     * Generated documentation projects. Only the properties corresponding to specified `documentation.formats` will be defined.
     */
    readonly documentation: GeneratedDocumentationProjects;
    /**
     * Collections of all sub-projects managed by this project
     */
    readonly all: ProjectCollections;
    constructor(options: TypeSafeApiProjectOptions);
    /**
     * @inheritDoc
     */
    postSynthesize(): void;
    private getNxWorkspace;
}
