"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.PolicyStatementWithPrincipal = exports.PrincipalType = void 0;
const _5_effect_1 = require("./5-effect");
var PrincipalType;
(function (PrincipalType) {
    PrincipalType["AWS"] = "AWS";
    PrincipalType["FEDERATED"] = "Federated";
    PrincipalType["CANONICAL_USER"] = "CanonicalUser";
    PrincipalType["SERVICE"] = "Service";
})(PrincipalType = exports.PrincipalType || (exports.PrincipalType = {}));
/**
 * (experimental) Adds "principal" functionality to the Policy Statement.
 *
 * @experimental
 */
class PolicyStatementWithPrincipal extends _5_effect_1.PolicyStatementWithEffect {
    constructor() {
        super(...arguments);
        /**
         * @experimental
         */
        this.useNotPrincipals = false;
        /**
         * @experimental
         */
        this.principals = {};
    }
    /**
     * (experimental) Injects principals into the statement.
     *
     * Only relevant for the main package. In CDK mode this only calls super.
     *
     * @experimental
     */
    toJSON() {
        // @ts-ignore only available after swapping 1-base
        if (typeof this.addResources == 'function') {
            this.cdkApplyPrincipals();
            return super.toJSON();
        }
        const mode = this.useNotPrincipals ? 'NotPrincipal' : 'Principal';
        const statement = super.toJSON();
        if (this.hasPrincipals()) {
            statement[mode] = this.principals;
        }
        return statement;
    }
    /**
     * @experimental
     */
    toStatementJson() {
        this.cdkApplyPrincipals();
        // @ts-ignore only available after swapping 1-base
        return super.toStatementJson();
    }
    /**
     * @experimental
     */
    cdkApplyPrincipals() { }
    /**
     * (experimental) Switches the statement to use [`notPrincipal`](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_notprincipal.html).
     *
     * @experimental
     */
    notPrincipals() {
        this.useNotPrincipals = true;
        return this;
    }
    /**
     * (experimental) Checks weather a principal was applied to the policy.
     *
     * @experimental
     */
    hasPrincipals() {
        return Object.keys(this.principals).length > 0;
    }
    /**
     * (experimental) Adds a principal to the statement.
     *
     * @param prefix One of **AWS**, **Federated**, **CanonicalUser** or **Service**.
     * @param principal The principal string.
     * @experimental
     */
    addPrincipal(prefix, principal) {
        this.skipAutoResource = true;
        if (!(prefix in this.principals)) {
            this.principals[prefix] = [];
        }
        this.principals[prefix].push(principal);
        return this;
    }
    /**
     * (experimental) Adds any principal to the statement.
     *
     * @param arn The ARN of the principal.
     * @param prefix One of **AWS**, **Federated**, **CanonicalUser** or **Service** - Default: **AWS**.
     * @experimental
     */
    for(arn, prefix) {
        return this.addPrincipal(prefix || PrincipalType.AWS, arn);
    }
    /**
     * (experimental) Adds an account principal to the statement.
     *
     * @param account ID of the AWS account.
     * @experimental
     */
    forAccount(account) {
        return this.addPrincipal(PrincipalType.AWS, `arn:aws:iam::${account}:root`);
    }
    /**
     * (experimental) Adds a [federated](https://aws.amazon.com/identity/federation/) (web identity) principal to the statement.
     *
     * @param provider ID of the AWS account.
     * @experimental
     */
    forFederated(provider) {
        return this.addPrincipal(PrincipalType.FEDERATED, provider);
    }
    /**
     * (experimental) Adds a federated [AWS Cognito](https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_providers_oidc_cognito.html) principal to the statement.
     *
     * @experimental
     */
    forFederatedCognito() {
        return this.forFederated('cognito-identity.amazonaws.com');
    }
    /**
     * (experimental) Adds a federated [Amazon](https://login.amazon.com/) principal to the statement.
     *
     * @experimental
     */
    forFederatedAmazon() {
        return this.forFederated('www.amazon.com');
    }
    /**
     * (experimental) Adds a federated [Facebook](https://developers.facebook.com/docs/facebook-login) principal to the statement.
     *
     * @experimental
     */
    forFederatedFacebook() {
        return this.forFederated('graph.facebook.com');
    }
    /**
     * (experimental) Adds a federated [Google](https://developers.google.com/identity/protocols/oauth2/openid-connect) principal to the statement.
     *
     * @experimental
     */
    forFederatedGoogle() {
        return this.forFederated('accounts.google.com');
    }
    /**
     * (experimental) Adds a canonical user principal to the statement.
     *
     * @param userID The user ID.
     * @experimental
     */
    forCanonicalUser(userID) {
        return this.addPrincipal(PrincipalType.CANONICAL_USER, userID);
    }
    /**
     * (experimental) Adds federated SAML principal to the statement.
     *
     * @param account ID of the AWS account.
     * @param providerName Name of the SAML provider.
     * @experimental
     */
    forSaml(account, providerName) {
        return this.forFederated(`arn:aws:iam::${account}:saml-provider/${providerName}`);
    }
    /**
     * (experimental) Adds an IAM user principal to the statement.
     *
     * @param account ID of the AWS account.
     * @param user Name of the IAM user.
     * @experimental
     */
    forUser(account, user) {
        return this.addPrincipal(PrincipalType.AWS, `arn:aws:iam::${account}:user/${user}`);
    }
    /**
     * (experimental) Adds an IAM role principal to the statement.
     *
     * @param account ID of the AWS account.
     * @param role Name of the IAM role.
     * @experimental
     */
    forRole(account, role) {
        return this.addPrincipal(PrincipalType.AWS, `arn:aws:iam::${account}:role/${role}`);
    }
    /**
     * (experimental) Adds a specific assumed role session principal to the statement.
     *
     * @param account ID of the AWS account.
     * @param roleName Name of the IAM role.
     * @param sessionName Name of the session.
     * @experimental
     */
    forAssumedRoleSession(account, roleName, sessionName) {
        return this.addPrincipal(PrincipalType.AWS, `arn:aws:sts::${account}:assumed-role/${roleName}/${sessionName}`);
    }
    /**
     * (experimental) Adds a service principal to the statement.
     *
     * @param service Long version of the service name. Usually in the format: `long_service-name.amazonaws.com`.
     * @experimental
     */
    forService(service) {
        return this.addPrincipal(PrincipalType.SERVICE, service);
    }
    /**
     * (experimental) Grants public asses.
     *
     * **EVERYONE IN THE WORLD HAS ACCESS**
     *
     * We strongly recommend that you do not use a wildcard in the Principal element in a role's trust policy unless you otherwise restrict access through a Condition element in the policy. Otherwise, any IAM user in any account in your partition can access the role.
     *
     * @experimental
     */
    forPublic() {
        return this.addPrincipal(PrincipalType.AWS, '*');
    }
}
exports.PolicyStatementWithPrincipal = PolicyStatementWithPrincipal;
//# sourceMappingURL=data:application/json;base64,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