"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ConnectionType = exports.PassthroughBehavior = exports.IntegrationType = exports.ContentHandling = exports.Integration = void 0;
/**
 * Base class for backend integrations for an API Gateway method.
 *
 * Use one of the concrete classes such as `MockIntegration`, `AwsIntegration`, `LambdaIntegration`
 * or implement on your own by specifying the set of props.
 */
class Integration {
    constructor(props) {
        this.props = props;
    }
    /**
     * Allows `Method` to access the integration props.
     *
     * @internal
     */
    get _props() {
        return this.props;
    }
    /**
     * Can be overridden by subclasses to allow the integration to interact with the method
     * being integrated, access the REST API object, method ARNs, etc.
     */
    bind(_method) {
        return;
    }
}
exports.Integration = Integration;
var ContentHandling;
(function (ContentHandling) {
    /**
     * Converts a request payload from a base64-encoded string to a binary blob.
     */
    ContentHandling["CONVERT_TO_BINARY"] = "CONVERT_TO_BINARY";
    /**
     * Converts a request payload from a binary blob to a base64-encoded string.
     */
    ContentHandling["CONVERT_TO_TEXT"] = "CONVERT_TO_TEXT";
})(ContentHandling = exports.ContentHandling || (exports.ContentHandling = {}));
var IntegrationType;
(function (IntegrationType) {
    /**
     * For integrating the API method request with an AWS service action,
     * including the Lambda function-invoking action. With the Lambda
     * function-invoking action, this is referred to as the Lambda custom
     * integration. With any other AWS service action, this is known as AWS
     * integration.
     */
    IntegrationType["AWS"] = "AWS";
    /**
     * For integrating the API method request with the Lambda function-invoking
     * action with the client request passed through as-is. This integration is
     * also referred to as the Lambda proxy integration
     */
    IntegrationType["AWS_PROXY"] = "AWS_PROXY";
    /**
     * For integrating the API method request with an HTTP endpoint, including a
     * private HTTP endpoint within a VPC. This integration is also referred to
     * as the HTTP custom integration.
     */
    IntegrationType["HTTP"] = "HTTP";
    /**
     * For integrating the API method request with an HTTP endpoint, including a
     * private HTTP endpoint within a VPC, with the client request passed
     * through as-is. This is also referred to as the HTTP proxy integration
     */
    IntegrationType["HTTP_PROXY"] = "HTTP_PROXY";
    /**
     * For integrating the API method request with API Gateway as a "loop-back"
     * endpoint without invoking any backend.
     */
    IntegrationType["MOCK"] = "MOCK";
})(IntegrationType = exports.IntegrationType || (exports.IntegrationType = {}));
var PassthroughBehavior;
(function (PassthroughBehavior) {
    /**
     * Passes the request body for unmapped content types through to the
     * integration back end without transformation.
     */
    PassthroughBehavior["WHEN_NO_MATCH"] = "WHEN_NO_MATCH";
    /**
     * Rejects unmapped content types with an HTTP 415 'Unsupported Media Type'
     * response
     */
    PassthroughBehavior["NEVER"] = "NEVER";
    /**
     * Allows pass-through when the integration has NO content types mapped to
     * templates. However if there is at least one content type defined,
     * unmapped content types will be rejected with the same 415 response.
     */
    PassthroughBehavior["WHEN_NO_TEMPLATES"] = "WHEN_NO_TEMPLATES";
})(PassthroughBehavior = exports.PassthroughBehavior || (exports.PassthroughBehavior = {}));
var ConnectionType;
(function (ConnectionType) {
    /**
     * For connections through the public routable internet
     */
    ConnectionType["INTERNET"] = "INTERNET";
    /**
     * For private connections between API Gateway and a network load balancer in a VPC
     */
    ConnectionType["VPC_LINK"] = "VPC_LINK";
})(ConnectionType = exports.ConnectionType || (exports.ConnectionType = {}));
//# sourceMappingURL=data:application/json;base64,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