"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.LambdaIntegration = void 0;
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const aws_1 = require("./aws");
/**
 * Integrates an AWS Lambda function to an API Gateway method.
 *
 * @example
 *
 *    const handler = new lambda.Function(this, 'MyFunction', ...);
 *    api.addMethod('GET', new LambdaIntegration(handler));
 *
 */
class LambdaIntegration extends aws_1.AwsIntegration {
    constructor(handler, options = {}) {
        const proxy = options.proxy === undefined ? true : options.proxy;
        super({
            proxy,
            service: 'lambda',
            path: `2015-03-31/functions/${handler.functionArn}/invocations`,
            options,
        });
        this.handler = handler;
        this.enableTest = options.allowTestInvoke === undefined ? true : false;
    }
    bind(method) {
        super.bind(method);
        const principal = new iam.ServicePrincipal('apigateway.amazonaws.com');
        const desc = `${method.api.node.uniqueId}.${method.httpMethod}.${method.resource.path.replace(/\//g, '.')}`;
        this.handler.addPermission(`ApiPermission.${desc}`, {
            principal,
            scope: method,
            sourceArn: core_1.Lazy.stringValue({ produce: () => method.methodArn }),
        });
        // add permission to invoke from the console
        if (this.enableTest) {
            this.handler.addPermission(`ApiPermission.Test.${desc}`, {
                principal,
                scope: method,
                sourceArn: method.testMethodArn,
            });
        }
        const cfnFunction = this.handler.node.defaultChild;
        let deploymentToken;
        if (!core_1.Token.isUnresolved(cfnFunction.functionName)) {
            deploymentToken = JSON.stringify({ functionName: cfnFunction.functionName });
        }
        return {
            deploymentToken,
        };
    }
}
exports.LambdaIntegration = LambdaIntegration;
//# sourceMappingURL=data:application/json;base64,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