"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AuthorizationType = exports.Method = void 0;
const core_1 = require("@aws-cdk/core");
const apigateway_generated_1 = require("./apigateway.generated");
const authorizer_1 = require("./authorizer");
const integration_1 = require("./integration");
const mock_1 = require("./integrations/mock");
const restapi_1 = require("./restapi");
const util_1 = require("./util");
class Method extends core_1.Resource {
    constructor(scope, id, props) {
        var _a, _b, _c;
        super(scope, id);
        this.resource = props.resource;
        this.api = props.resource.api;
        this.httpMethod = props.httpMethod.toUpperCase();
        util_1.validateHttpMethod(this.httpMethod);
        const options = props.options || {};
        const defaultMethodOptions = props.resource.defaultMethodOptions || {};
        const authorizer = options.authorizer || defaultMethodOptions.authorizer;
        const authorizerId = authorizer === null || authorizer === void 0 ? void 0 : authorizer.authorizerId;
        const authorizationTypeOption = options.authorizationType || defaultMethodOptions.authorizationType;
        const authorizationType = (authorizer === null || authorizer === void 0 ? void 0 : authorizer.authorizationType) || authorizationTypeOption || AuthorizationType.NONE;
        // if the authorizer defines an authorization type and we also have an explicit option set, check that they are the same
        if ((authorizer === null || authorizer === void 0 ? void 0 : authorizer.authorizationType) && authorizationTypeOption && (authorizer === null || authorizer === void 0 ? void 0 : authorizer.authorizationType) !== authorizationTypeOption) {
            throw new Error(`${this.resource}/${this.httpMethod} - Authorization type is set to ${authorizationTypeOption} ` +
                `which is different from what is required by the authorizer [${authorizer.authorizationType}]`);
        }
        if (authorizer_1.Authorizer.isAuthorizer(authorizer)) {
            authorizer._attachToApi(this.api);
        }
        const integration = (_b = (_a = props.integration) !== null && _a !== void 0 ? _a : this.resource.defaultIntegration) !== null && _b !== void 0 ? _b : new mock_1.MockIntegration();
        const bindResult = integration.bind(this);
        const methodProps = {
            resourceId: props.resource.resourceId,
            restApiId: this.api.restApiId,
            httpMethod: this.httpMethod,
            operationName: options.operationName || defaultMethodOptions.operationName,
            apiKeyRequired: options.apiKeyRequired || defaultMethodOptions.apiKeyRequired,
            authorizationType,
            authorizerId,
            requestParameters: options.requestParameters || defaultMethodOptions.requestParameters,
            integration: this.renderIntegration(integration),
            methodResponses: this.renderMethodResponses(options.methodResponses),
            requestModels: this.renderRequestModels(options.requestModels),
            requestValidatorId: this.requestValidatorId(options),
            authorizationScopes: (_c = options.authorizationScopes) !== null && _c !== void 0 ? _c : defaultMethodOptions.authorizationScopes,
        };
        const resource = new apigateway_generated_1.CfnMethod(this, 'Resource', methodProps);
        this.methodId = resource.ref;
        if (restapi_1.RestApiBase._isRestApiBase(props.resource.api)) {
            props.resource.api._attachMethod(this);
        }
        const deployment = props.resource.api.latestDeployment;
        if (deployment) {
            deployment.node.addDependency(resource);
            deployment.addToLogicalId({
                method: {
                    ...methodProps,
                    integrationToken: bindResult === null || bindResult === void 0 ? void 0 : bindResult.deploymentToken,
                },
            });
        }
    }
    /**
     * The RestApi associated with this Method
     * @deprecated - Throws an error if this Resource is not associated with an instance of `RestApi`. Use `api` instead.
     */
    get restApi() {
        return this.resource.restApi;
    }
    /**
     * Returns an execute-api ARN for this method:
     *
     *   arn:aws:execute-api:{region}:{account}:{restApiId}/{stage}/{method}/{path}
     *
     * NOTE: {stage} will refer to the `restApi.deploymentStage`, which will
     * automatically set if auto-deploy is enabled, or can be explicitly assigned.
     * When not configured, {stage} will be set to '*', as a shorthand for 'all stages'.
     *
     * @attribute
     */
    get methodArn() {
        var _a;
        const stage = (_a = this.api.deploymentStage) === null || _a === void 0 ? void 0 : _a.stageName;
        return this.api.arnForExecuteApi(this.httpMethod, pathForArn(this.resource.path), stage);
    }
    /**
     * Returns an execute-api ARN for this method's "test-invoke-stage" stage.
     * This stage is used by the AWS Console UI when testing the method.
     */
    get testMethodArn() {
        return this.api.arnForExecuteApi(this.httpMethod, pathForArn(this.resource.path), 'test-invoke-stage');
    }
    renderIntegration(integration) {
        const options = integration._props.options || {};
        let credentials;
        if (options.credentialsPassthrough !== undefined && options.credentialsRole !== undefined) {
            throw new Error('\'credentialsPassthrough\' and \'credentialsRole\' are mutually exclusive');
        }
        if (options.connectionType === integration_1.ConnectionType.VPC_LINK && options.vpcLink === undefined) {
            throw new Error('\'connectionType\' of VPC_LINK requires \'vpcLink\' prop to be set');
        }
        if (options.connectionType === integration_1.ConnectionType.INTERNET && options.vpcLink !== undefined) {
            throw new Error('cannot set \'vpcLink\' where \'connectionType\' is INTERNET');
        }
        if (options.credentialsRole) {
            credentials = options.credentialsRole.roleArn;
        }
        else if (options.credentialsPassthrough) {
            // arn:aws:iam::*:user/*
            // tslint:disable-next-line:max-line-length
            credentials = core_1.Stack.of(this).formatArn({ service: 'iam', region: '', account: '*', resource: 'user', sep: '/', resourceName: '*' });
        }
        return {
            type: integration._props.type,
            uri: integration._props.uri,
            cacheKeyParameters: options.cacheKeyParameters,
            cacheNamespace: options.cacheNamespace,
            contentHandling: options.contentHandling,
            integrationHttpMethod: integration._props.integrationHttpMethod,
            requestParameters: options.requestParameters,
            requestTemplates: options.requestTemplates,
            passthroughBehavior: options.passthroughBehavior,
            integrationResponses: options.integrationResponses,
            connectionType: options.connectionType,
            connectionId: options.vpcLink ? options.vpcLink.vpcLinkId : undefined,
            credentials,
        };
    }
    renderMethodResponses(methodResponses) {
        if (!methodResponses) {
            // Fall back to nothing
            return undefined;
        }
        return methodResponses.map(mr => {
            let responseModels;
            if (mr.responseModels) {
                responseModels = {};
                for (const contentType in mr.responseModels) {
                    if (mr.responseModels.hasOwnProperty(contentType)) {
                        responseModels[contentType] = mr.responseModels[contentType].modelId;
                    }
                }
            }
            const methodResponseProp = {
                statusCode: mr.statusCode,
                responseParameters: mr.responseParameters,
                responseModels,
            };
            return methodResponseProp;
        });
    }
    renderRequestModels(requestModels) {
        if (!requestModels) {
            // Fall back to nothing
            return undefined;
        }
        const models = {};
        for (const contentType in requestModels) {
            if (requestModels.hasOwnProperty(contentType)) {
                models[contentType] = requestModels[contentType].modelId;
            }
        }
        return models;
    }
    requestValidatorId(options) {
        var _a;
        if (options.requestValidator && options.requestValidatorOptions) {
            throw new Error('Only one of \'requestValidator\' or \'requestValidatorOptions\' must be specified.');
        }
        if (options.requestValidatorOptions) {
            const validator = this.restApi.addRequestValidator('validator', options.requestValidatorOptions);
            return validator.requestValidatorId;
        }
        // For backward compatibility
        return (_a = options.requestValidator) === null || _a === void 0 ? void 0 : _a.requestValidatorId;
    }
}
exports.Method = Method;
var AuthorizationType;
(function (AuthorizationType) {
    /**
     * Open access.
     */
    AuthorizationType["NONE"] = "NONE";
    /**
     * Use AWS IAM permissions.
     */
    AuthorizationType["IAM"] = "AWS_IAM";
    /**
     * Use a custom authorizer.
     */
    AuthorizationType["CUSTOM"] = "CUSTOM";
    /**
     * Use an AWS Cognito user pool.
     */
    AuthorizationType["COGNITO"] = "COGNITO_USER_POOLS";
})(AuthorizationType = exports.AuthorizationType || (exports.AuthorizationType = {}));
function pathForArn(path) {
    return path.replace(/\{[^\}]*\}/g, '*'); // replace path parameters (like '{bookId}') with asterisk
}
//# sourceMappingURL=data:application/json;base64,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