"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.UsagePlan = exports.Period = void 0;
const core_1 = require("@aws-cdk/core");
const apigateway_generated_1 = require("./apigateway.generated");
const util_1 = require("./util");
/**
 * Time period for which quota settings apply.
 */
var Period;
(function (Period) {
    Period["DAY"] = "DAY";
    Period["WEEK"] = "WEEK";
    Period["MONTH"] = "MONTH";
})(Period = exports.Period || (exports.Period = {}));
class UsagePlan extends core_1.Resource {
    constructor(scope, id, props = {}) {
        super(scope, id);
        this.apiStages = new Array();
        let resource;
        resource = new apigateway_generated_1.CfnUsagePlan(this, 'Resource', {
            apiStages: core_1.Lazy.anyValue({ produce: () => this.renderApiStages(this.apiStages) }),
            description: props.description,
            quota: this.renderQuota(props),
            throttle: this.renderThrottle(props.throttle),
            usagePlanName: props.name,
        });
        this.apiStages.push(...(props.apiStages || []));
        this.usagePlanId = resource.ref;
        // Add ApiKey when
        if (props.apiKey) {
            this.addApiKey(props.apiKey);
        }
    }
    /**
     * Adds an ApiKey.
     *
     * @param apiKey
     */
    addApiKey(apiKey) {
        const prefix = 'UsagePlanKeyResource';
        // Postfixing apikey id only from the 2nd child, to keep physicalIds of UsagePlanKey for existing CDK apps unmodifed.
        const id = this.node.tryFindChild(prefix) ? `${prefix}:${apiKey.node.uniqueId}` : prefix;
        new apigateway_generated_1.CfnUsagePlanKey(this, id, {
            keyId: apiKey.keyId,
            keyType: "API_KEY" /* API_KEY */,
            usagePlanId: this.usagePlanId,
        });
    }
    /**
     * Adds an apiStage.
     * @param apiStage
     */
    addApiStage(apiStage) {
        this.apiStages.push(apiStage);
    }
    /**
     *
     * @param props
     */
    renderApiStages(apiStages) {
        if (apiStages && apiStages.length > 0) {
            const stages = [];
            apiStages.forEach((apiStage) => {
                stages.push(this.createStage(apiStage));
            });
            return stages;
        }
        return undefined;
    }
    createStage(apiStage) {
        const stage = apiStage.stage ? apiStage.stage.stageName.toString() : undefined;
        const apiId = apiStage.stage ? apiStage.stage.restApi.restApiId : undefined;
        const throttle = this.renderThrottlePerMethod(apiStage.throttle);
        return {
            apiId,
            stage,
            throttle,
        };
    }
    renderQuota(props) {
        if (props.quota === undefined) {
            return undefined;
        }
        else {
            const limit = props.quota ? props.quota.limit : undefined;
            util_1.validateInteger(limit, 'Throttle quota limit');
            const ret = {
                limit: limit ? limit : undefined,
                offset: props.quota ? props.quota.offset : undefined,
                period: props.quota ? props.quota.period : undefined,
            };
            return ret;
        }
    }
    renderThrottle(props) {
        let ret;
        if (props !== undefined) {
            const burstLimit = props.burstLimit;
            util_1.validateInteger(burstLimit, 'Throttle burst limit');
            const rateLimit = props.rateLimit;
            util_1.validateInteger(rateLimit, 'Throttle rate limit');
            ret = {
                burstLimit: burstLimit ? burstLimit : undefined,
                rateLimit: rateLimit ? rateLimit : undefined,
            };
        }
        return ret;
    }
    renderThrottlePerMethod(throttlePerMethod) {
        const ret = {};
        if (throttlePerMethod && throttlePerMethod.length > 0) {
            throttlePerMethod.forEach((value) => {
                const method = value.method;
                // this methodId is resource path and method for example /GET or /pets/GET
                const methodId = `${method.resource.path}/${method.httpMethod}`;
                ret[methodId] = this.renderThrottle(value.throttle);
            });
        }
        return ret;
    }
}
exports.UsagePlan = UsagePlan;
//# sourceMappingURL=data:application/json;base64,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