"use strict";
const assert_1 = require("@aws-cdk/assert");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const core_1 = require("@aws-cdk/core");
const apigw = require("../lib");
module.exports = {
    'minimal setup'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        const api = new apigw.RestApi(stack, 'my-api');
        api.root.addMethod('GET'); // must have at least one method or an API definition
        // THEN
        assert_1.expect(stack).toMatch({
            Resources: {
                myapi4C7BF186: {
                    Type: 'AWS::ApiGateway::RestApi',
                    Properties: {
                        Name: 'my-api',
                    },
                },
                myapiGETF990CE3C: {
                    Type: 'AWS::ApiGateway::Method',
                    Properties: {
                        HttpMethod: 'GET',
                        ResourceId: { 'Fn::GetAtt': ['myapi4C7BF186', 'RootResourceId'] },
                        RestApiId: { Ref: 'myapi4C7BF186' },
                        AuthorizationType: 'NONE',
                        Integration: {
                            Type: 'MOCK',
                        },
                    },
                },
                myapiDeployment92F2CB4972a890db5063ec679071ba7eefc76f2a: {
                    Type: 'AWS::ApiGateway::Deployment',
                    Properties: {
                        RestApiId: { Ref: 'myapi4C7BF186' },
                        Description: 'Automatically created by the RestApi construct',
                    },
                    DependsOn: ['myapiGETF990CE3C'],
                },
                myapiDeploymentStageprod298F01AF: {
                    Type: 'AWS::ApiGateway::Stage',
                    Properties: {
                        RestApiId: { Ref: 'myapi4C7BF186' },
                        DeploymentId: { Ref: 'myapiDeployment92F2CB4972a890db5063ec679071ba7eefc76f2a' },
                        StageName: 'prod',
                    },
                },
                myapiCloudWatchRole095452E5: {
                    Type: 'AWS::IAM::Role',
                    Properties: {
                        AssumeRolePolicyDocument: {
                            Statement: [
                                {
                                    Action: 'sts:AssumeRole',
                                    Effect: 'Allow',
                                    Principal: { Service: 'apigateway.amazonaws.com' },
                                },
                            ],
                            Version: '2012-10-17',
                        },
                        ManagedPolicyArns: [
                            { 'Fn::Join': ['', ['arn:', { Ref: 'AWS::Partition' }, ':iam::aws:policy/service-role/AmazonAPIGatewayPushToCloudWatchLogs']] },
                        ],
                    },
                },
                myapiAccountEC421A0A: {
                    Type: 'AWS::ApiGateway::Account',
                    Properties: {
                        CloudWatchRoleArn: { 'Fn::GetAtt': ['myapiCloudWatchRole095452E5', 'Arn'] },
                    },
                    DependsOn: ['myapi4C7BF186'],
                },
            },
            Outputs: {
                myapiEndpoint3628AFE3: {
                    Value: {
                        'Fn::Join': ['', [
                                'https://',
                                { Ref: 'myapi4C7BF186' },
                                '.execute-api.',
                                { Ref: 'AWS::Region' },
                                '.',
                                { Ref: 'AWS::URLSuffix' },
                                '/',
                                { Ref: 'myapiDeploymentStageprod298F01AF' },
                                '/',
                            ]],
                    },
                },
            },
        });
        test.done();
    },
    'defaultChild is set correctly'(test) {
        const stack = new core_1.Stack();
        const api = new apigw.RestApi(stack, 'my-api');
        test.ok(api.node.defaultChild instanceof apigw.CfnRestApi);
        test.done();
    },
    '"name" is defaulted to resource physical name'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        const api = new apigw.RestApi(stack, 'restapi', {
            deploy: false,
            cloudWatchRole: false,
        });
        api.root.addMethod('GET');
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::RestApi', {
            Name: 'restapi',
        }));
        test.done();
    },
    'fails in synthesis if there are no methods or definition'(test) {
        // GIVEN
        const app = new core_1.App();
        const stack = new core_1.Stack(app, 'my-stack');
        const api = new apigw.RestApi(stack, 'API');
        // WHEN
        api.root.addResource('foo');
        api.root.addResource('bar').addResource('goo');
        // THEN
        test.throws(() => app.synth(), /The REST API doesn't contain any methods/);
        test.done();
    },
    '"addResource" can be used on "IRestApiResource" to form a tree'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const api = new apigw.RestApi(stack, 'restapi', {
            deploy: false,
            cloudWatchRole: false,
            restApiName: 'my-rest-api',
        });
        api.root.addMethod('GET');
        // WHEN
        const foo = api.root.addResource('foo');
        api.root.addResource('bar');
        foo.addResource('{hello}');
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Resource', {
            PathPart: 'foo',
            ParentId: { 'Fn::GetAtt': ['restapiC5611D27', 'RootResourceId'] },
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Resource', {
            PathPart: 'bar',
            ParentId: { 'Fn::GetAtt': ['restapiC5611D27', 'RootResourceId'] },
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Resource', {
            PathPart: '{hello}',
            ParentId: { Ref: 'restapifooF697E056' },
        }));
        test.done();
    },
    '"addResource" allows configuration of proxy paths'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const api = new apigw.RestApi(stack, 'restapi', {
            deploy: false,
            cloudWatchRole: false,
            restApiName: 'my-rest-api',
        });
        // WHEN
        const proxy = api.root.addResource('{proxy+}');
        proxy.addMethod('ANY');
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Resource', {
            PathPart: '{proxy+}',
            ParentId: { 'Fn::GetAtt': ['restapiC5611D27', 'RootResourceId'] },
        }));
        test.done();
    },
    '"addMethod" can be used to add methods to resources'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const api = new apigw.RestApi(stack, 'restapi', { deploy: false, cloudWatchRole: false });
        const r1 = api.root.addResource('r1');
        // WHEN
        api.root.addMethod('GET');
        r1.addMethod('POST');
        // THEN
        assert_1.expect(stack).toMatch({
            Resources: {
                restapiC5611D27: {
                    Type: 'AWS::ApiGateway::RestApi',
                    Properties: {
                        Name: 'restapi',
                    },
                },
                restapir1CF2997EA: {
                    Type: 'AWS::ApiGateway::Resource',
                    Properties: {
                        ParentId: {
                            'Fn::GetAtt': [
                                'restapiC5611D27',
                                'RootResourceId',
                            ],
                        },
                        PathPart: 'r1',
                        RestApiId: {
                            Ref: 'restapiC5611D27',
                        },
                    },
                },
                restapir1POST766920C4: {
                    Type: 'AWS::ApiGateway::Method',
                    Properties: {
                        HttpMethod: 'POST',
                        ResourceId: {
                            Ref: 'restapir1CF2997EA',
                        },
                        RestApiId: {
                            Ref: 'restapiC5611D27',
                        },
                        AuthorizationType: 'NONE',
                        Integration: {
                            Type: 'MOCK',
                        },
                    },
                },
                restapiGET6FC1785A: {
                    Type: 'AWS::ApiGateway::Method',
                    Properties: {
                        HttpMethod: 'GET',
                        ResourceId: {
                            'Fn::GetAtt': [
                                'restapiC5611D27',
                                'RootResourceId',
                            ],
                        },
                        RestApiId: {
                            Ref: 'restapiC5611D27',
                        },
                        AuthorizationType: 'NONE',
                        Integration: {
                            Type: 'MOCK',
                        },
                    },
                },
            },
        });
        test.done();
    },
    'resourcePath returns the full path of the resource within the API'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const api = new apigw.RestApi(stack, 'restapi');
        // WHEN
        const r1 = api.root.addResource('r1');
        const r11 = r1.addResource('r1_1');
        const r12 = r1.addResource('r1_2');
        const r121 = r12.addResource('r1_2_1');
        const r2 = api.root.addResource('r2');
        // THEN
        test.deepEqual(api.root.path, '/');
        test.deepEqual(r1.path, '/r1');
        test.deepEqual(r11.path, '/r1/r1_1');
        test.deepEqual(r12.path, '/r1/r1_2');
        test.deepEqual(r121.path, '/r1/r1_2/r1_2_1');
        test.deepEqual(r2.path, '/r2');
        test.done();
    },
    'resource path part validation'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const api = new apigw.RestApi(stack, 'restapi');
        // THEN
        test.throws(() => api.root.addResource('foo/'));
        api.root.addResource('boom-bam');
        test.throws(() => api.root.addResource('illegal()'));
        api.root.addResource('{foo}');
        test.throws(() => api.root.addResource('foo{bar}'));
        test.done();
    },
    'fails if "deployOptions" is set with "deploy" disabled'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // THEN
        test.throws(() => new apigw.RestApi(stack, 'myapi', {
            deploy: false,
            deployOptions: { cachingEnabled: true },
        }), /Cannot set 'deployOptions' if 'deploy' is disabled/);
        test.done();
    },
    'CloudWatch role is created for API Gateway'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const api = new apigw.RestApi(stack, 'myapi');
        api.root.addMethod('GET');
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Role'));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Account'));
        test.done();
    },
    '"url" and "urlForPath" return the URL endpoints of the deployed API'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const api = new apigw.RestApi(stack, 'api');
        api.root.addMethod('GET');
        // THEN
        test.deepEqual(stack.resolve(api.url), { 'Fn::Join': ['',
                ['https://',
                    { Ref: 'apiC8550315' },
                    '.execute-api.',
                    { Ref: 'AWS::Region' },
                    '.',
                    { Ref: 'AWS::URLSuffix' },
                    '/',
                    { Ref: 'apiDeploymentStageprod896C8101' },
                    '/']] });
        test.deepEqual(stack.resolve(api.urlForPath('/foo/bar')), { 'Fn::Join': ['',
                ['https://',
                    { Ref: 'apiC8550315' },
                    '.execute-api.',
                    { Ref: 'AWS::Region' },
                    '.',
                    { Ref: 'AWS::URLSuffix' },
                    '/',
                    { Ref: 'apiDeploymentStageprod896C8101' },
                    '/foo/bar']] });
        test.done();
    },
    '"urlForPath" would not work if there is no deployment'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const api = new apigw.RestApi(stack, 'api', { deploy: false });
        api.root.addMethod('GET');
        // THEN
        test.throws(() => api.url, /Cannot determine deployment stage for API from "deploymentStage". Use "deploy" or explicitly set "deploymentStage"/);
        test.throws(() => api.urlForPath('/foo'), /Cannot determine deployment stage for API from "deploymentStage". Use "deploy" or explicitly set "deploymentStage"/);
        test.done();
    },
    '"urlForPath" requires that path will begin with "/"'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const api = new apigw.RestApi(stack, 'api');
        api.root.addMethod('GET');
        // THEN
        test.throws(() => api.urlForPath('foo'), /Path must begin with \"\/\": foo/);
        test.done();
    },
    '"executeApiArn" returns the execute-api ARN for a resource/method'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const api = new apigw.RestApi(stack, 'api');
        api.root.addMethod('GET');
        // WHEN
        const arn = api.arnForExecuteApi('method', '/path', 'stage');
        // THEN
        test.deepEqual(stack.resolve(arn), { 'Fn::Join': ['',
                ['arn:',
                    { Ref: 'AWS::Partition' },
                    ':execute-api:',
                    { Ref: 'AWS::Region' },
                    ':',
                    { Ref: 'AWS::AccountId' },
                    ':',
                    { Ref: 'apiC8550315' },
                    '/stage/method/path']] });
        test.done();
    },
    '"executeApiArn" path must begin with "/"'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const api = new apigw.RestApi(stack, 'api');
        api.root.addMethod('GET');
        // THEN
        test.throws(() => api.arnForExecuteApi('method', 'hey-path', 'stage'), /"path" must begin with a "\/": 'hey-path'/);
        test.done();
    },
    '"executeApiArn" will convert ANY to "*"'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const api = new apigw.RestApi(stack, 'api');
        const method = api.root.addMethod('ANY');
        // THEN
        test.deepEqual(stack.resolve(method.methodArn), { 'Fn::Join': ['',
                ['arn:',
                    { Ref: 'AWS::Partition' },
                    ':execute-api:',
                    { Ref: 'AWS::Region' },
                    ':',
                    { Ref: 'AWS::AccountId' },
                    ':',
                    { Ref: 'apiC8550315' },
                    '/',
                    { Ref: 'apiDeploymentStageprod896C8101' },
                    '/*/']] });
        test.done();
    },
    '"endpointTypes" can be used to specify endpoint configuration for the api'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        const api = new apigw.RestApi(stack, 'api', {
            endpointTypes: [apigw.EndpointType.EDGE, apigw.EndpointType.PRIVATE],
        });
        api.root.addMethod('GET');
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::RestApi', {
            EndpointConfiguration: {
                Types: [
                    'EDGE',
                    'PRIVATE',
                ],
            },
        }));
        test.done();
    },
    '"endpointConfiguration" can be used to specify endpoint types for the api'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        const api = new apigw.RestApi(stack, 'api', {
            endpointConfiguration: {
                types: [apigw.EndpointType.EDGE, apigw.EndpointType.PRIVATE],
            },
        });
        api.root.addMethod('GET');
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::RestApi', {
            EndpointConfiguration: {
                Types: ['EDGE', 'PRIVATE'],
            },
        }));
        test.done();
    },
    '"endpointConfiguration" can be used to specify vpc endpoints on the API'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        const api = new apigw.RestApi(stack, 'api', {
            endpointConfiguration: {
                types: [apigw.EndpointType.EDGE, apigw.EndpointType.PRIVATE],
                vpcEndpoints: [
                    aws_ec2_1.GatewayVpcEndpoint.fromGatewayVpcEndpointId(stack, 'ImportedEndpoint', 'vpcEndpoint'),
                    aws_ec2_1.GatewayVpcEndpoint.fromGatewayVpcEndpointId(stack, 'ImportedEndpoint2', 'vpcEndpoint2'),
                ],
            },
        });
        api.root.addMethod('GET');
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::RestApi', {
            EndpointConfiguration: {
                Types: [
                    'EDGE',
                    'PRIVATE',
                ],
                VpcEndpointIds: [
                    'vpcEndpoint',
                    'vpcEndpoint2',
                ],
            },
        }));
        test.done();
    },
    '"endpointTypes" and "endpointConfiguration" can NOT both be used to specify endpoint configuration for the api'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // THEN
        test.throws(() => new apigw.RestApi(stack, 'api', {
            endpointConfiguration: {
                types: [apigw.EndpointType.PRIVATE],
                vpcEndpoints: [aws_ec2_1.GatewayVpcEndpoint.fromGatewayVpcEndpointId(stack, 'ImportedEndpoint', 'vpcEndpoint')],
            },
            endpointTypes: [apigw.EndpointType.PRIVATE],
        }), /Only one of the RestApi props, endpointTypes or endpointConfiguration, is allowed/);
        test.done();
    },
    '"cloneFrom" can be used to clone an existing API'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const cloneFrom = apigw.RestApi.fromRestApiId(stack, 'RestApi', 'foobar');
        // WHEN
        const api = new apigw.RestApi(stack, 'api', {
            cloneFrom,
        });
        api.root.addMethod('GET');
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::RestApi', {
            CloneFrom: 'foobar',
            Name: 'api',
        }));
        test.done();
    },
    'allow taking a dependency on the rest api (includes deployment and stage)'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const api = new apigw.RestApi(stack, 'myapi');
        api.root.addMethod('GET');
        const resource = new core_1.CfnResource(stack, 'DependsOnRestApi', { type: 'My::Resource' });
        // WHEN
        resource.node.addDependency(api);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('My::Resource', {
            DependsOn: [
                'myapiAccountC3A4750C',
                'myapiCloudWatchRoleEB425128',
                'myapiGET9B7CD29E',
                'myapiDeploymentB7EF8EB7b8edc043bcd33e0d85a3c85151f47e98',
                'myapiDeploymentStageprod329F21FF',
                'myapi162F20B8',
            ],
        }, assert_1.ResourcePart.CompleteDefinition));
        test.done();
    },
    'defaultIntegration and defaultMethodOptions can be used at any level'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const rootInteg = new apigw.AwsIntegration({
            service: 's3',
            action: 'GetObject',
        });
        // WHEN
        const api = new apigw.RestApi(stack, 'myapi', {
            defaultIntegration: rootInteg,
            defaultMethodOptions: {
                authorizer: { authorizerId: 'AUTHID' },
                authorizationType: apigw.AuthorizationType.IAM,
            },
        });
        // CASE #1: should inherit integration and options from root resource
        api.root.addMethod('GET');
        const child = api.root.addResource('child');
        // CASE #2: should inherit integration from root and method options, but
        // "authorizationType" will be overridden to "None" instead of "IAM"
        child.addMethod('POST', undefined, {
            authorizationType: apigw.AuthorizationType.COGNITO,
        });
        const child2 = api.root.addResource('child2', {
            defaultIntegration: new apigw.MockIntegration(),
            defaultMethodOptions: {
                authorizer: { authorizerId: 'AUTHID2' },
            },
        });
        // CASE #3: integartion and authorizer ID are inherited from child2
        child2.addMethod('DELETE');
        // CASE #4: same as case #3, but integration is customized
        child2.addMethod('PUT', new apigw.AwsIntegration({ action: 'foo', service: 'bar' }));
        // THEN
        // CASE #1
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ApiGateway::Method', {
            HttpMethod: 'GET',
            ResourceId: { 'Fn::GetAtt': ['myapi162F20B8', 'RootResourceId'] },
            Integration: { Type: 'AWS' },
            AuthorizerId: 'AUTHID',
            AuthorizationType: 'AWS_IAM',
        }));
        // CASE #2
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ApiGateway::Method', {
            HttpMethod: 'POST',
            ResourceId: { Ref: 'myapichildA0A65412' },
            Integration: { Type: 'AWS' },
            AuthorizerId: 'AUTHID',
            AuthorizationType: 'COGNITO_USER_POOLS',
        }));
        // CASE #3
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ApiGateway::Method', {
            HttpMethod: 'DELETE',
            Integration: { Type: 'MOCK' },
            AuthorizerId: 'AUTHID2',
            AuthorizationType: 'AWS_IAM',
        }));
        // CASE #4
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ApiGateway::Method', {
            HttpMethod: 'PUT',
            Integration: { Type: 'AWS' },
            AuthorizerId: 'AUTHID2',
            AuthorizationType: 'AWS_IAM',
        }));
        test.done();
    },
    'addApiKey is supported'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const api = new apigw.RestApi(stack, 'myapi');
        api.root.addMethod('OPTIONS');
        // WHEN
        api.addApiKey('myapikey', {
            apiKeyName: 'myApiKey1',
            value: '01234567890ABCDEFabcdef',
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::ApiKey', {
            Enabled: true,
            Name: 'myApiKey1',
            StageKeys: [
                {
                    RestApiId: { Ref: 'myapi162F20B8' },
                    StageName: { Ref: 'myapiDeploymentStageprod329F21FF' },
                },
            ],
            Value: '01234567890ABCDEFabcdef',
        }));
        test.done();
    },
    'addModel is supported'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const api = new apigw.RestApi(stack, 'myapi');
        api.root.addMethod('OPTIONS');
        // WHEN
        api.addModel('model', {
            schema: {
                schema: apigw.JsonSchemaVersion.DRAFT4,
                title: 'test',
                type: apigw.JsonSchemaType.OBJECT,
                properties: { message: { type: apigw.JsonSchemaType.STRING } },
            },
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Model', {
            RestApiId: { Ref: stack.getLogicalId(api.node.findChild('Resource')) },
            Schema: {
                $schema: 'http://json-schema.org/draft-04/schema#',
                title: 'test',
                type: 'object',
                properties: { message: { type: 'string' } },
            },
        }));
        test.done();
    },
    'addRequestValidator is supported'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const api = new apigw.RestApi(stack, 'myapi');
        api.root.addMethod('OPTIONS');
        // WHEN
        api.addRequestValidator('params-validator', {
            requestValidatorName: 'Parameters',
            validateRequestBody: false,
            validateRequestParameters: true,
        });
        api.addRequestValidator('body-validator', {
            requestValidatorName: 'Body',
            validateRequestBody: true,
            validateRequestParameters: false,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::RequestValidator', {
            RestApiId: { Ref: stack.getLogicalId(api.node.findChild('Resource')) },
            Name: 'Parameters',
            ValidateRequestBody: false,
            ValidateRequestParameters: true,
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::RequestValidator', {
            RestApiId: { Ref: stack.getLogicalId(api.node.findChild('Resource')) },
            Name: 'Body',
            ValidateRequestBody: true,
            ValidateRequestParameters: false,
        }));
        test.done();
    },
    'creates output with given "exportName"'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        const api = new apigw.RestApi(stack, 'myapi', { endpointExportName: 'my-given-export-name' });
        api.root.addMethod('GET');
        // THEN
        test.deepEqual(assert_1.SynthUtils.toCloudFormation(stack).Outputs, {
            myapiEndpoint8EB17201: {
                Value: {
                    'Fn::Join': [
                        '',
                        [
                            'https://',
                            { Ref: 'myapi162F20B8' },
                            '.execute-api.',
                            { Ref: 'AWS::Region' },
                            '.',
                            { Ref: 'AWS::URLSuffix' },
                            '/',
                            { Ref: 'myapiDeploymentStageprod329F21FF' },
                            '/',
                        ],
                    ],
                },
                Export: { Name: 'my-given-export-name' },
            },
        });
        test.done();
    },
    'creates output when "exportName" is not specified'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        const api = new apigw.RestApi(stack, 'myapi');
        api.root.addMethod('GET');
        // THEN
        test.deepEqual(assert_1.SynthUtils.toCloudFormation(stack).Outputs, {
            myapiEndpoint8EB17201: {
                Value: {
                    'Fn::Join': [
                        '',
                        [
                            'https://',
                            { Ref: 'myapi162F20B8' },
                            '.execute-api.',
                            { Ref: 'AWS::Region' },
                            '.',
                            { Ref: 'AWS::URLSuffix' },
                            '/',
                            { Ref: 'myapiDeploymentStageprod329F21FF' },
                            '/',
                        ],
                    ],
                },
            },
        });
        test.done();
    },
    'gateway response resource is created'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        const api = new apigw.RestApi(stack, 'restapi', {
            deploy: false,
            cloudWatchRole: false,
        });
        api.root.addMethod('GET');
        api.addGatewayResponse('test-response', {
            type: apigw.ResponseType.ACCESS_DENIED,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::GatewayResponse', {
            ResponseType: 'ACCESS_DENIED',
            RestApiId: stack.resolve(api.restApiId),
            StatusCode: assert_1.ABSENT,
            ResponseParameters: assert_1.ABSENT,
            ResponseTemplates: assert_1.ABSENT,
        }));
        test.done();
    },
    'gateway response resource is created with parameters'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        const api = new apigw.RestApi(stack, 'restapi', {
            deploy: false,
            cloudWatchRole: false,
        });
        api.root.addMethod('GET');
        api.addGatewayResponse('test-response', {
            type: apigw.ResponseType.AUTHORIZER_FAILURE,
            statusCode: '500',
            responseHeaders: {
                'Access-Control-Allow-Origin': 'test.com',
                'test-key': 'test-value',
            },
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::GatewayResponse', {
            ResponseType: 'AUTHORIZER_FAILURE',
            RestApiId: stack.resolve(api.restApiId),
            StatusCode: '500',
            ResponseParameters: {
                'gatewayresponse.header.Access-Control-Allow-Origin': 'test.com',
                'gatewayresponse.header.test-key': 'test-value',
            },
            ResponseTemplates: assert_1.ABSENT,
        }));
        test.done();
    },
    'gateway response resource is created with templates'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        const api = new apigw.RestApi(stack, 'restapi', {
            deploy: false,
            cloudWatchRole: false,
        });
        api.root.addMethod('GET');
        api.addGatewayResponse('test-response', {
            type: apigw.ResponseType.AUTHORIZER_FAILURE,
            statusCode: '500',
            templates: {
                'application/json': '{ "message": $context.error.messageString, "statusCode": "488" }',
            },
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::GatewayResponse', {
            ResponseType: 'AUTHORIZER_FAILURE',
            RestApiId: stack.resolve(api.restApiId),
            StatusCode: '500',
            ResponseParameters: assert_1.ABSENT,
            ResponseTemplates: {
                'application/json': '{ "message": $context.error.messageString, "statusCode": "488" }',
            },
        }));
        test.done();
    },
    '"restApi" and "api" properties return the RestApi correctly'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        const api = new apigw.RestApi(stack, 'test-api');
        const method = api.root.addResource('pets').addMethod('GET');
        // THEN
        test.ok(method.restApi);
        test.ok(method.api);
        test.deepEqual(stack.resolve(method.api.restApiId), stack.resolve(method.restApi.restApiId));
        test.done();
    },
    '"restApi" throws an error on imported while "api" returns correctly'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        const api = apigw.RestApi.fromRestApiAttributes(stack, 'test-api', {
            restApiId: 'test-rest-api-id',
            rootResourceId: 'test-root-resource-id',
        });
        const method = api.root.addResource('pets').addMethod('GET');
        // THEN
        test.throws(() => method.restApi, /not available on Resource not connected to an instance of RestApi/);
        test.ok(method.api);
        test.done();
    },
    'Import': {
        'fromRestApiId()'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            // WHEN
            const imported = apigw.RestApi.fromRestApiId(stack, 'imported-api', 'api-rxt4498f');
            // THEN
            test.deepEqual(stack.resolve(imported.restApiId), 'api-rxt4498f');
            test.done();
        },
        'fromRestApiAttributes()'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            // WHEN
            const imported = apigw.RestApi.fromRestApiAttributes(stack, 'imported-api', {
                restApiId: 'test-restapi-id',
                rootResourceId: 'test-root-resource-id',
            });
            const resource = imported.root.addResource('pets');
            resource.addMethod('GET');
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Resource', {
                PathPart: 'pets',
                ParentId: stack.resolve(imported.restApiRootResourceId),
            }));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Method', {
                HttpMethod: 'GET',
                ResourceId: stack.resolve(resource.resourceId),
            }));
            test.done();
        },
    },
    'SpecRestApi': {
        'add Methods and Resources'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            const api = new apigw.SpecRestApi(stack, 'SpecRestApi', {
                apiDefinition: apigw.ApiDefinition.fromInline({ foo: 'bar' }),
            });
            // WHEN
            const resource = api.root.addResource('pets');
            resource.addMethod('GET');
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Resource', {
                PathPart: 'pets',
                ParentId: stack.resolve(api.restApiRootResourceId),
            }));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Method', {
                HttpMethod: 'GET',
                ResourceId: stack.resolve(resource.resourceId),
            }));
            test.done();
        },
    },
};
//# sourceMappingURL=data:application/json;base64,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