# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['uniplot', 'uniplot.axis_labels']

package_data = \
{'': ['*']}

install_requires = \
['numpy>=1.22.0']

setup_kwargs = {
    'name': 'uniplot',
    'version': '0.9.0',
    'description': 'Lightweight plotting to the terminal. 4x resolution via Unicode.',
    'long_description': '# Uniplot\n[![Build Status](https://github.com/olavolav/uniplot/workflows/Unit%20Tests/badge.svg)](https://github.com/olavolav/uniplot/actions?query=workflow%3A"Unit+Tests")\n[![PyPI Version](https://badge.fury.io/py/uniplot.svg)](https://pypi.org/project/uniplot/)\n[![PyPI Downloads](https://pepy.tech/badge/uniplot)](https://pepy.tech/project/uniplot)\n\nLightweight plotting to the terminal. 4x resolution via Unicode.\n\n![uniplot demo GIF](https://github.com/olavolav/uniplot/raw/master/resource/uniplot-demo.gif)\n\nWhen working with production data science code it can be handy to have plotting\ntool that does not rely on graphics dependencies or works only in a Jupyter notebook.\n\nThe **use case** that this was built for is to have plots as part of your data science /\nmachine learning CI pipeline - that way whenever something goes wrong, you get not only\nthe error and backtrace but also plots that show what the problem was.\n\n\n## Features\n\n* Unicode drawing, so 4x the resolution (pixels) of usual ASCII plots\n* Super simple API\n* Interactive mode (pass `interactive=True`)\n* Color mode (pass `color=True`) useful in particular when plotting multiple series\n* It\'s fast: Plotting 1M data points takes 100ms thanks to NumPy magic\n* Only one dependency: NumPy (but you have that anyway don\'t you)\n\nPlease note that Unicode drawing will work correctly only when using a font that\nfully supports the [Box-drawing character set](https://en.wikipedia.org/wiki/Box-drawing_character).\nPlease refer to [this page for a (incomplete) list of supported fonts](https://www.fileformat.info/info/unicode/block/block_elements/fontsupport.htm).\n\n\n## Examples\n\nNote that all the examples are without color and plotting only a single series od data. For using color see the GIF example above.\n\n### Plot sine wave\n\n```python\nimport math\nx = [math.sin(i/20)+i/300 for i in range(600)]\nfrom uniplot import plot\nplot(x, title="Sine wave")\n```\n\nResult:\n```\n                          Sine wave\n┌────────────────────────────────────────────────────────────┐\n│                                                    ▟▀▚     │\n│                                                   ▗▘ ▝▌    │\n│                                       ▗▛▜▖        ▞   ▐    │\n│                                       ▞  ▜       ▗▌    ▌   │ 2\n│                           ▟▀▙        ▗▘  ▝▌      ▐     ▜   │\n│                          ▐▘ ▝▖       ▞    ▜      ▌     ▝▌  │\n│              ▗▛▜▖        ▛   ▜      ▗▌    ▝▌    ▐▘      ▜  │\n│              ▛  ▙       ▗▘   ▝▖     ▐      ▚    ▞       ▝▌ │\n│  ▟▀▖        ▐▘  ▝▖      ▟     ▚     ▌      ▝▖  ▗▌        ▜▄│ 1\n│ ▐▘ ▐▖       ▛    ▙      ▌     ▐▖   ▗▘       ▚  ▞           │\n│ ▛   ▙      ▗▘    ▐▖    ▐       ▙   ▞        ▝▙▟▘           │\n│▐▘   ▐▖     ▐      ▌    ▛       ▐▖ ▗▘                       │\n│▞     ▌     ▌      ▐   ▗▘        ▜▄▛                        │\n│▌─────▐────▐▘───────▙──▞────────────────────────────────────│ 0\n│       ▌   ▛        ▝▙▟▘                                    │\n│       ▜  ▐▘                                                │\n│        ▙▄▛                                                 │\n└────────────────────────────────────────────────────────────┘\n         100       200       300       400       500       600\n```\n\n### Plot global temperature data\n\nHere we\'re using Pandas to load and prepare global temperature data from the [Our World in Data GitHub repository](https://github.com/owid/owid-datasets).\n\nFirst we load the data, rename a column and and filter the data:\n\n```python\nimport pandas as pd\nuri = "https://github.com/owid/owid-datasets/raw/master/datasets/Global%20average%20temperature%20anomaly%20-%20Hadley%20Centre/Global%20average%20temperature%20anomaly%20-%20Hadley%20Centre.csv"\ndata = pd.read_csv(uri)\ndata = data.rename(columns={"Global average temperature anomaly (Hadley Centre)": "Global"})\ndata = data[data.Entity == "median"]\n```\n\nThen we can plot it:\n\n```python\nfrom uniplot import plot\nplot(xs=data.Year, ys=data.Global, lines=True, title="Global normalized land-sea temperature anomaly", y_unit=" °C")\n```\n\nResult:\n```\n        Global normalized land-sea temperature anomaly\n┌────────────────────────────────────────────────────────────┐\n│                                                          ▞▀│\n│                                                         ▐  │\n│                                                         ▐  │\n│                                                     ▗   ▌  │ 0.6 °C\n│                                           ▙  ▗▄ ▛▄▖▗▘▌ ▞   │\n│                                          ▗▜  ▌ ▜  ▚▞ ▚▞    │\n│                                          ▐▝▖▐      ▘       │\n│                                    ▗   ▗ ▌ ▙▌              │ 0.3 °C\n│                                    ▛▖  ▞▙▘  ▘              │\n│                              ▖  ▗▄▗▘▐ ▐▘▜                  │\n│                            ▟ █  ▞ ▜ ▝▄▘                    │\n│   ▗▚   ▗    ▖       ▗   ▖▗▞ █▐  ▌    ▘                     │\n│▁▁▁▞▐▁▁▗▘▜▗▀▀▌▁▁▁▁▙▁▁▟▁▁▁▙▐▁▁▜▁▌▞▁▁▁▁▁▁▁▁▁▁▁▁▁▁▁▁▁▁▁▁▁▁▁▁▁▁▁│ 0 °C\n│▚ ▐ ▝▖ ▐  ▛  ▌ ▗▄▐ ▌▗▘▌ ▐▝▌    ▝▘                           │\n│ ▌▌  ▌ ▞     ▐▗▘ ▛ ▐▞ ▌ ▐                                   │\n│ ▝   ▝▖▌     ▐▞    ▝▌ ▚▜▐                                   │\n│      ▗▌     ▝        ▝ ▌                                   │\n└────────────────────────────────────────────────────────────┘\n1,950    1,960    1,970   1,980    1,990    2,000   2,010\n```\n\n\n## Parameters\n\nThe `plot` function accepts a number of parameters, all listed below. Note that only\n`ys` is required, all others are optional.\n\nThere is also a `plot_to_string` function with the same signature, if you want the result as a list of strings, to include the output elsewhere.\n\n### Data\n\n* `xs` - The x coordinates of the points to plot. Can either be `None`, or a list or NumPy array for plotting a single series, or a list of those for plotting multiple series. Defaults to `None`, meaning that the x axis will be just the sample index of\n`ys`.\n* `ys` - The y coordinates of the points to plot. Can either be a list or NumPy array for plotting a single series, or a list of those for plotting multiple series.\n\nIn both cases, NaN values are ignored.\n\n### Options\n\nIn alphabetical order:\n\n* `color` - Draw series in color. Defaults to `False` when plotting a single series, and to `True` when plotting multiple.\n* `height` - The height of the plotting region, in characters. Default is `17`.\n* `interactive` - Enable interactive mode. Defaults to `False`.\n* `legend_labels` - Labels for the series. Can be `None` or a list of strings. Defaults to `None`.\n* `lines` - Enable lines between points. Can either be `True` or `False`, or a list of those values for plotting multiple series. Defaults to `False`.\n* `line_length_hard_cap` - Enforce a hard limit on the number of characters per line of the plot area. This may override the `width` option if there is not enough space. Defaults to `None`.\n* `title` - The title of the plot. Defaults to `None`.\n* `width` - The width of the plotting region, in characters. Default is `60`. Note that if the `line_length_hard_cap` option is used and there is not enough space, the actual width may be smaller.\n* `x_as_log` - Plot the x axis as logarithmic scale. Defaults to `False`.\n* `x_gridlines` - A list of x values that have a vertical line for better orientation. Defaults to `[0]`, or to `[]` if `x_as_log` is enabled.\n* `x_max` - Maximum x value of the view. Defaults to a value that shows all data points.\n* `x_min` - Minimum x value of the view. Defaults to a value that shows all data points.\n* `x_unit` - Unit of the x axis. This is a string that is appended to the axis labels. Defaults to `""`.\n* `y_as_log` - Plot the y axis as logarithmic scale. Defaults to `False`.\n* `y_gridlines` - A list of y values that have a horizontal line for better orientation. Defaults to `[0]`, or to `[]` if `y_as_log` is enabled.\n* `y_max` - Maximum y value of the view. Defaults to a value that shows all data points.\n* `y_min` - Minimum y value of the view. Defaults to a value that shows all data points.\n* `y_unit` - Unit of the y axis. This is a string that is appended to the axis labels. Defaults to `""`.\n\n\n## Experimental features\n\nFor convenience there is also a `histogram` function that accepts one or more series and\nplots bar-chart like histograms. It will automatically discretize the series into a\nnumber of bins given by the `bins` option and display the result.\n\nWhen calling the `histogram` function, the `lines` option is `True` by default.\n\nExample:\n\n```python\nimport numpy as np\nx = np.sin(np.linspace(1, 1000))\nfrom uniplot import histogram\nhistogram(x)\n```\n\nResult:\n```\n┌────────────────────────────────────────────────────────────┐\n│   ▛▀▀▌                       │                   ▐▀▀▜      │ 5\n│   ▌  ▌                       │                   ▐  ▐      │\n│   ▌  ▌                       │                   ▐  ▐      │\n│   ▌  ▀▀▀▌                    │                ▐▀▀▀  ▝▀▀▜   │ 4\n│   ▌     ▌                    │                ▐        ▐   │\n│   ▌     ▌                    │                ▐        ▐   │\n│   ▌     ▙▄▄▄▄▄▖              │          ▗▄▄▄  ▐        ▐   │ 3\n│   ▌           ▌              │          ▐  ▐  ▐        ▐   │\n│   ▌           ▌              │          ▐  ▐  ▐        ▐   │\n│   ▌           ▌              │          ▐  ▐  ▐        ▐   │\n│   ▌           ▀▀▀▀▀▀▀▀▀▀▀▀▀▀▀▀▀▀▀▀▜  ▐▀▀▀  ▝▀▀▀        ▐   │ 2\n│   ▌                          │    ▐  ▐                 ▐   │\n│   ▌                          │    ▐  ▐                 ▐   │\n│   ▌                          │    ▐▄▄▟                 ▐   │ 1\n│   ▌                          │                         ▐   │\n│   ▌                          │                         ▐   │\n│▄▄▄▌▁▁▁▁▁▁▁▁▁▁▁▁▁▁▁▁▁▁▁▁▁▁▁▁▁▁│▁▁▁▁▁▁▁▁▁▁▁▁▁▁▁▁▁▁▁▁▁▁▁▁▁▐▄▄▄│ 0\n└────────────────────────────────────────────────────────────┘\n     -1                        0                       1\n```\n\n\n## Installation\n\nInstall via pip using:\n\n```\npip install uniplot\n```\n\n\n## Roadmap\n\nComing up:\n\n* Fill area under curve\n* Add generated page with list of supported fonts\n* Auto-detect color mode depending on terminal capabilities\n* Possibly: Fallback to ASCII characters\n\nInput is always welcome, let me know what is most needed for this to be as useful as possible.\n\n\n## Contributing\n\nClone this repository, and install dependecies via `poetry install`.\n\nYou can run the tests vie `poetry run ./run_tests` to make sure your setup is good. Then proceed with issues, PRs etc. the usual way.\n',
    'author': 'Olav Stetter',
    'author_email': 'olav.stetter@googlemail.com',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'None',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.8',
}


setup(**setup_kwargs)
